package org.jfrog.storage;

import java.sql.Connection;

/**
 * Data object to control various aspects of database query.
 *
 * @author Yossi Shaul
 */
public class QueryControls {

    static final QueryControls DEFAULT_QUERY_CONTROLS = new QueryControls.Builder().build();

    private final boolean allowDirtyReads;
    private final int queryTimeout;
    private final int networkTimeout;

    private QueryControls(boolean allowDirtyReads, int queryTimeout, int networkTimeout) {
        this.allowDirtyReads = allowDirtyReads;
        this.queryTimeout = queryTimeout;
        this.networkTimeout = networkTimeout;
    }

    /**
     * @return Whether to allow dirty read byt setting the transaction isolation level to
     * {@link Connection#TRANSACTION_READ_UNCOMMITTED}.
     */
    public boolean isAllowDirtyReads() {
        return allowDirtyReads;
    }

    /**
     * @return Query execution timout in seconds. O if no timeout is set.
     */
    public int getQueryTimeout() {
        return queryTimeout;
    }

    /**
     * @return Network execution timout in seconds. O if no timeout is set.
     */
    public int getNetworkTimeout() {
        return networkTimeout;
    }

    public static class Builder {
        private boolean allowDirtyReads;
        private int queryTimeout;
        private int networkTimeout;

        public QueryControls build() {
            return new QueryControls(allowDirtyReads, queryTimeout, networkTimeout);
        }

        /**
         * Allow dirty read byt setting the transaction isolation level to
         * {@link Connection#TRANSACTION_READ_UNCOMMITTED}. Use with caution there are very rare circumstances in which
         * this is required by our apps.
         */
        public Builder allowDirtyReads() {
            this.allowDirtyReads = true;
            return this;
        }

        /**
         * Set the query execution time in seconds. 0 means no timeout
         */
        public Builder queryTimeout(int queryTimeout) {
            this.queryTimeout = queryTimeout;
            return this;
        }

        /**
         * Set the network execution time in seconds. 0 means no timeout
         */
        public Builder networkTimeout(int networkTimeout) {
            this.networkTimeout = networkTimeout;
            return this;
        }
    }

}
