/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2016 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.storage;

import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jfrog.storage.dbtype.DbSpecificHelper;
import org.jfrog.storage.dbtype.DefaultDbTypeHelper;
import org.jfrog.storage.dbtype.OracleSpecificHelper;
import org.jfrog.storage.util.DbUtils;
import org.jfrog.storage.util.PerfTimer;
import org.jfrog.storage.util.SqlTracer;
import org.jfrog.storage.util.TxHelper;
import org.jfrog.storage.wrapper.BlobWrapper;
import org.jfrog.storage.wrapper.ResultSetWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.io.Closeable;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.Optional;

import static org.jfrog.storage.util.DbStatementUtils.parseInListQuery;
import static org.jfrog.storage.util.DbStatementUtils.setParamsToStmt;

/**
 * A helper class to execute jdbc queries.
 *
 * @author Yossi Shaul
 */
public class JdbcHelper implements Closeable, DatabaseHelper {
    private static final Logger log = LoggerFactory.getLogger(JdbcHelper.class);
    public static final int NO_DB_ID = -1;

    private final JFrogDataSource dataSource;
    private boolean closed = false;
    private final SqlTracer tracer = new SqlTracer();
    private final Map<DbType, DbSpecificHelper> dbSpecificHelpers = Maps.newHashMap();

    public JdbcHelper(JFrogDataSource dataSource) {
        this.dataSource = dataSource;
        initDbSpecificTypeHelpers();
    }

    private Connection getConnection(QueryControls controls) throws SQLException {
        assertNotClosed();
        return DbUtils.getConnection(controls, dataSource, dataSource.getExecutorService());
    }

    private void initDbSpecificTypeHelpers() {
        dbSpecificHelpers.put(DbType.ORACLE, new OracleSpecificHelper());
        dbSpecificHelpers.put(null, new DefaultDbTypeHelper());
    }

    public JFrogDataSource getDataSource() {
        assertNotClosed();
        return dataSource;
    }

    @Nonnull
    public ResultSet executeSelect(String query, Object... params) throws SQLException {
        return executeSelect(QueryControls.DEFAULT_QUERY_CONTROLS, query, params);
    }

    @Nonnull
    @SuppressWarnings("squid:S2095")
    public ResultSet executeSelect(QueryControls controls, String query, Object... params) throws SQLException {
        if (closed) {
            throw new IllegalStateException("DataSource is closed cannot execute select query:\n'" + query + "'");
        }
        PerfTimer timer = traceSqlStart(query, params);

        Connection con = null;
        Statement stmt = null;
        ResultSet rs = null;
        try {
            con = getConnection(controls);
            allowDirtyReads(controls.isAllowDirtyReads(), con);
            if (params == null || params.length == 0) {
                stmt = con.createStatement();
                setQueryTimeout(stmt, controls);
                rs = stmt.executeQuery(query);
            } else {
                PreparedStatement pstmt = con.prepareStatement(parseInListQuery(query, params));
                stmt = pstmt;
                setQueryTimeout(stmt, controls);
                setParamsToStmt(pstmt, params);
                rs = pstmt.executeQuery();
            }
            if (!TxHelper.isInTransaction() && !con.getAutoCommit()) {
                con.commit();
            }
            traceSqlFinish(query, params, timer);
            return ResultSetWrapper.newInstance(con, stmt, rs, dataSource, controls.isAllowDirtyReads());
        } catch (Exception e) {
            DbUtils.close(con, stmt, rs, dataSource);
            if (e instanceof SQLException) {
                throw (SQLException) e;
            } else {
                throw new SQLException("Unexpected exception: " + e.getMessage(), e);
            }
        }
    }

    public int executeUpdate(String query, Object... params) throws SQLException {
        return executeUpdate(QueryControls.DEFAULT_QUERY_CONTROLS, query, params);
    }

    public int executeUpdate(QueryControls controls, String query, Object... params) throws SQLException {
        if (closed) {
            throw new IllegalStateException("DataSource is closed cannot execute update query:\n'" + query + "'");
        }
        PerfTimer timer = traceSqlStart(query, params);

        Connection con = null;
        Statement stmt = null;
        int results;
        try {
            con = getConnection(controls);
            if (params == null || params.length == 0) {
                stmt = con.createStatement();
                setQueryTimeout(stmt, controls);
                results = stmt.executeUpdate(query);
            } else {
                PreparedStatement pstmt = con.prepareStatement(parseInListQuery(query, params));
                stmt = pstmt;
                setQueryTimeout(stmt, controls);
                setParamsToStmt(pstmt, params);
                results = pstmt.executeUpdate();
            }
            traceSqlFinish(query, params, timer, results);
            return results;
        } finally {
            DbUtils.close(con, stmt, null, dataSource);
        }
    }

    public int executeSelectCount(String query, Object... params) throws SQLException {
        try (ResultSet resultSet = executeSelect(query, params)) {
            int count = 0;
            if (resultSet.next()) {
                count = resultSet.getInt(1);
            }
            return count;
        }
    }

    /**
     * Helper method to select long value. This method expects long value returned as the first column.
     * It ignores multiple results.
     *
     * @param query  The select query to execute
     * @param params The query parameters
     * @return Long value if a result was found or {@link #NO_DB_ID} if not found
     */
    public long executeSelectLong(String query, Object... params) throws SQLException {
        try (ResultSet resultSet = executeSelect(query, params)) {
            long result = NO_DB_ID;
            if (resultSet.next()) {
                result = resultSet.getLong(1);
            }
            return result;
        }
    }

    public static String resolveQuery(String sql, Object[] params) {
        int expectedParamsCount = StringUtils.countMatches(sql, "?");
        int paramsLength = params == null ? 0 : params.length;
        if (expectedParamsCount != paramsLength) {
            log.warn("Unexpected parameters count. Expected {} but got {}", expectedParamsCount, paramsLength);
        }

        if (params == null || params.length == 0) {
            return sql;
        } else if (expectedParamsCount == paramsLength) {
            sql = buildSqlWithParams(sql, params);
        } else {    // params not null but there's a difference between actual and expected
            sql = buildErrorSql(sql, params);
        }
        return sql;
    }

    @NotNull
    private static String buildErrorSql(String sql, Object[] params) {
        StringBuilder builder = new StringBuilder();
        builder.append("Executing SQL: '").append(sql).append("'");
        builder.append(" with params: ");
        for (int i = 0; i < params.length; i++) {
            builder.append("'");
            builder.append(params[i]);
            builder.append("'");
            if (i < params.length - 1) {
                builder.append(", ");
            }
        }
        sql = builder.toString();
        return sql;
    }

    private static String buildSqlWithParams(String sql, Object[] params) {
        Object[] printParams = buildPrintParams(params);

        //RTFACT-10291 escape % sent in the SQL like statement in order to avoid String.format
        // MissingFormatArgumentException
        sql = sql.replace("%", "%%").replaceAll("\\?", "%s");
        sql = String.format(sql, printParams);
        return sql;
    }

    @NotNull
    private static Object[] buildPrintParams(Object[] params) {
        // replace placeholders in the query with matching parameter values
        // this method doesn't attempt to escape characters
        Object[] printParams = new Object[params.length];
        for (int i = 0; i < params.length; i++) {
            Object current = params[i];
            if (current == null) {
                current = "NULL";
            } else if (current instanceof String) {
                current = "'" + params[i] + "'";
            } else if (current instanceof BlobWrapper) {
                current = "BLOB(length: " + ((BlobWrapper) params[i]).getLength() + ")";
            }
            printParams[i] = current;
        }
        return printParams;
    }

    public boolean isClosed() {
        return closed;
    }

    public long getSelectQueriesCount() {
        return tracer.getSelectQueriesCount();
    }

    public long getUpdateQueriesCount() {
        return tracer.getUpdateQueriesCount();
    }

    public SqlTracer getTracer() {
        return tracer;
    }

    DbSpecificHelper getDbSpecificHelper(DbType dbType) {
        return Optional.ofNullable(dbSpecificHelpers.get(dbType))
                .orElseGet(() -> dbSpecificHelpers.get(null));
    }

    @Override
    public void close() throws IOException {
        if (!isClosed()) {
            destroy();
        }
    }

    public void destroy() {
        closed = true;
        DbUtils.closeDataSource(dataSource);
    }

    private void assertNotClosed() {
        if (closed) {
            throw new IllegalStateException("DataSource is closed");
        }
    }

    private void setQueryTimeout(Statement stmt, QueryControls controls) throws SQLException {
        if (controls.getQueryTimeout() > 0) {
            stmt.setQueryTimeout(controls.getQueryTimeout());
        }
    }

    private void allowDirtyReads(boolean allowDirtyReads, Connection con) throws SQLException {
        if (allowDirtyReads) {
            con.setTransactionIsolation(Connection.TRANSACTION_READ_UNCOMMITTED);
        }
    }

    private PerfTimer traceSqlStart(String query, Object[] params) {
        tracer.traceSelectQuery(query);
        if (log.isDebugEnabled() && includeQueryLogging(query)) {
            String resolved = resolveQuery(query, params);
            log.trace("Executing SQL: '{}'.", resolved);
            return initPerfTimer();
        }
        return null;
    }

    private void traceSqlFinish(String query, Object[] params, PerfTimer timer) {
        traceSqlFinish(query, params, timer, -1);
    }

    private void traceSqlFinish(String query, Object[] params, PerfTimer timer, int results) {
        if (timer != null && log.isDebugEnabled()) {
            timer.stop();
            if (results < 0) {
                log.debug("Query returned in {}: \"{}\"", timer, resolveQuery(query, params));
            } else {
                log.debug("Query returned in {} with {} result{}: \"{}\"", timer, results, results > 1 ? "s" : "",
                        resolveQuery(query, params));
            }
        }
    }

    private boolean includeQueryLogging(String query) {
        return log.isTraceEnabled() || !excludeQueryLogging(query);
    }

    /**
     * Override to skip logging of specific queries in debug mode. Trace mode will always print all queries
     */
    protected boolean excludeQueryLogging(String query) {
        return false; // no excludes by default
    }

    @Nullable
    private PerfTimer initPerfTimer() {
        return log.isDebugEnabled() ? new PerfTimer() : null;
    }
}
