/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2018 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.storage;

import com.zaxxer.hikari.HikariDataSource;
import com.zaxxer.hikari.util.IsolationLevel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.TimeUnit;

/**
 * A pooling data source based on Hikari pool library.
 * See https://github.com/brettwooldridge/HikariCP for configuration details.
 *
 * @author Yossi Shaul
 */
public class HikariJDBCDataSource extends HikariDataSource implements JFrogDataSource {
    private static final Logger log = LoggerFactory.getLogger(HikariJDBCDataSource.class);

    public HikariJDBCDataSource(DbProperties p) {
        log.debug("Initializing Hikari pooling data source");
        setPoolName("HikariCP Main");

        setJdbcUrl(p.getConnectionUrl());
        setDriverClassName(p.getDriverClass());
        setUsername(p.getUsername());
        setPassword(p.getPassword());
        setCatalog(p.getProperty("defaultCatalog", null));
        setSchema(p.getProperty("schema", null));

        setAutoCommit(p.getBooleanProperty("defaultAutoCommit", true));
        setTransactionIsolation(IsolationLevel.TRANSACTION_READ_COMMITTED.name());

        setMaximumPoolSize(p.getMaxActiveConnections());
        setMaxLifetime(p.getIntProperty("maxAge", 1800000));

        setMinimumIdle(p.getIntProperty("minIdle", 1));
        // minimum time before idle connection can be evicted (minimum is 10000)
        setIdleTimeout(p.getIntProperty("minEvictableIdleTimeMillis", 300000));

        // maximum time to get a connection from the pool
        setConnectionTimeout(p.getIntProperty("maxWait", (int) TimeUnit.SECONDS.toMillis(120)));

        setConnectionInitSql(p.getProperty("initSQL", null));

        setConnectionTestQuery(p.getProperty("validationQuery", null)); // use jdbc4 isValid by default
        setValidationTimeout(p.getIntProperty("validationQueryTimeout", 5000));

        setLeakDetectionThreshold(p.getLongProperty("leakDetectionThreshold", 0));

        setAllowPoolSuspension(p.getBooleanProperty("allowPoolSuspension", false));

        setInitializationFailTimeout(p.getLongProperty("initializationFailTimeout", 1));
        setIsolateInternalQueries(p.getBooleanProperty("isolateInternalQueries", false));
    }

    public static JFrogDataSource createUniqueIdDataSource(DbProperties p) {
        HikariJDBCDataSource ds = new HikariJDBCDataSource(p);
        ds.setPoolName("HikariCP Unique");

        ds.setAutoCommit(true);     // always true for the id generator
        ds.setMaximumPoolSize(1);   // we need just one connection
        ds.setMinimumIdle(p.getIntProperty("minIdle", 0));  // allow dropping to 0 connections
        ds.setLeakDetectionThreshold(p.getLongProperty("leakDetectionThreshold", 60_000)); // 60 secs
        return ds;
    }
}