package org.jfrog.storage;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.jfrog.sysconf.SysConfig;
import org.apache.commons.lang.StringUtils;

import java.io.File;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

/**
 * Holds database configuration info.
 *
 * @author Yossi Shaul
 */
public abstract class DbProperties {
    public static final String DERBY_DB_HOME_PLACE_HOLDER = "{db.home}";
    public static final String TOMCAT_POOL_TYPE = "tomcat-jdbc";
    public static final String HIKARI_POOL_TYPE = "hikari";
    public static final String DEFAULT_POOL_TYPE = HIKARI_POOL_TYPE;

    protected final Map<String, String> props;
    protected final DbType dbType;
    private final SysConfig sysConfig;

    public DbProperties(SysConfig sysConfig, String defaultDbHome, String defaultDbLog) {
        this.sysConfig = sysConfig;
        props = Maps.newHashMap();
        try {
            dbType = resolveDbType(sysConfig);
            populateDbProperties(props, sysConfig);
            trimValues();
            assertMandatoryProperties(getMandatoryProperties());
            populateDerbyPropsIfNeeded(sysConfig, defaultDbHome, defaultDbLog);
        } catch (Exception e) {
            throw new IllegalStateException("Failed to init db properties: ", e);
        }
    }

    private void populateDerbyPropsIfNeeded(SysConfig config, String defaultDbHome, String defaultDbLog) {
        // configure embedded derby
        if (dbType == DbType.DERBY) {
            System.setProperty("derby.stream.error.file", new File(defaultDbLog).getAbsolutePath());

            String dbHome = config.get(dbHomePropertyYamlKey(), defaultDbHome);
            props.put(dbHomePropertyYamlKey(), dbHome);

            String url = props.get(Key.URL.key()).replace(DERBY_DB_HOME_PLACE_HOLDER, dbHome);
            props.put(Key.URL.key(), url);
        }
    }

    private void populateDbProperties(Map<String, String> props, SysConfig config) {
        Stream.of(Key.values())
                .filter(Key::hasDefaultValue)
                .forEach(prop -> props.put(prop.key(), config.get(yamlKey(prop), prop.getDefaultValue())));

        Stream.of(Key.values())
                .filter(prop -> config.get(yamlKey(prop)).isPresent())
                .forEach(prop -> props.put(prop.key(), config.getOrFail(yamlKey(prop))));
    }

    public String getUsername() {
        return props.get(Key.USERNAME.key());
    }

    public String getConnectionUrl() {
        return props.get(Key.URL.key());
    }

    public String getPassword() {
        return props.get(Key.PASSWORD.key());
    }

    public String getDriverClass() {
        return props.get(Key.DRIVER.key());
    }

    public int getMaxActiveConnections() {
        return getIntProperty(Key.MAX_ACTIVE_CONNECTIONS.key(), getDefaultMaxActiveConnections());
    }

    public int getMaxIdleConnections() {
        return getIntProperty(Key.MAX_IDLE_CONNECTIONS.key(), getDefaultMaxIdleConnections());
    }

    public DbType getDbType() {
        return dbType;
    }

    public String getPoolType() {
        return getProperty(Key.POOL_TYPE);
    }

    public String getProperty(Key property) {
        return props.get(property.key());
    }

    public String getProperty(Key key, String defaultValue) {
        return props.getOrDefault(key.key(), defaultValue);
    }

    public String getProperty(String key, String defaultValue) {
        String value = getProperty(key);
        if (value == null) {
            value = defaultValue;
        }
        return value;
    }

    public String getProperty(String key) {
        String value = props.get(key);
        if (value == null) {
            value = sysConfig.get(yamlKey(key), null);
        }
        return value;
    }

    public boolean getBooleanProperty(String key, boolean defaultValue) {
        return Boolean.parseBoolean(getProperty(key, defaultValue + ""));
    }

    public int getIntProperty(String key, int defaultValue) {
        return Integer.parseInt(getProperty(key, defaultValue + ""));
    }

    public long getLongProperty(String key, long defaultValue) {
        return Long.parseLong(getProperty(key, defaultValue + ""));
    }

    private DbType resolveDbType(SysConfig config) {
        // default value is derby
        return DbType.parse(config.get(dbTypeYamlKey(), "derby"));
    }

    protected abstract int getDefaultMaxActiveConnections();

    protected abstract int getDefaultMaxIdleConnections();

    private String dbTypeYamlKey() {
        return yamlKey(DbProperties.Key.TYPE);
    }

    private String dbHomePropertyYamlKey() {
        return yamlKey(Key.DB_HOME);
    }

    private List<String> getMandatoryProperties() {
        return ImmutableList.of(Key.TYPE.key(), Key.URL.key(), Key.DRIVER.key());
    }

    protected void assertMandatoryProperties(List<String> mandatoryProps) {
        for (String mandatoryProperty : mandatoryProps) {
            String value = getProperty(mandatoryProperty);
            if (StringUtils.isBlank(value)) {
                throw new IllegalStateException(
                        "Mandatory database parameter '" + mandatoryProperty + "' doesn't exist");
            }
        }
    }

    private void trimValues() {
        for (Map.Entry<String, String> entry : props.entrySet()) {
            String value = entry.getValue();
            if (!StringUtils.trimToEmpty(value).equals(value)) {
                entry.setValue(StringUtils.trim(value));
            }
        }
    }

    private String yamlKey(Key propKey) {
        return yamlKey(propKey.key());
    }

    /**
     * Transform the shorthand key to a db properties key.db properties keys are composed of the service name, the word
     * database and the short property name.
     * For example, is service id is artifactory:
     * <pre>
     *     yamlKey("maxOpenConnections") = artifactory.database.maxOpenConnections
     * </pre>
     */
    String yamlKey(String propKey) {
        return getServicePrefix() + "database." + propKey;
    }

    protected abstract String getServicePrefix();

    /**
     * Each key is expected to be found under the same name in at least one of the ways to pass db params
     * (shared, service-specific)
     * The config lib is responsible to resolve the key with the default fall back to the 'shared' section.
     */
    public enum Key {
        DB_HOME("dbHome"), //Internal use, for derby only
        USERNAME("username"),
        PASSWORD("password"),
        TYPE("type", "derby"),
        URL("url", "jdbc:derby:" + DERBY_DB_HOME_PLACE_HOLDER + ";create=true"),
        DRIVER("driver", "org.apache.derby.jdbc.EmbeddedDriver"),
        MAX_ACTIVE_CONNECTIONS("maxOpenConnections"),
        MAX_IDLE_CONNECTIONS("maxIdleConnections"),
        POOL_TYPE("poolType", DEFAULT_POOL_TYPE),
        POOL_MAX_ACTIVE("maxOpenConnections"),
        POOL_MAX_IDLE("maxIdleConnections"),
        LOCKING_DB_USERNAME("lockingdb.username"),
        LOCKING_DB_PASSWORD("lockingdb.password"),
        LOCKING_DB_TYPE("lockingdb.type"),
        LOCKING_DB_URL("lockingdb.url"),
        LOCKING_DB_DRIVER("lockingdb.driver"),
        LOCKING_DB_CATALOG("lockingdb.defaultCatalog");

        private final String propKey;
        private final String defaultValue;

        Key(String propKey) {
            this.propKey = propKey;
            this.defaultValue = "";
        }

        Key(String propKey, String defaultValue) {
            this.propKey = propKey;
            this.defaultValue = defaultValue;
        }

        public String key() {
            return propKey;
        }

        private String getDefaultValue() {
            return defaultValue;
        }

        private boolean hasDefaultValue() {
            return StringUtils.isNotBlank(defaultValue);
        }

    }
}
