/*
 * Decompiled with CFR 0.152.
 */
package org.jfrog.storage;

import com.google.common.collect.Maps;
import java.io.Closeable;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.Optional;
import javax.annotation.Nonnull;
import javax.sql.DataSource;
import org.apache.commons.lang.StringUtils;
import org.jfrog.storage.DatabaseHelper;
import org.jfrog.storage.DbType;
import org.jfrog.storage.dbtype.DbSpecificHelper;
import org.jfrog.storage.dbtype.DefaultDbTypeHelper;
import org.jfrog.storage.dbtype.OracleSpecificHelper;
import org.jfrog.storage.util.DbStatementUtils;
import org.jfrog.storage.util.DbUtils;
import org.jfrog.storage.util.PerfTimer;
import org.jfrog.storage.util.SqlTracer;
import org.jfrog.storage.util.TxHelper;
import org.jfrog.storage.wrapper.BlobWrapper;
import org.jfrog.storage.wrapper.ResultSetWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.datasource.DataSourceUtils;

public class JdbcHelper
implements Closeable,
DatabaseHelper {
    private static final Logger log = LoggerFactory.getLogger(JdbcHelper.class);
    public static final int NO_DB_ID = -1;
    private final DataSource dataSource;
    private boolean closed = false;
    private final SqlTracer tracer = new SqlTracer();
    private Map<DbType, DbSpecificHelper> dbSpecificHelpers = Maps.newHashMap();

    public JdbcHelper(DataSource dataSource) {
        this.dataSource = dataSource;
        this.initDbSpecificTypeHelpers();
    }

    private void initDbSpecificTypeHelpers() {
        this.dbSpecificHelpers.put(DbType.ORACLE, new OracleSpecificHelper());
        this.dbSpecificHelpers.put(null, new DefaultDbTypeHelper());
    }

    private static void debugSql(String sql, Object[] params) {
        if (log.isDebugEnabled()) {
            String resolved = JdbcHelper.resolveQuery(sql, params);
            log.debug("Executing SQL: '" + resolved + "'.");
        }
    }

    public static String resolveQuery(String sql, Object[] params) {
        int paramsLength;
        int expectedParamsCount = StringUtils.countMatches((String)sql, (String)"?");
        int n = paramsLength = params == null ? 0 : params.length;
        if (expectedParamsCount != paramsLength) {
            log.warn("Unexpected parameters count. Expected {} but got {}", (Object)expectedParamsCount, (Object)paramsLength);
        }
        if (params == null || params.length == 0) {
            return sql;
        }
        if (expectedParamsCount == paramsLength) {
            Object[] printParams = new Object[params.length];
            for (int i = 0; i < params.length; ++i) {
                Object current = params[i];
                if (current == null) {
                    current = "NULL";
                } else if (current instanceof String) {
                    current = "'" + params[i] + "'";
                } else if (current instanceof BlobWrapper) {
                    current = "BLOB(length: " + ((BlobWrapper)params[i]).getLength() + ")";
                }
                printParams[i] = current;
            }
            sql = sql.replace("%", "%%").replaceAll("\\?", "%s");
            sql = String.format(sql, printParams);
        } else {
            StringBuilder builder = new StringBuilder();
            builder.append("Executing SQL: '").append(sql).append("'");
            if (params.length > 0) {
                builder.append(" with params: ");
                for (int i = 0; i < params.length; ++i) {
                    builder.append("'");
                    builder.append(params[i]);
                    builder.append("'");
                    if (i >= params.length - 1) continue;
                    builder.append(", ");
                }
            }
            sql = builder.toString();
        }
        return sql;
    }

    private Connection getConnection() throws SQLException {
        if (this.closed) {
            throw new IllegalStateException("DataSource is closed");
        }
        Connection connection = DataSourceUtils.doGetConnection((DataSource)this.dataSource);
        if (2 != connection.getTransactionIsolation()) {
            connection.setTransactionIsolation(2);
        }
        return connection;
    }

    public DataSource getDataSource() {
        if (this.closed) {
            throw new IllegalStateException("DataSource is closed");
        }
        return this.dataSource;
    }

    @Override
    @Nonnull
    public ResultSet executeSelect(String query, Object ... params) throws SQLException {
        return this.executeSelect(query, false, params);
    }

    @Nonnull
    public ResultSet executeSelect(String query, boolean allowDirtyReads, Object ... params) throws SQLException {
        if (this.closed) {
            throw new IllegalStateException("DataSource is closed cannot execute select query:\n'" + query + "'");
        }
        this.tracer.traceSelectQuery(query);
        JdbcHelper.debugSql(query, params);
        PerfTimer timer = null;
        if (log.isDebugEnabled()) {
            timer = new PerfTimer();
        }
        Connection con = null;
        Statement stmt = null;
        ResultSet rs = null;
        try {
            con = this.getConnection();
            TxHelper.allowDirtyReads(allowDirtyReads, con);
            if (params == null || params.length == 0) {
                stmt = con.createStatement();
                rs = stmt.executeQuery(query);
            } else {
                PreparedStatement pstmt = con.prepareStatement(DbStatementUtils.parseInListQuery(query, params));
                stmt = pstmt;
                DbStatementUtils.setParamsToStmt(pstmt, params);
                rs = pstmt.executeQuery();
            }
            if (!TxHelper.isInTransaction() && !con.getAutoCommit()) {
                con.commit();
            }
            if (timer != null && log.isDebugEnabled()) {
                timer.stop();
                log.debug("Query returned in {} : '{}'", (Object)timer, (Object)JdbcHelper.resolveQuery(query, params));
            }
            return ResultSetWrapper.newInstance(con, stmt, rs, this.dataSource);
        }
        catch (Exception e) {
            TxHelper.disableDirtyReads(allowDirtyReads, con);
            DbUtils.close(con, stmt, rs, this.dataSource);
            if (e instanceof SQLException) {
                throw (SQLException)e;
            }
            throw new SQLException("Unexpected exception: " + e.getMessage(), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int executeUpdate(String query, Object ... params) throws SQLException {
        if (this.closed) {
            throw new IllegalStateException("DataSource is closed cannot execute update query:\n'" + query + "'");
        }
        this.tracer.traceSelectQuery(query);
        JdbcHelper.debugSql(query, params);
        PerfTimer timer = null;
        if (log.isDebugEnabled()) {
            timer = new PerfTimer();
        }
        Connection con = null;
        Statement stmt = null;
        try {
            int results;
            con = this.getConnection();
            if (params == null || params.length == 0) {
                stmt = con.createStatement();
                results = stmt.executeUpdate(query);
            } else {
                PreparedStatement pstmt = con.prepareStatement(DbStatementUtils.parseInListQuery(query, params));
                stmt = pstmt;
                DbStatementUtils.setParamsToStmt(pstmt, params);
                results = pstmt.executeUpdate();
            }
            if (timer != null && log.isDebugEnabled()) {
                timer.stop();
                log.debug("Query returned with {} results in {} : '{}'", new Object[]{results, timer, JdbcHelper.resolveQuery(query, params)});
            }
            int n = results;
            return n;
        }
        finally {
            DbUtils.close(con, stmt, null, this.dataSource);
        }
    }

    public int executeSelectCount(String query, Object ... params) throws SQLException {
        try (ResultSet resultSet = this.executeSelect(query, params);){
            int count = 0;
            if (resultSet.next()) {
                count = resultSet.getInt(1);
            }
            int n = count;
            return n;
        }
    }

    @Override
    public long executeSelectLong(String query, Object ... params) throws SQLException {
        try (ResultSet resultSet = this.executeSelect(query, params);){
            long result = -1L;
            if (resultSet.next()) {
                result = resultSet.getLong(1);
            }
            long l = result;
            return l;
        }
    }

    public boolean isClosed() {
        return this.closed;
    }

    public long getSelectQueriesCount() {
        return this.tracer.getSelectQueriesCount();
    }

    public long getUpdateQueriesCount() {
        return this.tracer.getUpdateQueriesCount();
    }

    public SqlTracer getTracer() {
        return this.tracer;
    }

    DbSpecificHelper getDbSpecificHelper(DbType dbType) {
        return Optional.ofNullable(this.dbSpecificHelpers.get((Object)dbType)).orElseGet(() -> this.dbSpecificHelpers.get(null));
    }

    @Override
    public void close() throws IOException {
        if (!this.isClosed()) {
            this.destroy();
        }
    }

    public void destroy() {
        this.closed = true;
        DbUtils.closeDataSource(this.dataSource);
    }
}

