package org.jfrog.storage;

import org.apache.commons.lang.StringUtils;
import org.jfrog.common.ThrowingFunction;
import org.apache.commons.lang3.tuple.Pair;

import javax.annotation.Nonnull;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.Objects.requireNonNull;
import static org.jfrog.common.ArgUtils.requireNonBlank;
import static org.jfrog.common.ArgUtils.requirePositive;

/**
 * @author Noam Shemesh
 */
public class SqlDaoHelper<V> {

    // Sonar :-\
    private static final String SELECT_FROM = "SELECT * FROM ";
    private static final String WHERE = " WHERE ";
    private static final String SET = " SET ";
    private static final String DELETE_FROM = "DELETE FROM ";
    private static final String EQUALS_PARAM = " = ? ";

    private final JdbcHelper jdbcHelper;
    private final String tableName;
    private final DbType dbType;
    private final ThrowingFunction<ResultSet, V, SQLException> readValue;
    private final int batchSize;

    public SqlDaoHelper(JdbcHelper jdbcHelper, DbType dbType, String tableName, ThrowingFunction<ResultSet, V, SQLException> readValue
    , int batchSize) {
        this.dbType = dbType;
        this.jdbcHelper = requireNonNull(jdbcHelper, "jdbc helper is required");
        this.tableName = requireNonBlank(tableName, "table name is required");
        this.readValue = requireNonNull(readValue, "readValue function is required");
        this.batchSize = requirePositive(batchSize, "readValue function is required");
    }

    @Nonnull
    public List<V> getAll() throws SQLException {
        return findAllByQuery(SELECT_FROM + tableName);
    }

    /**
     * Find the first matching entity according to the given property and value
     * @param property the column name to use in the WHERE clause
     * @param value the property value for the WHERE clause
     * @return An {@link Optional} with the entity, or {@link Optional#empty} if not matching was found.
     * @throws SQLException
     */
    @Nonnull
    public Optional<V> findFirstByProperty(@Nonnull String property, @Nonnull Object value) throws SQLException {
        return findFirstValueByProperty(readValue, property, value);
    }

    /**
     * Find the first matching entity according to the given query
     * @param query the query to execute
     * @param params the parameters for the query
     * @return An {@link Optional} with the entity, or {@link Optional#empty} if not matching was found.
     * @throws SQLException
     */
    @Nonnull
    public Optional<V> findFirstByQuery(@Nonnull String query, Object... params) throws SQLException {
        return findFirstValueByQuery(readValue, query, params);
    }

    /**
     * Find the first matching row according to a given property and value and return a value.
     * @param valueReader the function to use for reading the value to return from the first matching result set row
     * @param property the column name to use in the WHERE clause
     * @param value the property value for the WHERE clause
     * @param <T> the type of value to return
     * @return An {@link Optional} with the value, or {@link Optional#empty} if not matching was found.
     * @throws SQLException
     */
    @Nonnull
    public <T> Optional<T> findFirstValueByProperty(@Nonnull ThrowingFunction<ResultSet, T, SQLException> valueReader,
            @Nonnull String property, @Nonnull Object value) throws SQLException {
        String query = SELECT_FROM + tableName + WHERE + property + " = ?";
        return findFirstValueByQuery(valueReader, query, value);
    }

    /**
     * Find the first matching row according to a given query and return a value.
     * @param valueReader the function to use for reading the value to return from the first matching result set row
     * @param query the query to execute
     * @param params the parameter values for the query
     * @param <T> the type of value to return
     * @return An {@link Optional} with the value, or {@link Optional#empty} if not matching was found.
     * @throws SQLException
     */
    @Nonnull
    public <T> Optional<T> findFirstValueByQuery(@Nonnull ThrowingFunction<ResultSet, T, SQLException> valueReader,
            @Nonnull String query, Object... params) throws SQLException {
        T value = null;
        try (ResultSet resultSet = jdbcHelper.executeSelect(query, params)) {
            if (resultSet.next()) {
                value = valueReader.apply(resultSet);
            }
        }

        return Optional.ofNullable(value);
    }

    @Nonnull
    public <K, T> Map<K, T> findValuesByProperties(@Nonnull ThrowingFunction<ResultSet, K, SQLException> keyReader,
            @Nonnull ThrowingFunction<ResultSet, T, SQLException> valueReader,
            @Nonnull String property, @Nonnull List<K> values) throws SQLException {
        Map<K, T> result = new HashMap<>();
        if (values.isEmpty()) {
            return result;
        }

        ThrowingFunction<ResultSet, T, SQLException> resultCollector = resultSet -> result
                .put(keyReader.apply(resultSet), valueReader.apply(resultSet));
        findByFieldWithPagination("*", tableName + WHERE + property, values, resultCollector);
        return result;
    }

    public <T> int paginationCreate(String tableNameAndFields,
            int numberOfFields,
            Function<T, Stream<Object>> groupToValues,
            T[] values)
            throws SQLException {
        String valuesString = buildValuesString(numberOfFields);
        Function<Integer, String> sqlBuilder = (Integer subListSize) -> jdbcHelper
                .getDbSpecificHelper(dbType)
                .getInsertMultipleValuesSql(tableNameAndFields, valuesString, subListSize);
        return updatePagination(groupToValues, sqlBuilder, Function.identity(), values);
    }

    public <T> int paginationDelete(String tableNameAndFields,
            int numberOfFields,
            Function<T, Stream<Object>> groupToValues,
            Function<Object[], Object[]> paramsDecorator,
            T[] values)
            throws SQLException {
        String valuesString = buildValuesString(numberOfFields);
        Function<Integer, String> sqlBuilder = (Integer subListSize) -> jdbcHelper
                .getDbSpecificHelper(dbType)
                .getDeleteMultipleValuesSql(tableNameAndFields, valuesString, subListSize);
        return updatePagination(groupToValues, sqlBuilder, paramsDecorator, values);
    }

    private <T> int paginationDeleteMultiColumnCondition(String tableNameAndOtherConditions,
            List<String> conditionColumns,
            Function<T, Stream<Object>> groupToValues,
            Function<Object[], Object[]> paramsDecorator,
            T[] values) throws SQLException {
        Function<Integer, String> sqlBuilder = (Integer subListSize) -> jdbcHelper
                .getDbSpecificHelper(dbType)
                .getDeleteMultipleColumnsMultipleValuesSql(tableNameAndOtherConditions, conditionColumns, subListSize);
        return updatePagination(groupToValues, sqlBuilder, paramsDecorator, values);
    }

    private <T> int updatePagination(Function<T, Stream<Object>> groupToValues,
            Function<Integer, String> sqlBuilder,
            Function<Object[], Object[]> paramsDecorator,
            T[] values) throws SQLException {
        List<T> fullList = Arrays.asList(values);
        if (fullList.isEmpty()) {
            return 0;
        }

        int res = 0;

        List<T> subList;
        int passed = 0;
        int toIndex = Math.min(batchSize + passed, fullList.size());
        while (passed <= toIndex &&
                !(subList = fullList.subList(passed, toIndex)).isEmpty()) {
            String query = sqlBuilder.apply(subList.size());

            Object[] params = subList.stream().flatMap(groupToValues).toArray(Object[]::new);
            params = paramsDecorator.apply(params);
            res += jdbcHelper.executeUpdate(query, params);

            passed += subList.size();
            toIndex = Math.min(batchSize + passed, fullList.size());
        }

        return res;
    }

    private String buildValuesString(int numberOfFields) {
        return numberOfFields == 1 ? "? " : "(" + StringUtils.repeat("?", ",", numberOfFields) + ")";
    }

    public <T> List<V> findByFieldWithPagination(@Nonnull String returnValues, @Nonnull String tableNameAndField,
            Set<T> fieldValues) throws SQLException {
        List<V> result = new ArrayList<>();
        ThrowingFunction<ResultSet, Boolean, SQLException> resultCollector = resultSet -> result
                .add(readValue.apply(resultSet));
        findByFieldWithPagination(returnValues, tableNameAndField, fieldValues, resultCollector);
        return Collections.unmodifiableList(result);
    }

    public <T> List<V> findByMultiColumnQueryWithPagination(@Nonnull String returnValues, @Nonnull String tableNameAndFields,
            @Nonnull List<String> columns, Collection<T> params) throws SQLException {
        List<V> result = new ArrayList<>();
        ThrowingFunction<ResultSet, Boolean, SQLException> resultCollector = resultSet -> result
                .add(readValue.apply(resultSet));
        Function<Integer, String> queryBuilder = i -> jdbcHelper
                .getDbSpecificHelper(dbType)
                .getQueryMultipleColumnsMultipleValuesSql(returnValues, tableNameAndFields, columns, i);
        batchApply(params, resultCollector, queryBuilder, batchSize - (batchSize % columns.size()));
        return Collections.unmodifiableList(result);
    }

    private <T, V> void findByFieldWithPagination(@Nonnull String returnValues, @Nonnull String tableNameAndFields,
            Collection<T> params, ThrowingFunction<ResultSet, V, SQLException> resultsCollector) throws SQLException {
        Function<Integer, String> queryBuilder = i -> jdbcHelper
                .getDbSpecificHelper(dbType)
                .getQueryMultipleValuesSql(returnValues, tableNameAndFields, "?", i);
        batchApply(params, resultsCollector, queryBuilder, batchSize);
    }

    private <T, V> void batchApply(Collection<T> params, ThrowingFunction<ResultSet, V, SQLException> resultsCollector,
            Function<Integer, String> queryBuilder, int batchSize) throws SQLException {
        List<T> fullList = new ArrayList<>(params);
        if (fullList.isEmpty()) {
            return;
        }

        List<T> subList;
        int passed = 0;
        int toIndex = Math.min(batchSize + passed, fullList.size());
        while (passed <= toIndex &&
                !(subList = fullList.subList(passed, toIndex)).isEmpty()) {
            Object[] currentParams = subList.toArray();
            String query = queryBuilder.apply(subList.size());
            try (ResultSet resultSet = jdbcHelper.executeSelect(query, currentParams)) {
                while (resultSet.next()) {
                    resultsCollector.apply(resultSet);
                }
            }

            passed += subList.size();
            toIndex = Math.min(batchSize + passed, fullList.size());
        }
    }

    @Nonnull
    public List<V> findAllByProperty(@Nonnull String property, Object value) throws SQLException {
        String query = SELECT_FROM + tableName + WHERE + property + " = ?";
        return findAllByQuery(query, value);
    }

    @Nonnull
    public List<V> findAllByQuery(@Nonnull String query, Object... params) throws SQLException {
        return findAllByQuery(query, readValue, params);
    }

    @Nonnull
    public <T> List<T> findAllByQuery(@Nonnull String query, ThrowingFunction<ResultSet, T, SQLException> converter, Object... params) throws SQLException {
        List<T> result = new LinkedList<>();
        try (ResultSet resultSet = jdbcHelper.executeSelect(query, params)) {
            while (resultSet.next()) {
                T entity = converter.apply(resultSet);
                result.add(entity);
            }
        }
        return Collections.unmodifiableList(result);
    }


    public int deleteAll() throws SQLException {
        return jdbcHelper.executeUpdate(DELETE_FROM + tableName);
    }

    public int deleteByProperty(String property, Object value) throws SQLException {
        return jdbcHelper.executeUpdate(DELETE_FROM + tableName + WHERE + property + " = ?", value);
    }

    public int deleteByProperties(List<Pair<String, Object>> properties) throws SQLException {
        return jdbcHelper.executeUpdate(DELETE_FROM + tableName + WHERE +
                        properties.stream().map(pair -> pair.getLeft() + " = ?").collect(Collectors.joining(" AND ")),
                properties.stream().map(Pair::getRight).toArray());
    }

    public int updateByProperty(Pair<String, Object> findBy, Pair<String, Object> toUpdate) throws SQLException {
        return jdbcHelper.executeUpdate("UPDATE " + tableName + " SET " +
                toUpdate.getLeft() + " = ? WHERE " + findBy.getLeft() + " = ?", toUpdate.getRight(), findBy.getRight());
    }

    public int updateByProperties(List<String> fieldsToFindBy, Stream<Object> valuesToFindBy ,List<String> fieldsToUpdate, Stream<Object> updateValues) throws SQLException {
        Object[] params = Stream.concat(updateValues, valuesToFindBy).toArray();
        int numberOfFields = fieldsToUpdate.size() + fieldsToFindBy.size();
        if (numberOfFields != params.length) {
            throw new IllegalStateException(
                    "number of fields (" + numberOfFields + ") didn't match number of values (" + params.length + ")");
        }
        return jdbcHelper.executeUpdate(
                "UPDATE " + tableName + buildSetClause(fieldsToUpdate) + buildWhereClause(fieldsToFindBy),
                params);
    }

    //" WHERE (key1=? AND key2=? AND ...) "
    private String buildWhereClause(List<String> columns) {
        return columns.stream()
                .collect(Collectors.joining(EQUALS_PARAM + " AND ", WHERE + " ( ", EQUALS_PARAM + " ) "));
    }

    //" SET key1=?, key2=?, ... "
    private String buildSetClause(List<String> columns) {
        return columns.stream().collect(Collectors.joining(EQUALS_PARAM + " , ", SET, EQUALS_PARAM));
    }

    public static Function<Object[], Object[]> buildDecorator(long userId) {
        return params -> {
            if (params.length == 0) {
                return params;
            }
            Object[] result = new Object[params.length + 1];
            result[0] = userId;
            System.arraycopy(params, 0, result, 1, params.length);
            return result;
        };
    }
}
