/*
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2016 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.storage.binstore.utils;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

/**
 * @author Yoav Landman
 */
public class Checksum {

    private final long length;
    private final MessageDigest digest;
    private String checksum;

    public static Checksum md5() {
        return new Checksum("MD5", 32);
    }

    public static Checksum sha1() {
        return new Checksum("SHA-1", 40);
    }

    public static Checksum sha256() {
        return new Checksum("SHA-256", 64);
    }

    /**
     * @param algorithm The algorithm to use for checksum calculation
     * @param length    The expected string length output
     */
    public Checksum(String algorithm, long length) {
        this.length = length;
        try {
            digest = MessageDigest.getInstance(algorithm);
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("Cannot create a digest for algorithm: " + algorithm, e);
        }
    }

    public String getAlgorithm() {
        return digest.getAlgorithm();
    }

    public long getLength() {
        return length;
    }

    public String getChecksum() {
        if (checksum == null) {
            throw new IllegalStateException("Checksum not calculated yet.");
        }
        return checksum;
    }

    public void reset() {
        digest.reset();
    }

    public void update(byte[] bytes, int off, int length) {
        digest.update(bytes, off, length);
    }

    public void calc() {
        if (checksum != null) {
            throw new IllegalStateException("Checksum already calculated.");
        }
        // Encodes a byte array into a String that should be the length of the type (2 chars per byte)
        byte[] bytes = digest.digest();
        if (bytes.length * 2 != length) {
            int bitLength = bytes.length * 8;
            throw new IllegalArgumentException(
                    "Unrecognised length for binary data: " + bitLength + " bits instead of " + (length * 4));
        }
        StringBuilder sb = new StringBuilder();
        for (byte aBinaryData : bytes) {
            String t = Integer.toHexString(aBinaryData & 0xff);
            if (t.length() == 1) {
                sb.append("0");
            }
            sb.append(t);
        }
        checksum = sb.toString().trim();
    }

}
