/*
 * Decompiled with CFR 0.152.
 */
package org.jfrog.security.crypto;

import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Objects;
import javax.annotation.Nonnull;
import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.spec.GCMParameterSpec;
import org.apache.commons.codec.digest.DigestUtils;
import org.jfrog.security.crypto.JFrogCryptoHelper;
import org.jfrog.security.crypto.encoder.EncryptedString;
import org.jfrog.security.crypto.exception.CryptoRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JFrogMasterKeyEncrypter {
    private static final Logger log = LoggerFactory.getLogger(JFrogMasterKeyEncrypter.class);
    public static final String AES_CYPHER_TRANSFORM = "AES/GCM/NoPadding";
    public static final String ALG_AES_GCM_128 = "aesgcm128";
    public static final String ALG_AES_GCM_256 = "aesgcm256";
    private static final int GCM_TAG_LENGTH = 16;
    private static final int GCM_IV_LENGTH = 12;
    private final SecretKey secretKey;
    final String keyId;
    final String alg;

    public JFrogMasterKeyEncrypter(String key) {
        this.secretKey = JFrogCryptoHelper.aesFromString(key);
        this.keyId = this.calculateKeyId(this.secretKey);
        this.alg = this.secretKey.getEncoded().length == 16 ? ALG_AES_GCM_128 : ALG_AES_GCM_256;
    }

    @Nonnull
    public EncryptedString encrypt(@Nonnull String plainText) {
        Objects.requireNonNull(plainText, "Cannot encrypt null value");
        byte[] cipherText = this.encryptInternal(plainText);
        return new EncryptedString(this.keyId, this.alg, cipherText);
    }

    private byte[] decryptInternal(byte[] cipherText) {
        try {
            Cipher cipher = Cipher.getInstance(AES_CYPHER_TRANSFORM);
            AesCipherText aesCipherText = AesCipherText.fromCipherTextWithIv(cipherText);
            GCMParameterSpec gcmParameterSpec = new GCMParameterSpec(128, aesCipherText.iv);
            cipher.init(2, (Key)this.secretKey, gcmParameterSpec);
            return cipher.doFinal(aesCipherText.cipherNoIv);
        }
        catch (GeneralSecurityException e) {
            throw new CryptoRuntimeException(e);
        }
    }

    public String decrypt(String encrypted) {
        return this.decrypt(EncryptedString.parse(encrypted));
    }

    public String decrypt(EncryptedString encrypted) {
        if (!this.isEncryptedByMe(encrypted.encode())) {
            throw new CryptoRuntimeException("Input is not encrypted by current encrypter");
        }
        return new String(this.decryptInternal(encrypted.getCipherText()), StandardCharsets.UTF_8);
    }

    public boolean isEncryptedByMe(String encryptedString) {
        if (!EncryptedString.isEncodedByMe(encryptedString)) {
            return false;
        }
        EncryptedString encrypted = EncryptedString.parse(encryptedString);
        if (!this.keyId.equalsIgnoreCase(encrypted.getKeyId())) {
            log.warn("Encrypted data with key id {} is not encrypted with current key id of {}", (Object)encrypted.getKeyId(), (Object)this.keyId);
            this.logStackTrackIfNeeded();
            return false;
        }
        if (!this.alg.equalsIgnoreCase(encrypted.getAlg())) {
            log.warn("Encrypted data with algorithm {} is not encrypted with current algorithm of {}", (Object)encrypted.getAlg(), (Object)this.alg);
            this.logStackTrackIfNeeded();
            return false;
        }
        if (encrypted.getCipherText().length < 28) {
            log.warn("Encrypted data size of {} is smaller than the minimum required of {}", (Object)encrypted.getCipherText().length, (Object)28);
            this.logStackTrackIfNeeded();
            return false;
        }
        return true;
    }

    private void logStackTrackIfNeeded() {
        if (log.isTraceEnabled()) {
            StackTraceElement[] stackTrace;
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.append("Decryption mismatch stacktrace:");
            for (StackTraceElement element : stackTrace = Thread.currentThread().getStackTrace()) {
                stringBuilder.append(System.lineSeparator());
                stringBuilder.append("\tat ").append(element);
            }
            log.trace(stringBuilder.toString());
        }
    }

    private byte[] encryptInternal(String plainText) {
        try {
            Cipher cipher = Cipher.getInstance(AES_CYPHER_TRANSFORM);
            byte[] iv = this.generateRandomInitializationVector();
            GCMParameterSpec gcmParameterSpec = new GCMParameterSpec(128, iv);
            cipher.init(1, (Key)this.secretKey, gcmParameterSpec);
            byte[] cipherNoIv = cipher.doFinal(plainText.getBytes());
            return AesCipherText.fromIvAndCipher(iv, cipherNoIv).getCipherText();
        }
        catch (GeneralSecurityException e) {
            throw new CryptoRuntimeException(e);
        }
    }

    private byte[] generateRandomInitializationVector() {
        SecureRandom random = new SecureRandom();
        byte[] iv = new byte[12];
        random.nextBytes(iv);
        return iv;
    }

    private String calculateKeyId(SecretKey secretKey) {
        return DigestUtils.sha256Hex((byte[])secretKey.getEncoded()).substring(0, 6).toLowerCase();
    }

    private static class AesCipherText {
        private final byte[] iv;
        private final byte[] cipherNoIv;

        private static AesCipherText fromIvAndCipher(byte[] iv, byte[] cipherNoIv) {
            return new AesCipherText(iv, cipherNoIv);
        }

        private static AesCipherText fromCipherTextWithIv(byte[] cipherText) {
            byte[] iv = AesCipherText.extractInitializationVector(cipherText);
            byte[] cipherNoIv = AesCipherText.removeInitializationVector(cipherText);
            return new AesCipherText(iv, cipherNoIv);
        }

        private AesCipherText(byte[] iv, byte[] cipherNoIv) {
            this.iv = iv;
            this.cipherNoIv = cipherNoIv;
        }

        private byte[] getCipherText() {
            byte[] cipherText = new byte[this.cipherNoIv.length + 12];
            System.arraycopy(this.iv, 0, cipherText, 0, this.iv.length);
            System.arraycopy(this.cipherNoIv, 0, cipherText, this.iv.length, this.cipherNoIv.length);
            return cipherText;
        }

        private static byte[] extractInitializationVector(byte[] cipherText) {
            return Arrays.copyOfRange(cipherText, 0, 12);
        }

        private static byte[] removeInitializationVector(byte[] cipherText) {
            return Arrays.copyOfRange(cipherText, 12, cipherText.length);
        }
    }
}

