package org.jfrog.security.crypto.signing.jws;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.JWSHeader;
import com.nimbusds.jose.JWSObject;
import com.nimbusds.jose.JWSSigner;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.Payload;
import com.nimbusds.jose.crypto.ECDSASigner;
import com.nimbusds.jose.crypto.RSASSASigner;
import com.nimbusds.jose.crypto.factories.DefaultJWSVerifierFactory;
import com.nimbusds.jose.util.Base64URL;
import org.jfrog.security.crypto.exception.CryptoRuntimeException;
import org.jfrog.security.crypto.signing.gpg.PGPKeyParser;
import org.jfrog.security.crypto.signing.gpg.WrappedPrivateKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.interfaces.ECPrivateKey;
import java.util.Arrays;
import java.util.List;


/**
 * Created by tomerm on 15.6.17.
 */
public class JwsImpl implements Jws {

    private static final Logger log = LoggerFactory.getLogger(JwsImpl.class);

    @Override
    public String createJwsSignature(byte[] jsonPayload, byte[] privateKeyBytes, String passphrase) {
        try {
            Payload payload = new Payload(jsonPayload);
            WrappedPrivateKey wrappedPrivateKey = PGPKeyParser.privateKeyParse(privateKeyBytes, passphrase);
            PrivateKey privateKey = wrappedPrivateKey.getPrivateKey();
            String kid = wrappedPrivateKey.getKid();
            JWSObject jwsObject = initAndSignJwsObject(payload, wrappedPrivateKey, privateKey, kid);
            return jwsObject.serialize();
        } catch (Exception e) {
            log.error("Error Creating JWS " + e.getMessage());
            throw new CryptoRuntimeException(e);
        }
    }

    @SuppressWarnings("ConstantConditions")
    private JWSObject initAndSignJwsObject(Payload payload, WrappedPrivateKey wrappedPrivateKey,
            PrivateKey privateKey, String kid) throws JOSEException {
        JWSHeader headers;
        JWSSigner signer;
        String jwsAlgorithmName = wrappedPrivateKey.getAlgorithmName();
        if ("RSA".equals(jwsAlgorithmName)) {
            headers = new JWSHeader.Builder(JWSAlgorithm.RS256).keyID(kid).build();
            signer = new RSASSASigner(privateKey);
        } else {
            headers = new JWSHeader.Builder(JWSAlgorithm.parse(jwsAlgorithmName)).keyID(kid).build();
            signer = new ECDSASigner((ECPrivateKey) privateKey);
        }
        JWSObject jwsObject = new JWSObject(headers, payload);
        jwsObject.sign(signer);
        return jwsObject;
    }

    @Override
    public boolean verifyJwsSignature(byte[] publicKeyBytes, String base64JwsSignedString) {
        try {
            DefaultJWSVerifierFactory factory = new DefaultJWSVerifierFactory();
            PublicKey publicKey = PGPKeyParser.publicKeyParse(publicKeyBytes);
            List<Base64URL> sigParts = getSigParts(base64JwsSignedString);
            JWSObject jwsObject = new JWSObject(sigParts.get(0), sigParts.get(1), sigParts.get(2));
            JWSVerifier verifier = factory.createJWSVerifier(jwsObject.getHeader(), publicKey);
            return jwsObject.verify(verifier);
        } catch (Exception e) {
            log.error("Error Verifying JWS " + e.getMessage());
            throw new CryptoRuntimeException(e);
        }
    }

    private static List<Base64URL> getSigParts(String base64JwsSignedString) {
        String[] sigParts = base64JwsSignedString.split("\\.");
        return Arrays.asList(new Base64URL(sigParts[0]), new Base64URL(sigParts[1]), new Base64URL(sigParts[2]));
    }
}
