/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2016 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.security.crypto;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.bouncycastle.util.encoders.Hex;
import org.jfrog.security.crypto.encrypter.BytesEncrypterHelper;
import org.jfrog.security.crypto.exception.CryptoException;
import org.jfrog.security.crypto.exception.CryptoRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.PBEParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.spec.EncodedKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.InvalidParameterSpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.HashMap;
import java.util.Map;

import static java.util.UUID.randomUUID;

/**
 * @author Fred Simon on 4/30/16.
 */
public abstract class JFrogCryptoHelper {

    private static final Logger log = LoggerFactory.getLogger(JFrogCryptoHelper.class);

    static final String ASYM_ALGORITHM = "RSA";
    public static final String COLON = ":";

    public static final String AES_ALGORITHM = "AES";
    static final String AES_CYPHER_INSTANCE = "AES/CBC/PKCS5Padding";
    static final String AES_SYM_ALGORITHM = "PBKDF2WithHmacSHA1";
    static final int AES_ITERATION_COUNT = 65536;
    static final int AES_KEY_SIZE = 128;
    private static final byte[] AES_SALT = new byte[]{
            (byte) 0xF5, (byte) 0xED, (byte) 0x51, (byte) 0x9A,
            (byte) 0x51, (byte) 0x93, (byte) 0x32, (byte) 0x76
    };

    // DESede is also called "TripleDES"
    static final String DESEDE_SYM_ALGORITHM = "PBEWithSHA1AndDESede";
    private static final byte[] PBE_SALT = new byte[]{
            (byte) 0xCA, (byte) 0xFE, (byte) 0xBA, (byte) 0xBE,
            (byte) 0xEB, (byte) 0xAB, (byte) 0xEF, (byte) 0xAC
    };
    private static final int PBE_ITERATION_COUNT = 20;

    private static final int DEFAULT_KEY_SIZE = 512;


    public static final Map<CipherAlg, String> SymCipherPBE = createSymCipherPBEMap();

    private static Map<CipherAlg, String> createSymCipherPBEMap() {
        Map<CipherAlg, String> res = new HashMap<>();
        res.put(CipherAlg.DESede, DESEDE_SYM_ALGORITHM);
        res.put(CipherAlg.AES128, AES_SYM_ALGORITHM);
        // java requires additional to
        //static final String AES256_SYM_ALGORITHM = "PBKDF2WithHmacSHA256";
        //res.put(CipherAlg.AES256, AES256_SYM_ALGORITHM);
        return res;
    }


    private JFrogCryptoHelper() {
        // utility class
    }

    public static String generateUniqueApiKeyToken() throws GeneralSecurityException {
        String data = String.valueOf(System.currentTimeMillis()) + COLON + randomUUID().toString();
        return EncodingType.ARTIFACTORY_API_KEY.encode(EncodingType.stringToBytes(data));
    }

    public static KeyPair generateKeyPair() {
        return generateKeyPair(DEFAULT_KEY_SIZE);
    }

    public static KeyPair generateKeyPair(int keySize) {
        try {
            KeyPairGenerator keyGen = KeyPairGenerator.getInstance(ASYM_ALGORITHM);
            SecureRandom random = SecureRandom.getInstance("SHA1PRNG");
            keyGen.initialize(keySize, random);
            return keyGen.generateKeyPair();
        } catch (NoSuchAlgorithmException e) {
            throw new CryptoRuntimeException(e);
        }
    }

    static KeyPair convertToKeyPair(byte[] encodedPrivateKey, byte[] encodedPublicKey) {
        try {
            KeyFactory generator = KeyFactory.getInstance(ASYM_ALGORITHM);

            EncodedKeySpec privateKeySpec = new PKCS8EncodedKeySpec(encodedPrivateKey);
            PrivateKey privateKey = generator.generatePrivate(privateKeySpec);

            EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(encodedPublicKey);
            PublicKey publicKey = generator.generatePublic(publicKeySpec);

            return new KeyPair(publicKey, privateKey);
        } catch (Exception e) {
            throw new CryptoRuntimeException("Failed to create KeyPair from provided encoded keys", e);
        }
    }

    static PublicKey convertToPublicKey(byte[] encodedPublicKey) {
        try {
            KeyFactory generator = KeyFactory.getInstance(ASYM_ALGORITHM);
            EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(encodedPublicKey);
            return generator.generatePublic(publicKeySpec);
        } catch (Exception e) {
            throw new CryptoRuntimeException("Failed to create PublicKey from provided encoded key", e);
        }
    }

    static public SecretKey generatePbeKeyFromKeyPair(KeyPair keyPair) {
        return generatePbeKey(EncodingType.bytesToString(Base64.encodeBase64(keyPair.getPrivate().getEncoded())));
    }

    public static SecretKey generatePbeKey(String password) {
        try {
            PBEKeySpec pbeKeySpec = new PBEKeySpec(password.toCharArray());
            SecretKeyFactory keyFactory = SecretKeyFactory.getInstance(DESEDE_SYM_ALGORITHM);
            return keyFactory.generateSecret(pbeKeySpec);
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new CryptoRuntimeException(e);
        }
    }

    public static SecretKey generateAesKeyFromKeyPair(KeyPair keyPair) {
        // Using the bytes of the private key to initialize an AES key
        byte[] bytes = keyPair.getPrivate().getEncoded();
        return generateAesKeyFromPrivateKeyBytes(bytes);
    }

    public static SecretKey generateAesKeyFromPrivateKeyBytes(byte[] bytes) {
        // Salt made of the first bytes of the private key
        byte[] aesSalt = new byte[AES_SALT.length];
        System.arraycopy(bytes, 0, aesSalt, 0, aesSalt.length);

        // Using a salted SHA256 of the private key as a password
        byte[] forSha256 = new byte[2 * AES_SALT.length + bytes.length];
        System.arraycopy(AES_SALT, 0, forSha256, 0, AES_SALT.length);
        System.arraycopy(bytes, 0, forSha256, AES_SALT.length, bytes.length);
        System.arraycopy(AES_SALT, 0, forSha256, AES_SALT.length + bytes.length, AES_SALT.length);
        String password = JFrogBase58.encode(JFrogBase58.getSha256MessageDigest().digest(forSha256));

        return generateAesKeyUsingPbe(password, aesSalt);
    }

    static SecretKey generateAesKeyUsingPbe(String password, byte[] salt) {
        try {
            if (salt == null) {
                salt = AES_SALT;
            }
            PBEKeySpec pbeKeySpec = new PBEKeySpec(password.toCharArray(), salt, AES_ITERATION_COUNT, AES_KEY_SIZE);
            SecretKeyFactory keyFactory = SecretKeyFactory.getInstance(AES_SYM_ALGORITHM);
            SecretKey tempSecretKey = keyFactory.generateSecret(pbeKeySpec);
            return new SecretKeySpec(tempSecretKey.getEncoded(), AES_ALGORITHM);
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new CryptoRuntimeException(e);
        }
    }

    static SecretKey generateAesKey() {
        try {
            KeyGenerator keyGen = KeyGenerator.getInstance(AES_ALGORITHM);
            keyGen.init(AES_KEY_SIZE);
            return keyGen.generateKey();
        } catch (NoSuchAlgorithmException e) {
            log.error("Could not generate key. {}", e.getMessage());
            log.trace("Could not generate key.");
            throw new RuntimeException("Could not generate key. " + e.getMessage());
        }
    }

    public static String generateAES128SymKey() {
        SecretKey secretKey = JFrogCryptoHelper.generateAesKey();
        try {
            String keyId = BytesEncrypterHelper.keyIdFromKey(secretKey.getEncoded());
            return JFrogEnvelop.encode(EncodingType.SYMMETRIC_KEY,
                    keyId, CipherAlg.AES128, secretKey.getEncoded());
        } catch (CryptoException e) {
            throw new RuntimeException("Can't generate keyId", e);
        }
    }

    static SecretKey loadAesKeyFromFile(File keyFile) {
        if (keyFile == null) {
            throw new IllegalArgumentException("Key file must be supplied");
        }
        if (!keyFile.exists()) {
            throw new RuntimeException("Could not find key file: " + keyFile.getAbsolutePath());
        }

        try (BufferedReader reader = new BufferedReader(new FileReader(keyFile))) {
            String aesKeyFileAsString = reader.readLine();
            byte[] aesKeyFileBytes =
                    StringUtils.isNotBlank(aesKeyFileAsString) ? sanitizedKey(aesKeyFileAsString).getBytes() : null;
            validateAesKey(aesKeyFileBytes);
            byte[] decodedKey = Hex.decode(aesKeyFileBytes);
            return getAesSecret(decodedKey);
        } catch (IOException e) {
            log.error("Could not load key. {}", e);
            log.trace("Could not load key.");
            throw new CryptoRuntimeException("Could not load key. " + e.getMessage());
        }
    }

    @Nonnull
    static SecretKey aesFromString(String key) {
        key = sanitizedKey(key);
        validateAesKey(key);
        byte[] decodeKey = Hex.decode(key);
        return getAesSecret(decodeKey);
    }

    public static SecretKey getAesSecret(byte[] decodedKey) {
        return new SecretKeySpec(decodedKey, AES_ALGORITHM);
    }

    private static String sanitizedKey(String key) {
        if (StringUtils.isNotBlank(key)) {
            return StringUtils.substringBefore(key, System.lineSeparator()).trim();
        }
        return "";
    }

    public static AesEncryptData encryptAes(byte[] bytes, SecretKey aesKey) {
        if (!AES_ALGORITHM.equals(aesKey.getAlgorithm())) {
            throw new IllegalArgumentException("This method encrypts using AES keys only not " + aesKey.getAlgorithm());
        }
        try {
            Cipher aesCipher = Cipher.getInstance(AES_CYPHER_INSTANCE);
            aesCipher.init(Cipher.ENCRYPT_MODE, aesKey);
            return new AesEncryptData(aesCipher.getParameters().getParameterSpec(IvParameterSpec.class),
                    aesCipher.doFinal(bytes));
        } catch (NoSuchAlgorithmException | NoSuchPaddingException | BadPaddingException | IllegalBlockSizeException |
                InvalidKeyException | InvalidParameterSpecException e) {
            throw new CryptoRuntimeException(e);
        }
    }

    public static byte[] decryptAes(AesEncryptData data, SecretKey aesKey) {
        if (!AES_ALGORITHM.equals(aesKey.getAlgorithm())) {
            throw new IllegalArgumentException("This method encrypts using AES keys only not " + aesKey.getAlgorithm());
        }
        try {
            Cipher aesCipher = Cipher.getInstance(AES_CYPHER_INSTANCE);
            aesCipher.init(Cipher.DECRYPT_MODE, aesKey, data.getInitVector());
            return aesCipher.doFinal(data.getEncryptedData());
        } catch (NoSuchAlgorithmException | NoSuchPaddingException | BadPaddingException | IllegalBlockSizeException |
                InvalidKeyException | InvalidAlgorithmParameterException e) {
            throw new CryptoRuntimeException(e);
        }
    }

    public static byte[] encryptSymmetric(byte[] bytes, SecretKey pbeKey) {
        if (AES_ALGORITHM.equals(pbeKey.getAlgorithm())) {
            throw new IllegalArgumentException("This method cannot encrypt using AES keys!");
        }
        try {
            Cipher pbeCipher = Cipher.getInstance(DESEDE_SYM_ALGORITHM);
            PBEParameterSpec pbeParamSpec = new PBEParameterSpec(PBE_SALT, PBE_ITERATION_COUNT);
            pbeCipher.init(Cipher.ENCRYPT_MODE, pbeKey, pbeParamSpec);
            return pbeCipher.doFinal(bytes);
        } catch (Exception e) {
            throw new CryptoRuntimeException(e);
        }
    }

    public static byte[] decryptSymmetric(byte[] encryptedBytes, SecretKey pbeKey) {
        if (AES_ALGORITHM.equals(pbeKey.getAlgorithm())) {
            throw new IllegalArgumentException("This method cannot decrypt using AES keys!");
        }
        try {
            Cipher pbeCipher = Cipher.getInstance(DESEDE_SYM_ALGORITHM);
            PBEParameterSpec pbeParamSpec = new PBEParameterSpec(PBE_SALT, PBE_ITERATION_COUNT);
            pbeCipher.init(Cipher.DECRYPT_MODE, pbeKey, pbeParamSpec);
            return pbeCipher.doFinal(encryptedBytes);
        } catch (Exception e) {
            throw new CryptoRuntimeException(e);
        }
    }

    static byte[] encryptAsymmetric(byte[] bytes, PublicKey publicKey) {
        try {
            Cipher asymCipher = Cipher.getInstance(ASYM_ALGORITHM);
            asymCipher.init(Cipher.ENCRYPT_MODE, publicKey);
            return asymCipher.doFinal(bytes);
        } catch (Exception e) {
            throw new CryptoRuntimeException(e);
        }
    }

    static byte[] decryptAsymmetric(byte[] bytes, PrivateKey privateKey) {
        try {
            Cipher asymCipher = Cipher.getInstance(ASYM_ALGORITHM);
            asymCipher.init(Cipher.DECRYPT_MODE, privateKey);
            return asymCipher.doFinal(bytes);
        } catch (Exception e) {
            throw new CryptoRuntimeException(e);
        }
    }

    public static EncodedKeyPair encodeKeyPair(KeyPair keyPair) {
        return new EncodedKeyPair(
                EncodingType.SAVED_PRIVATE_KEY.encode(keyPair.getPrivate().getEncoded()),
                EncodingType.SAVED_PUBLIC_KEY.encode(keyPair.getPublic().getEncoded())
        );
    }

    public static String debugMessageForSensitiveStrings(String... fields) {
        StringBuilder result = new StringBuilder("[");
        for (String field : fields) {
            if (result.length() > 1) {
                result.append(',');
            }
            if (field == null) {
                result.append("null");
            } else {
                result.append('\'');
                for (int i = 0; i < field.length(); i++) {
                    char c = field.charAt(i);
                    if (i < 4) {
                        result.append(c);
                    } else {
                        result.append('X');
                    }
                    if (i >= 10) {
                        result.append('_').append(field.length());
                        break;
                    }
                }
                result.append('\'');
            }
        }
        result.append(']');
        return result.toString();
    }

    private static void validateAesKey(String key) {
        if (StringUtils.isBlank(key)) {
            throw new CryptoRuntimeException("Key cannot be empty.");
        }
        byte[] keyBytes = key.getBytes();
        validateAesKey(keyBytes);
    }

    private static void validateAesKey(byte[] keyBytes) {
        if (keyBytes == null || keyBytes.length == 0) {
            throw new CryptoRuntimeException("Invalid empty key. Key must be 128 or 256 bits hexadecimal encoded.");
        }

        byte[] decodedKeyBytes;
        try {
            decodedKeyBytes = Hex.decode(keyBytes);
        } catch (Exception e) {
            throw new CryptoRuntimeException("Could not decode key. " +
                    "Key must be 128 or 256 bits hexadecimal encoded.", e);
        }
        if (decodedKeyBytes.length != 16 && decodedKeyBytes.length != 32) {
            throw new CryptoRuntimeException("Invalid key size of " + decodedKeyBytes.length * 8 + " bits. " +
                    "Expected key size is 128 or 256 bits.");
        }
    }
}
