package org.jfrog.security.crypto;

import org.jfrog.security.crypto.encrypter.AesBytesEncrypter;
import org.jfrog.security.crypto.encrypter.BytesEncrypterBase;
import org.jfrog.security.crypto.encrypter.BytesEncrypterHelper;
import org.jfrog.security.crypto.encrypter.DESedeBytesEncrypter;
import org.jfrog.security.crypto.result.DecryptionStatusHolder;
import org.jfrog.security.file.SecurityFolderHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.SecretKey;
import java.io.File;
import java.io.FileFilter;
import java.security.KeyPair;
import java.security.PublicKey;
import java.util.ArrayList;

import static org.jfrog.security.crypto.EncodingType.ARTIFACTORY_MASTER;
import static org.jfrog.security.crypto.EncodingType.ARTIFACTORY_PASSWORD;
import static org.jfrog.security.crypto.EncodingType.JFMC_MASTER;
import static org.jfrog.security.crypto.EncodingType.JFROG_AES_256;
import static org.jfrog.security.crypto.EncodingType.JFROG_TOKEN;
import static org.jfrog.security.crypto.EncodingType.MASTER_LEVEL;
import static org.jfrog.security.crypto.EncodingType.SYMMETRIC_KEY;

/**
 * @author Fred Simon on 8/19/16.
 */
public abstract class EncryptionWrapperFactory {
    private static final Logger log = LoggerFactory.getLogger(EncryptionWrapperFactory.class);

    // This method is being used by Both Artifactory and Access
    // Access need artifactory.key for legacy:
    // 1. The password in the db.properties was encrypted by artifactory.key until version 5.7
    // 2. Convert communication key to master.key  
    public static EncryptionWrapper createArtifactoryKeyWrapper(File masterKeyFile, File masterKeyChainDir,
            int numOfOldKeys, FileFilter keyFileFilter) {
        EncryptionWrapper res = createMultiKeyFileWrapper(
                masterKeyFile, masterKeyChainDir, numOfOldKeys, keyFileFilter);
        log.debug("createArtifactoryKeyWrapper  {}", res);
        return res;
    }

    private static EncryptionWrapper getMasterKeyEncryptionWrapper(SecretKey secretKey) {
        return new EncryptionWrapperBase(MASTER_LEVEL, new AesBytesEncrypter(secretKey), null, FormatUsed.OldFormat);
    }

    /**
     * This method is being used for Artifactory and Access Master.key
     * MASTER KEY USE  of aesKeyWrapperFrom.... and createAesKey
     * ALL From : org.jfrog.access.server.startup.AccessStartupServiceImpl#getLocalMasterKeyWrapper()
     */
    public static EncryptionWrapper aesKeyWrapperFromString(String key) {
        SecretKey secretKey = JFrogCryptoHelper.aesFromString(key);
        return getMasterKeyEncryptionWrapper(secretKey);
    }

    /**
     * This method is being used for Artifactory and Access Master.key
     * MASTER KEY USE  of aesKeyWrapperFrom.... and createAesKey
     * ALL From : org.jfrog.access.server.startup.AccessStartupServiceImpl#getLocalMasterKeyWrapper()
     */
    public static EncryptionWrapper createAesKeyFile() {
        SecretKey secretKey = JFrogCryptoHelper.generateAesKey();
        return getMasterKeyEncryptionWrapper(secretKey);
    }

    /**
     * This method is being used by Access generating MasterKey and AOL(Call to store)
     */
    public static EncryptionWrapper createKeyWrapper(EncodingType encodingType, String symmetricKeyPass) {
        if (MASTER_LEVEL.equals(encodingType) || JFROG_AES_256.equals(encodingType)) {
            SecretKey aesKey = JFrogCryptoHelper.generateAesKeyUsingPbe(symmetricKeyPass, null);
            return new EncryptionWrapperBase(encodingType, new AesBytesEncrypter(aesKey), null, FormatUsed.OldFormat);
            //return new AesKeyEncryptionWrapper(encodingType, aesKey);
        } else {
            SecretKey secretKey = JFrogCryptoHelper.generatePbeKey(symmetricKeyPass);
            return new EncryptionWrapperBase(encodingType, new DESedeBytesEncrypter(secretKey), null,
                    FormatUsed.OldFormat);
        }
    }

    /**
     * This method is being used crypto CLI
     * SHOULD BE MASTER ONLY
     */
    public static EncryptionWrapper createKeyWrapper(EncryptionWrapper masterWrapper, EncodingType encodingType,
            EncodedKeyPair encodedKeyPair) {
        KeyPair keyPair = encodedKeyPair.decode(masterWrapper, new DecryptionStatusHolder()).createKeyPair();

        if (MASTER_LEVEL.equals(encodingType) || JFROG_AES_256.equals(encodingType)) {
            BytesEncrypterBase aesEncrypter = BytesEncrypterHelper.getAESEncrypterCreateSecret(keyPair);
            return new EncryptionWrapperBase(encodingType, aesEncrypter, null, FormatUsed.OldFormat);
        } else {
            BytesEncrypterBase desedeEncrypter = BytesEncrypterHelper.getDesedeEncrypterCreateSecret(keyPair);
            ArrayList<BytesEncrypterBase> decrypters = new ArrayList<>();
            return new EncryptionWrapperBase(encodingType, desedeEncrypter, decrypters, FormatUsed.OldFormat);
        }
    }

    /**
     * Used for Artifactory USER LEVEL ENCRYPTION in DB
     */
    public static EncryptionWrapper createKeyWrapper(EncryptionWrapper masterWrapper, EncodedKeyPair encodedKeyPair) {
        return createKeyWrapper(masterWrapper, ARTIFACTORY_PASSWORD, encodedKeyPair);
    }

    /**
     * Used for Artifactory USER LEVEL ENCRYPTION in DB
     * used for Decrypting user password. takes user cert and enc-pass create a secret and decrypt using that password
     */
    public static EncryptionWrapper createKeyWrapper(DecodedKeyPair decodedKeyPair) {
        BytesEncrypterBase desedeEncrypter =
                BytesEncrypterHelper.getDesedeEncrypterCreateSecret(decodedKeyPair.createKeyPair());
        return new EncryptionWrapperBase(ARTIFACTORY_PASSWORD, desedeEncrypter, null, FormatUsed.OldFormat);
    }

    /**
     * Ensure that the keyPair private key and public keys are matches by encrypt text asymmetric, decrypt asymmetric
     * and eventually compare the original text and the decrypted text
     *
     * @param keyPair The keyPair to match it's keys
     */
    public static void ensureMatchingPrivatePublicKeys(KeyPair keyPair) {
        EncodedKeyPair encodedKeyPair = JFrogCryptoHelper.encodeKeyPair(keyPair);
        EncryptionWrapperBase asymKeyWrapper = EncryptionWrapperFactory.createAsymKeyWrapper(encodedKeyPair);
        asymKeyWrapper.ensureMatchingPrivatePublicKeys();
    }

    /**
     * Used for user passwords encryption with artifactory.key
     */
    static EncryptionWrapper createMultiKeyFileWrapper(File currentMasterKeyFile, File masterKeyChainDir,
            int numOfOldKeys, FileFilter keyFileFilter) {
        KeyTool keyTool = new KeyTool(currentMasterKeyFile, masterKeyChainDir, numOfOldKeys, keyFileFilter);
        return createMultiKeyFileEncryptionWrapper(ARTIFACTORY_MASTER, keyTool);
    }

    /**
     * Used for checking if pair match. (later saved and used)
     */
    private static EncryptionWrapperBase createAsymKeyWrapper(EncodedKeyPair encodedKeyPair) {
        DecodedKeyPair decodedKeyPair = encodedKeyPair.decode(null, new DecryptionStatusHolder());
        AsymKeyBytesEncrypter encrypter;
        if (encodedKeyPair.hasPrivateKey()) {
            KeyPair keyPair = decodedKeyPair.createKeyPair();
            encrypter = new AsymKeyBytesEncrypter(keyPair.getPublic(), keyPair.getPrivate());
            return new EncryptionWrapperBase(JFROG_TOKEN, encrypter, null, FormatUsed.OldFormat);
        } else {
            PublicKey publicKey = decodedKeyPair.createPublicKey();
            encrypter = new AsymKeyBytesEncrypter(publicKey, null);
        }
        return new EncryptionWrapperBase(JFROG_TOKEN, encrypter, null, FormatUsed.OldFormat);
    }

    /**
     * Used by Artifactory.key and JFMC
     * Format chosen is DotFormat unless we use DESede - the old algorithm
     */
    private static EncryptionWrapperBase createMultiKeyFileEncryptionWrapper(EncodingType targetEncodingType,
            KeyTool keyTool) {
        BytesEncrypterBase topEncryptor = createTopEncryptor(keyTool);
        FormatUsed useFormat = FormatUsed.DotFormat;
        if (CipherAlg.DESede.equals(topEncryptor.getCipherAlg())) {
            // OLD encryption format chosen
            useFormat = FormatUsed.OldFormat;
        }
        return new EncryptionWrapperBase(targetEncodingType, topEncryptor,
                BytesEncrypterHelper.buildFallbackBytesEncryptersList(keyTool.getKeyFiles()), useFormat);
    }

    /**
     * create MultiKeyEncrypter: topEncrypter with decrypt fallback list.
     */
    private static BytesEncrypterBase createTopEncryptor(KeyTool keyTool) {
        BytesEncrypterBase topEncrypter;
        JFrogEnvelop topKey = keyTool.getTopKey();
        if (SYMMETRIC_KEY.equals(topKey.encodingType)) {
            topEncrypter = BytesEncrypterHelper.getBytesEncrypterForSymKey(topKey);
        } else { // must be some keyPair file
            KeyPair keyPair = SecurityFolderHelper.getKeyPairFromFile(keyTool.currentMasterKeyFile);
            topEncrypter = BytesEncrypterHelper.getDesedeBytesEncrypterForKeyPair(keyPair);
        }
        return topEncrypter;
    }

    // ??? do we need to use the same for mission control wrapper
    static EncryptionWrapper createMissionControlMasterWrapper(File masterKeyFile, File masterKeyChainDir,
            FileFilter keyFileFilter) {
        KeyTool keyTool = new KeyTool(masterKeyFile, masterKeyChainDir, 0, keyFileFilter);

        return createMultiKeyFileEncryptionWrapper(JFMC_MASTER, keyTool);
    }

    static EncryptionWrapper createMultiKeyFileWrapperForTest(
            File currentMasterKeyFile, File masterKeyChainDir,
            FileFilter keyFileFilter) {
        KeyTool keyTool = new KeyTool(currentMasterKeyFile, masterKeyChainDir, 0, keyFileFilter);

        return createMultiKeyFileEncryptionWrapper(
                ARTIFACTORY_MASTER,
                keyTool);
    }
}
