package org.jfrog.security.ssl;

import org.apache.commons.codec.digest.DigestUtils;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x500.style.IETFUtils;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.Extensions;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.cert.jcajce.JcaX509CertificateHolder;
import org.bouncycastle.x509.extension.X509ExtensionUtil;
import org.jfrog.security.util.BCProviderFactory;

import java.io.IOException;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.security.KeyPair;
import java.security.cert.CertificateEncodingException;
import java.util.Date;
import java.util.UUID;

/**
 * @author Yinon Avraham.
 */
public abstract class CertificateHelper {

    private CertificateHelper() {
        // utility class should not be instantiated
    }

    public static final ASN1ObjectIdentifier CERT_VERSION_OID = (new ASN1ObjectIdentifier("2.5.29.17.1")).intern();

    static {
        BCProviderFactory.getProvider(); // Make sure that the factory is initialised
    }

    public static class CryptoInfo {
        private KeyPair keyPair;
        private java.security.cert.X509Certificate certificate;

        public CryptoInfo(KeyPair keyPair, java.security.cert.X509Certificate certificate) {
            this.keyPair = keyPair;
            this.certificate = certificate;
        }

        public KeyPair getKeyPair() {
            return keyPair;
        }

        public void setKeyPair(KeyPair keyPair) {
            this.keyPair = keyPair;
        }

        public java.security.cert.X509Certificate getCertificate() {
            return certificate;
        }

        public void setCertificate(java.security.cert.X509Certificate certificate) {
            this.certificate = certificate;
        }
    }

    public static int getCertificateVersion(java.security.cert.X509Certificate certificate) throws IOException {
        byte[] version = certificate.getExtensionValue(CERT_VERSION_OID.getId());
        if (version != null) {
            Integer versionFromCertificate = null;
            GeneralNames gn = GeneralNames.getInstance(X509ExtensionUtil.fromExtensionValue(version));
            GeneralName[] names = gn.getNames();
            for (GeneralName name : names) {
                if (name.getTagNo() == GeneralName.otherName) {
                    ASN1Sequence seq = ASN1Sequence.getInstance(name.getName());
                    ASN1Integer value = (ASN1Integer) seq.getObjectAt(1);
                    versionFromCertificate = value.getValue().intValue();
                }
            }
            return versionFromCertificate;
        }
        return 0;
    }

    public static java.security.cert.X509Certificate generateRootCertificate(
            KeyPair keyPair,
            String commonName,
            int certVersion) throws CertificateGenerationException {
        BigInteger serialNumber = randomSerialNumber();

        return SignedCertificateBuilder.builder()
                .iss(commonName)
                .issPrivateKey(keyPair.getPrivate())
                .sub(commonName)
                .subPublicKey(keyPair.getPublic())
                .serialNumber(serialNumber)
                .expireIn(Long.MAX_VALUE)
                .certVersion(certVersion)
                .useSubForSAN()
                .isCA(true)
                .buildX509Certificate();

    }

    public static BigInteger randomSerialNumber() {
        UUID uuid = UUID.randomUUID();
        ByteBuffer bb = ByteBuffer.wrap(new byte[16]);
        bb.putLong(uuid.getMostSignificantBits());
        bb.putLong(uuid.getLeastSignificantBits());
        return new BigInteger(1, bb.array());
    }

    /**
     * Get the common name (CN) of the issuer of a given certificate
     *
     * @param certificate the certificate
     * @return the common name
     * @throws CertificateEncodingException
     */
    public static String getCertificateIssuerCommonName(java.security.cert.X509Certificate certificate)
            throws CertificateEncodingException {
        String result = "Unknown";
        JcaX509CertificateHolder certificateHolder = new JcaX509CertificateHolder(certificate);
        RDN[] rdNs = certificateHolder.getIssuer().getRDNs(BCStyle.CN);
        return getName(result, rdNs);
    }

    /**
     * Get the common name (CN) of the issuer of a given certificate
     *
     * @param certificate the certificate
     * @return the common name
     * @throws CertificateEncodingException
     */
    public static String getCertificateSubjectAlternativeName(java.security.cert.X509Certificate certificate)
            throws CertificateEncodingException {
        JcaX509CertificateHolder certificateHolder = new JcaX509CertificateHolder(certificate);
        Extensions extensions = certificateHolder.getExtensions();
        GeneralNames generalNames = GeneralNames.fromExtensions(extensions, Extension.subjectAlternativeName);
        if (generalNames == null) {
            return "";
        }
        GeneralName[] names = generalNames.getNames();
        if (names == null || names.length == 0) {
            return "";
        }
        GeneralName generalName = names[0]; //since we only put in one name
        return generalName.getName().toString();

    }

    public static String getCertificateSubjectCommonName(java.security.cert.X509Certificate certificate)
            throws CertificateEncodingException {
        String result = "Unknown";
        JcaX509CertificateHolder certificateHolder = new JcaX509CertificateHolder(certificate);
        RDN[] rdNs = certificateHolder.getSubject().getRDNs(BCStyle.CN);
        return getName(result, rdNs);
    }

    public static Date getIssuedAt(java.security.cert.X509Certificate certificate) {
        return certificate.getNotBefore();
    }

    public static String getCertificateIssuerOrganizationUnit(java.security.cert.X509Certificate certificate)
            throws CertificateEncodingException {
        String result = "Unknown";
        JcaX509CertificateHolder certificateHolder = new JcaX509CertificateHolder(certificate);
        RDN[] rdNs = certificateHolder.getIssuer().getRDNs(BCStyle.OU);
        return getName(result, rdNs);
    }

    public static String getCertificateIssuerOrganization(java.security.cert.X509Certificate certificate)
            throws CertificateEncodingException {
        String result = "Unknown";
        JcaX509CertificateHolder certificateHolder = new JcaX509CertificateHolder(certificate);
        RDN[] rdNs = certificateHolder.getIssuer().getRDNs(BCStyle.O);
        return getName(result, rdNs);
    }

    public static String getCertificateSubjectOrganizationUnit(java.security.cert.X509Certificate certificate)
            throws CertificateEncodingException {
        String result = "Unknown";
        JcaX509CertificateHolder certificateHolder = new JcaX509CertificateHolder(certificate);
        RDN[] rdNs = certificateHolder.getSubject().getRDNs(BCStyle.OU);
        return getName(result, rdNs);
    }

    public static String getCertificateSubjectOrganization(java.security.cert.X509Certificate certificate)
            throws CertificateEncodingException {
        String result = "Unknown";
        JcaX509CertificateHolder certificateHolder = new JcaX509CertificateHolder(certificate);
        RDN[] rdNs = certificateHolder.getSubject().getRDNs(BCStyle.O);
        return getName(result, rdNs);
    }

    public static Date getValidUntil(java.security.cert.X509Certificate certificate) {
        return certificate.getNotAfter();
    }

    public static String getCertificateFingerprint(java.security.cert.X509Certificate certificate)
            throws CertificateEncodingException {
        byte[] bytes = DigestUtils.sha256(certificate.getEncoded());
        StringBuilder builder = new StringBuilder();
        for (int i = 0; i < bytes.length; i++) {
            String appendString = Integer.toHexString(0xFF & bytes[i]);
            if (appendString.length() == 1) {
                builder.append("0");
            }
            builder.append(appendString.toUpperCase());
            if (i != bytes.length - 1) {
                builder.append(':');
            }
        }
        return builder.toString();
    }

    private static String getName(String result, RDN[] rdNs) {
        if (rdNs != null && rdNs.length > 0) {
            RDN firstRDN = rdNs[0];
            if (firstRDN.getFirst() != null && firstRDN.getFirst().getValue() != null) {
                result = IETFUtils.valueToString(firstRDN.getFirst().getValue());
            }
        }
        return result;
    }
}
