
package org.jfrog.security.crypto.signing.gpg;

import org.apache.commons.io.IOUtils;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.bcpg.BCPGOutputStream;
import org.bouncycastle.openpgp.*;
import org.bouncycastle.openpgp.operator.KeyFingerPrintCalculator;
import org.bouncycastle.openpgp.operator.PBESecretKeyDecryptor;
import org.bouncycastle.openpgp.operator.PGPContentVerifierBuilderProvider;
import org.bouncycastle.openpgp.operator.bc.*;
import org.iostreams.streams.in.StringInputStream;
import org.jfrog.security.util.BCProviderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;

import static org.bouncycastle.bcpg.HashAlgorithmTags.SHA256;

/**
 * Signs files/streams with PGP signature.
 *
 * @author Yossi shaul
 */
public class GpgSigner {
    private static final Logger log = LoggerFactory.getLogger(GpgSigner.class);

    static {
        // initialize provider if not already registered
        // noinspection ResultOfMethodCallIgnored
        BCProviderFactory.getProvider();
    }

    //used in package indexer
    public static String signFile(String privateKey, String password, InputStream content) throws Exception {
        return signFile(privateKey, password, false, content);
    }

    /**
     * Signing the content using the private key and password. Also, if includeBeginSignedMessageAndContent set to true, adds the clearText sign:
     * -----BEGIN PGP SIGNED MESSAGE----- and the content to the signature. This requires for example for Debian InRelease, for example:
     *
     * -----BEGIN PGP SIGNED MESSAGE-----
     * Hash: SHA256
     *
     * Origin: Artifactory
     * Label: Artifactory
     * ......
     * -----BEGIN PGP SIGNATURE-----
     * ....
     * -----END PGP SIGNATURE-----
     *
     * @param privateKey                          - the private key to use
     * @param password                            - the password for the signing
     * @param includeBeginSignedMessageAndContent - should include the clear text sign
     * @param content                             - The content to sign
     */
    //used in package indexer
    public static String signFile(String privateKey, String password, boolean includeBeginSignedMessageAndContent, InputStream content) throws Exception {
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()){
            PGPSignatureGenerator signatureGenerator = getPgpSignatureGenerator(privateKey, password);
            writeSignatureToStream(signatureGenerator, includeBeginSignedMessageAndContent, out, content);
            return new String(out.toByteArray(), "UTF-8");
        } catch (Exception e) {
            throw new PGPException("Fail to sign file, please verify that there is match between the private key and the passphrase.", e);
        } finally {
            IOUtils.closeQuietly(content);
        }
    }

    //used in bintray
    public static void signContentAndWriteToOutputStream(OutputStream out, String privateKey, String password, InputStream content) throws Exception {
        try {
            PGPSignatureGenerator signatureGenerator = getPgpSignatureGenerator(privateKey, password);
            writeSignatureToStream(signatureGenerator, false, out, content);
        } catch (Exception e) {
            throw new PGPException("Fail to sign file, please verify that there is match between the private key and the passphrase.", e);
        } finally {
            IOUtils.closeQuietly(content);
        }
    }

    private static void writeSignatureToStream(PGPSignatureGenerator signatureGenerator, boolean includeBeginSignedMessageAndContent, OutputStream out, InputStream content) throws PGPException, IOException {
        ArmoredOutputStream armoredOutput = new ArmoredOutputStream(out);
        if (includeBeginSignedMessageAndContent) {
            armoredOutput.beginClearText(SHA256);
        }
        byte[] buf = new byte[4096];
        int n;
        while ((n = content.read(buf)) >= 0) {
            signatureGenerator.update(buf, 0, n);
            if (includeBeginSignedMessageAndContent) {
                armoredOutput.write(buf, 0, n);
            }
        }
        if (includeBeginSignedMessageAndContent) {
            armoredOutput.write("\n".getBytes());
        }
        armoredOutput.endClearText();
        PGPSignature signature = signatureGenerator.generate();
        BCPGOutputStream bcpgOutputStream = new BCPGOutputStream(armoredOutput);
        signature.encode(bcpgOutputStream);
        armoredOutput.close();
        bcpgOutputStream.close();
    }

    private static PGPSignatureGenerator getPgpSignatureGenerator(String privateKey, String password) throws PGPException {
        PGPSecretKey pgpSecretKey = PGPKeyParser.findSecretGPGKey(privateKey.getBytes());
        PBESecretKeyDecryptor decryptor = new BcPBESecretKeyDecryptorBuilder(new BcPGPDigestCalculatorProvider()).build(password.toCharArray());
        PGPPrivateKey pgpPrivateKey = pgpSecretKey.extractPrivateKey(decryptor);
        int algorithm = pgpSecretKey.getPublicKey().getAlgorithm();
        PGPSignatureGenerator signatureGenerator = new PGPSignatureGenerator(new BcPGPContentSignerBuilder(algorithm, SHA256));
        signatureGenerator.init(PGPSignature.BINARY_DOCUMENT, pgpPrivateKey);
        return signatureGenerator;
    }

    public static boolean verifyFile(String stringPublicKey, String signature, InputStream content) throws Exception {
        try {
            InputStream keyInputStream = new StringInputStream(stringPublicKey);
            InputStream sigInputStream = PGPUtil.getDecoderStream(new StringInputStream(signature));
            KeyFingerPrintCalculator keyFingerPrintCalculator = new BcKeyFingerprintCalculator();
            PGPObjectFactory pgpObjFactory = new PGPObjectFactory(sigInputStream, keyFingerPrintCalculator);
            PGPSignatureList pgpSigList;
            Object obj = pgpObjFactory.nextObject();
            if (obj instanceof PGPCompressedData) {
                PGPCompressedData c1 = (PGPCompressedData) obj;
                pgpObjFactory = new PGPObjectFactory(c1.getDataStream(), keyFingerPrintCalculator);
                pgpSigList = (PGPSignatureList) pgpObjFactory.nextObject();
            } else {
                pgpSigList = (PGPSignatureList) obj;
            }
            PGPPublicKeyRingCollection pgpPubRingCollection = new PGPPublicKeyRingCollection(PGPUtil.getDecoderStream(keyInputStream)
                    , keyFingerPrintCalculator);
            InputStream inputStream = new BufferedInputStream(content);
            PGPSignature sig = pgpSigList.get(0);
            PGPPublicKey pubKey = pgpPubRingCollection.getPublicKey(sig.getKeyID());
            PGPContentVerifierBuilderProvider pgpContentVerifierBuilderProvider = new BcPGPContentVerifierBuilderProvider();
            sig.init(pgpContentVerifierBuilderProvider, pubKey);
            byte[] buf = new byte[4096];
            int n;
            while ((n = content.read(buf)) >= 0) {
                sig.update(buf, 0, n);
            }
            inputStream.close();
            keyInputStream.close();
            sigInputStream.close();
            return sig.verify();
        } catch (Exception e) {
            throw new PGPException("Fail to verify file, please verify that there is match between the public key, signature and the file.", e);
        } finally {
            IOUtils.closeQuietly(content);
        }
    }

    public static boolean verifyPrivateKey(String privateKey) throws Exception {
        try {
            PGPKeyParser.findSecretGPGKey(privateKey.getBytes());
            return true;
        } catch (Exception e) {
            log.info("Fail to verify private pgp: {}", privateKey, e);
            return true;
        }
    }

    public static boolean verifyPublicKey(String publicKey) throws Exception {
        try {
            InputStream keyInputStream = new StringInputStream(publicKey);
            KeyFingerPrintCalculator keyFingerPrintCalculator = new BcKeyFingerprintCalculator();
            PGPPublicKeyRingCollection pgpPublicKeyRingCollection = new PGPPublicKeyRingCollection(PGPUtil.getDecoderStream(keyInputStream)
                    , keyFingerPrintCalculator);
            return pgpPublicKeyRingCollection.getKeyRings().hasNext();
        } catch (Exception e) {
            log.info("Fail to verify public pgp: {}", publicKey, e);
            return true;
        }
    }

}