/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2016 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.security.crypto;

import org.jfrog.security.crypto.result.DecryptionBytesResult;
import org.jfrog.security.crypto.result.DecryptionStatusHolder;

import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.jfrog.security.crypto.EncodingType.*;
import static org.jfrog.security.crypto.JFrogCryptoHelper.debugMessageForSensitiveStrings;
import static org.jfrog.security.crypto.result.DecryptionStatus.SUCCESS_WITH_FALLBACK;

public class EncodedKeyPair {
    public static final String NO_PRIVATE_KEY = "NO PRIVATE KEY";
    public final String encodedPrivateKey;
    public final String encodedPublicKey;

    public EncodedKeyPair(String encodedPublicKey) {
        this(NO_PRIVATE_KEY, encodedPublicKey);
    }

    public EncodedKeyPair(String encodedPrivateKey, String encodedPublicKey) {
        if (encodedPublicKey == null) {
            throw new IllegalArgumentException("Public key cannot be null");
        }
        if (encodedPrivateKey == null) {
            this.encodedPrivateKey = NO_PRIVATE_KEY;
        } else {
            this.encodedPrivateKey = encodedPrivateKey.trim();
        }
        this.encodedPublicKey = encodedPublicKey.trim();
    }

    public EncodedKeyPair(DecodedKeyPair decodedKeyPair, EncryptionWrapper keyWrapper) {
        if (keyWrapper == null || keyWrapper.getEncodingType() == NO_ENCODING) {
            if (decodedKeyPair.hasPrivateKey()) {
                this.encodedPrivateKey = SAVED_PRIVATE_KEY.encode(decodedKeyPair.privateKey);
            } else {
                this.encodedPrivateKey = NO_PRIVATE_KEY;
            }
            this.encodedPublicKey = SAVED_PUBLIC_KEY.encode(decodedKeyPair.publicKey);
        } else {
            if (decodedKeyPair.hasPrivateKey()) {
                this.encodedPrivateKey = ARTIFACTORY_PRIVATE_KEY.encode(keyWrapper.encrypt(decodedKeyPair.privateKey));
            } else {
                this.encodedPrivateKey = NO_PRIVATE_KEY;
            }
            this.encodedPublicKey = ARTIFACTORY_PUBLIC_KEY.encode(keyWrapper.encrypt(decodedKeyPair.publicKey));
        }
        if (this.encodedPublicKey == null) {
            throw new IllegalArgumentException("Public key cannot be null");
        }
    }

    public boolean hasPrivateKey() {
        return isNotBlank(encodedPrivateKey) && !NO_PRIVATE_KEY.equals(encodedPrivateKey);
    }

    public String getEncodedPrivateKey() {
        return encodedPrivateKey;
    }

    public String getEncodedPublicKey() {
        return encodedPublicKey;
    }

    class EncodingTypes {
        final EncodingType privateEncodedBy;
        final EncodingType publicEncodedBy;

        public EncodingTypes() {
            if (hasPrivateKey()) {
                privateEncodedBy = validPrivateEncodedKey(encodedPrivateKey);
            } else {
                privateEncodedBy = null;
            }
            publicEncodedBy = validPublicEncodedKey(encodedPublicKey);
        }
    }

    public DecodedKeyPair decode(EncryptionWrapper masterWrapper, DecryptionStatusHolder statusHolder) {
        EncodingTypes encodingTypes = new EncodingTypes();
        byte[] privateKeyBytes = null;


        EncodingType publicEncodedBy = validPublicEncodedKey(encodedPublicKey);
        byte[] publicKeyBytes = publicEncodedBy.decode(encodedPublicKey);

        // not encrypted
        if (masterWrapper == null || masterWrapper.getEncodingType() == NO_ENCODING) {
            return asPlainTextDecodedKeyPair(encodingTypes, publicKeyBytes);
        }
        // Check if encrypted by the top encrypter - or fallback.
        EncryptionWrapperBase multiForFallbackCheck= (EncryptionWrapperBase) masterWrapper;
        if (hasPrivateKey()) {
            privateKeyBytes = encodingTypes.privateEncodedBy.decode(encodedPrivateKey);
            // todo can we fail to decrypt ...
            if ((encodingTypes.privateEncodedBy == ARTIFACTORY_PRIVATE_KEY) ) {
                String encodedPrivateKey = this.encodedPrivateKey;
                DecryptionBytesResult privateKeyDecryptionResult = decryptKey(multiForFallbackCheck, encodedPrivateKey, statusHolder);
                privateKeyBytes = privateKeyDecryptionResult.getDecryptedData();
            }
        }
        if (encodingTypes.publicEncodedBy == ARTIFACTORY_PUBLIC_KEY) {
            String encodedPublicKey = this.encodedPublicKey;
            DecryptionBytesResult publicKeyDecryptionResult = decryptKey(multiForFallbackCheck, encodedPublicKey, statusHolder);
            publicKeyBytes = publicKeyDecryptionResult.getDecryptedData();
        }

        return new DecodedKeyPair(privateKeyBytes, publicKeyBytes);
    }

    private DecryptionBytesResult decryptKey(EncryptionWrapperBase encrypterWrapper, String encodedKey,
            DecryptionStatusHolder statusHolder) {
        JFrogEnvelop jfe = JFrogEnvelop.parse(encodedKey);
        DecryptionBytesResult decryptionResult = encrypterWrapper.decrypt(jfe.extractBytes());
        if (SUCCESS_WITH_FALLBACK.equals(decryptionResult.getStatus())) {
            statusHolder.addSuccessWithFallback();
        }
        return decryptionResult;
    }

    private DecodedKeyPair asPlainTextDecodedKeyPair(EncodingTypes encodingTypes,
            byte[] publicKeyBytes) {
        // Cannot decode if one encoding type needs decryption and no master wrapper provided
        if (needsMasterWrapper(encodingTypes)) {
            throw new IllegalStateException("Cannot decode encrypted key pair without master wrapper for " + toString());
        }
        byte[] privateKeyBytes = null;
        if (hasPrivateKey()) {
            privateKeyBytes = encodingTypes.privateEncodedBy.decode(encodedPrivateKey);
        }
        return new DecodedKeyPair(privateKeyBytes, publicKeyBytes);
    }

    private boolean needsMasterWrapper(EncodingTypes encodedBy) {
        return encodedBy.privateEncodedBy == ARTIFACTORY_PRIVATE_KEY
                || encodedBy.publicEncodedBy == ARTIFACTORY_PUBLIC_KEY;
    }

    public EncodedKeyPair toSaveEncodedKeyPair(EncryptionWrapper artifactoryKeyWrapper) {
        EncodingType privateEncodedBy = null;
        if (hasPrivateKey()) {
            privateEncodedBy = validPrivateEncodedKey(encodedPrivateKey);
        }
        EncodingType publicEncodedBy = validPublicEncodedKey(encodedPublicKey);
        // If one using old format, needs re saving
        if (privateEncodedBy == ARTIFACTORY_MASTER || publicEncodedBy == ARTIFACTORY_MASTER) {
            return new EncodedKeyPair(decode(artifactoryKeyWrapper, new DecryptionStatusHolder()), artifactoryKeyWrapper);
        }
        // Good new format nothing to save
        return null;
    }

    private static EncodingType validPrivateEncodedKey(String privateKey) {
        EncodingType privateEncodedBy = EncodingType.findEncodedBy(privateKey);
        if (privateEncodedBy == null
                || (privateEncodedBy != ARTIFACTORY_MASTER && // Old format
                privateEncodedBy != SAVED_PRIVATE_KEY && // Non encrypted format
                privateEncodedBy != ARTIFACTORY_PRIVATE_KEY // Encrypted format
        )) {
            throw new IllegalStateException("Illegal private key encoding " + privateEncodedBy + " for " + debugMessageForSensitiveStrings(privateKey));
        }
        return privateEncodedBy;
    }

    private static EncodingType validPublicEncodedKey(String publicKey) {
        EncodingType publicEncodedBy = EncodingType.findEncodedBy(publicKey);
        if (publicEncodedBy == null
                || (publicEncodedBy != ARTIFACTORY_MASTER && // Old format
                publicEncodedBy != SAVED_PUBLIC_KEY && // Non encrypted format
                publicEncodedBy != ARTIFACTORY_PUBLIC_KEY // Encrypted format
        )) {
            throw new IllegalStateException("Illegal public key encoding " + publicEncodedBy + " for " + debugMessageForSensitiveStrings(publicKey));
        }
        return publicEncodedBy;
    }

    @Override
    public String toString() {
        return "Encoded key " + debugMessageForSensitiveStrings(encodedPrivateKey, encodedPublicKey);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        EncodedKeyPair that = (EncodedKeyPair) o;

        if (encodedPrivateKey != null ? !encodedPrivateKey.equals(that.encodedPrivateKey) : that.encodedPrivateKey != null)
            return false;
        return encodedPublicKey.equals(that.encodedPublicKey);

    }

    @Override
    public int hashCode() {
        int result = encodedPrivateKey != null ? encodedPrivateKey.hashCode() : 0;
        result = 31 * result + encodedPublicKey.hashCode();
        return result;
    }
}
