package org.jfrog.metadata.client.http;

import org.apache.commons.io.Charsets;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;
import org.apache.http.util.EntityUtils;
import org.jfrog.client.http.*;
import org.jfrog.metadata.client.SystemClient;
import org.jfrog.metadata.client.exception.MetadataClientException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.regex.Pattern;

import static org.apache.http.HttpHeaders.AUTHORIZATION;

/**
 * @author Uriah Levy
 */
public class MetadataHttpClient implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(MetadataHttpClient.class);

    private final CloseableHttpClientDecorator httpClient;
    private final String serviceAuthToken;
    private String metadataServerUrl;
    private static final String API_V1 = "/api/v1";

    public MetadataHttpClient(CloseableHttpClientDecorator httpClient, String metadataServerUrl,
            String serviceAuthToken) {
        this.metadataServerUrl = metadataServerUrl;
        this.httpClient = httpClient;
        this.serviceAuthToken = serviceAuthToken;
    }

    public RestResponse rest(@Nonnull RestRequest request, boolean throwOnUnsuccessfulResponse) {
        return rest(request, null, throwOnUnsuccessfulResponse);
    }

    public RestResponse rest(@Nonnull RestRequest request, String activePrincipalAuthToken,
            boolean throwOnUnsuccessfulResponse) {
        HttpRequestBase httpRequest = HttpRequest.fromRestRequest(request, metadataServerUrl);
        addAuthentication(activePrincipalAuthToken, httpRequest);
        log.debug("Executing : {} {}", httpRequest.getMethod(), httpRequest.getURI());
        try (CloseableHttpResponse response = httpClient.execute(httpRequest)) {
            return createRestResponse(response, throwOnUnsuccessfulResponse, request.getPath());
        } catch (IOException e) {
            if (throwOnUnsuccessfulResponse) {
                throw new MetadataClientException(
                        "Unable to send request to '" + httpRequest.getURI() + "'. Caught IO Exception: ", e);
            }
            log.error("Caught IO Exception while executing {}. Exception message: {}", request.getPath(),
                    e.getMessage());
            log.debug("", e);
        }
        return null;
    }

    /**
     * see {@link SystemClient#ping()}
     */
    public void adHocPing() {
        rest(RestRequest.get(API_V1 + "/system/ping").build(), null, true);
    }

    private void addAuthentication(String activePrincipalAuthToken, HttpRequestBase request) {
        if (StringUtils.isNotBlank(activePrincipalAuthToken)) {
            request.addHeader(new BasicHeader(AUTHORIZATION, "Bearer " + activePrincipalAuthToken));
        } else if (StringUtils.isNotBlank(serviceAuthToken)) {
            request.addHeader(new BasicHeader(AUTHORIZATION, "Bearer " + serviceAuthToken));
        }
    }

    private RestResponse createRestResponse(CloseableHttpResponse response, boolean throwOnUnsuccessfulResponse,
            String requestPath)
            throws IOException {
        int statusCode = response.getStatusLine().getStatusCode();
        HttpEntity entity = response.getEntity();
        if (throwOnUnsuccessfulResponse && statusCode >= 300) {
            try {
                String message = extractErrorMessage(response);
                throw new IllegalStateException(
                        "Failed executing " + requestPath + ", with response code: " + response.getStatusLine()
                                + " and response message: " + message);
            } finally {
                EntityUtils.consumeQuietly(entity);
            }
        }
        ContentType contentType = ContentType.getOrDefault(entity);
        Charset charset = contentType == null ? null : contentType.getCharset();
        byte[] bytes = entity == null ? null : EntityUtils.toByteArray(entity);
        return new RestResponseImpl(statusCode, bytes, charset);
    }

    private String extractErrorMessage(HttpResponse response) {
        try {
            HttpEntity entity = response.getEntity();
            if (entity != null && isApplicationJsonResponse(response) && entity.getContentLength() < 1024) {
                return IOUtils.toString(entity.getContent(), Charsets.UTF_8);
            }
        } catch (Exception e) {
            log.debug("Unable to extract error message from response.", e);
        }
        return response.getStatusLine().getReasonPhrase();
    }

    private boolean isApplicationJsonResponse(HttpResponse response) {
        Header contentTypeHeader = response.getFirstHeader(HttpHeaders.CONTENT_TYPE);
        String contentType = contentTypeHeader != null ? contentTypeHeader.getValue() : null;
        return contentType != null && CONTENT_TYPE_APP_JSON_PATTERN.matcher(contentType).matches();
    }

    private static final Pattern CONTENT_TYPE_APP_JSON_PATTERN = Pattern.compile("application/(json|.+\\+json).*");

    @Override
    public void close() {
        closeClient();
    }

    private void closeClient() {
        try {
            httpClient.close();
        } catch (IOException e) {
            throw new MetadataClientException("Failed to shutdown the http client", e);
        }
    }
}
