package org.jfrog.client.http;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;

import javax.annotation.Nonnull;
import java.net.URI;
import java.util.function.BiFunction;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * @author Yinon Avraham.
 */
public class HttpRequest extends HttpEntityEnclosingRequestBase {

    public static final String X_ORIGIN_USERNAME = "X-Origin-Username";
    public static final String X_ORIGIN_USER_ADDRESS = "X-Origin-User-Address";

    private final String method;

    HttpRequest(@Nonnull String method) {
        this.method = requireNonNull(method, "method is required").toUpperCase();
    }

    @Override
    public String getMethod() {
        return method;
    }

    public static HttpRequestBase fromRestRequest(RestRequest originalReq, String baseUrl,
            RequestConfig defaultRequestConfig) {
        HttpRequest httpRequest = new HttpRequest(originalReq.getMethod());
        if (originalReq.getBody() != null) {
            httpRequest.setEntity(new ByteArrayEntity(originalReq.getBody()));
        }
        httpRequest.setURI(createUri(originalReq, baseUrl));

        setTimeoutIfNeeded(httpRequest, defaultRequestConfig, originalReq::getSocketTimeout,
                (timeoout1, restConfigBuilder) -> restConfigBuilder.setSocketTimeout(timeoout1));
        setTimeoutIfNeeded(httpRequest, defaultRequestConfig, originalReq::getConnectTimeout,
                (timeoout1, restConfigBuilder) -> restConfigBuilder.setConnectTimeout(timeoout1));
        originalReq.getHeaders().forEach(header -> httpRequest.addHeader(header.getKey(), header.getValue()));
        return httpRequest;
    }

    private static void setTimeoutIfNeeded(HttpRequest httpRequest,
            RequestConfig defaultRequestConfig,
            Supplier<Integer> timeoutSupplier,
            BiFunction<Integer, RequestConfig.Builder, RequestConfig.Builder> configTimeoutSetter) {
        final Integer timeoout = timeoutSupplier.get();
        if (timeoout == null || timeoout <= 0) {
            return;
        }

        if (defaultRequestConfig == null) {
            throw new IllegalStateException("Default request config must be provided when setting different timeout");
        }

        final RequestConfig.Builder restConfigBuilder = RequestConfig.copy(defaultRequestConfig);
        httpRequest.setConfig(configTimeoutSetter.apply(timeoout, restConfigBuilder).build());
    }

    private static URI createUri(RestRequest originalReq, String baseUrl) {
        String uri = (isFullPath(originalReq) ? "" : baseUrl + '/')
                + StringUtils.removeStart(originalReq.getPath(), "/");
        for (Pair<String, String> param : originalReq.getQueryParams()) {
            uri = QueryParamsBuilder.param(uri, param.getKey(), param.getValue());
        }
        return URI.create(uri);
    }

    private static boolean isFullPath(RestRequest originalReq) {
        return originalReq.getPath().matches("^https?://.*");
    }
}
