package org.jfrog.client.http;

import org.apache.commons.lang3.tuple.Pair;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static java.util.Objects.requireNonNull;

/**
 * Example:
 * <pre>
 * RestRequest request = RestRequest.get("/api/v1/system/ping").build();
 * RestResponse response = accessClient.restCall(request);
 * </pre>
 * @see org.jfrog.access.client.AccessClient#restCall(RestRequest)
 * @author Yinon Avraham.
 */
public class RestRequest {

    private final String method;
    private final String path;
    private final byte[] body;
    private final List<Pair<String, String>> queryParams;
    private final List<Pair<String, String>> headers;
    private final Integer socketTimeout;

    private RestRequest(@Nonnull String method, @Nonnull String path, @Nullable byte[] body,
            @Nonnull List<Pair<String, String>> queryParams, @Nonnull List<Pair<String, String>> headers, Integer socketTimeout) {
        this.method = requireNonNull(method, "method is required");
        this.path = requireNonNull(path, "path is required");
        this.body = body == null ? null : Arrays.copyOf(body, body.length);
        this.queryParams = Collections.unmodifiableList(new ArrayList<>(queryParams));
        this.headers = Collections.unmodifiableList(new ArrayList<>(headers));
        this.socketTimeout = socketTimeout;
    }

    /**
     * Start building an HTTP request
     * @param method the request method
     * @param path the request path, relative to the base URL.
     * @return a new request builder
     */
    public static Builder request(@Nonnull String method, @Nonnull String path) {
        return new Builder(method, path);
    }

    /**
     * Start building an HTTP GET request
     * @param path the request path, relative to the base URL.
     * @return a new request builder
     */
    public static Builder get(@Nonnull String path) {
        return request("GET", path);
    }

    /**
     * Start building an HTTP POST request
     * @param path the request path, relative to the base URL.
     * @return a new request builder
     */
    public static Builder post(@Nonnull String path) {
        return request("POST", path);
    }

    /**
     * Start building an HTTP PUT request
     * @param path the request path, relative to the base URL.
     * @return a new request builder
     */
    public static Builder put(@Nonnull String path) {
        return request("PUT", path);
    }

    /**
     * Start building an HTTP PATCH request
     * @param path the request path, relative to the base URL.
     * @return a new request builder
     */
    public static Builder patch(@Nonnull String path) {
        return request("PATCH", path);
    }

    /**
     * Start building an HTTP DELETE request
     * @param path the request path, relative to the base URL.
     * @return a new request builder
     */
    public static Builder delete(@Nonnull String path) {
        return request("DELETE", path);
    }

    /**
     * Start building an HTTP HEAD request
     * @param path the request path, relative to the base URL.
     * @return a new request builder
     */
    public static Builder head(@Nonnull String path) {
        return request("HEAD", path);
    }

    @Nonnull
    public String getMethod() {
        return method;
    }

    @Nonnull
    public String getPath() {
        return path;
    }

    @Nullable
    public byte[] getBody() {
        return body;
    }

    @Nonnull
    public List<Pair<String, String>> getQueryParams() {
        return queryParams;
    }

    @Nonnull
    public List<Pair<String, String>> getHeaders() {
        return headers;
    }

    @Nullable
    public Integer getSocketTimeout() {
        return socketTimeout;
    }

    public static class Builder {
        private final String method;
        private final String path;
        private byte[] body;
        private final List<Pair<String, String>> queryParams = new ArrayList<>();
        private final List<Pair<String, String>> headers = new ArrayList<>();
        private Integer socketTimeout = 0;

        private Builder(@Nonnull String method, @Nonnull String path) {
            this.method = requireNonNull(method, "method is required");
            this.path = requireNonNull(path, "path is required");
        }

        /**
         * Set the body for the request.
         * This is relevant only for an entity enclosing request.
         * @param body the bytes with the content of the body to send, or <tt>null</tt> if there is no body to send (default).
         * @return this builder
         */
        @Nonnull
        public Builder body(byte[] body) {
            this.body = body;
            return this;
        }

        /**
         * Set the body for the request.
         * This is relevant only for an entity enclosing request.
         * @param body the string content of the body to send, or <tt>null</tt> if there is no body to send (default).
         * @param charset the charset to use for decoding of the string content (default: UTF-8)
         * @return this builder
         */
        @Nonnull
        public Builder body(String body, Charset charset) {
            charset = charset == null ? Charset.forName("UTF-8") : charset;
            this.body = body == null ? null : body.getBytes(charset);
            return this;
        }

        /**
         * Add a query parameter to the request.
         * A query parameter key can be added multiple times - this will not override existing values but will have
         * multiple values.
         * @param key the key of the query parameter
         * @param value the value for the query parameter
         * @return this builder
         */
        @Nonnull
        public Builder queryParam(@Nonnull String key, String value) {
            queryParams.add(Pair.of(key, value));
            return this;
        }

        /**
         * Add a header to the request.
         * A header name can be added multiple times - this will not override existing values but will have
         * multiple values.
         * @param name the name of the header
         * @param value the value for the header
         * @return this builder
         */
        @Nonnull
        public Builder header(@Nonnull String name, String value) {
            headers.add(Pair.of(name, value));
            return this;
        }

        /**
         * Set the content type of the request's body.
         * @param contentType the content type to set
         * @return this builder
         */
        @Nonnull
        public Builder contentType(@Nonnull String contentType) {
            return header("Content-Type", contentType);
        }

        /**
         * Set different socket timeout for this request
         */
        public Builder socketTimeout(Integer socketTimeout) {
            this.socketTimeout = socketTimeout;
            return this;
        }

        /**
         * Build the request.
         * This creates a new {@link RestRequest} instance. Calling this method yields new distinct instances. This way
         * the builder can be used multiple times with slightly different inputs (e.g. same request, different body)
         * @return a new {@link RestRequest} instance
         */
        public RestRequest build() {
            return new RestRequest(method, path, body, queryParams, headers, socketTimeout);
        }
    }
}
