/*
 * Decompiled with CFR 0.152.
 */
package org.jetbrains.jet.lang.resolve.constants;

import com.google.common.base.Function;
import com.google.common.collect.Sets;
import com.intellij.psi.tree.IElementType;
import java.util.HashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.jet.JetNodeTypes;
import org.jetbrains.jet.lang.diagnostics.AbstractDiagnosticFactory;
import org.jetbrains.jet.lang.diagnostics.Diagnostic;
import org.jetbrains.jet.lang.diagnostics.Errors;
import org.jetbrains.jet.lang.psi.JetConstantExpression;
import org.jetbrains.jet.lang.psi.JetElement;
import org.jetbrains.jet.lang.resolve.constants.BooleanValue;
import org.jetbrains.jet.lang.resolve.constants.ByteValue;
import org.jetbrains.jet.lang.resolve.constants.CharValue;
import org.jetbrains.jet.lang.resolve.constants.CompileTimeConstant;
import org.jetbrains.jet.lang.resolve.constants.DoubleValue;
import org.jetbrains.jet.lang.resolve.constants.ErrorValue;
import org.jetbrains.jet.lang.resolve.constants.FloatValue;
import org.jetbrains.jet.lang.resolve.constants.IntValue;
import org.jetbrains.jet.lang.resolve.constants.LongValue;
import org.jetbrains.jet.lang.resolve.constants.NullValue;
import org.jetbrains.jet.lang.resolve.constants.ShortValue;
import org.jetbrains.jet.lang.types.ErrorUtils;
import org.jetbrains.jet.lang.types.JetType;
import org.jetbrains.jet.lang.types.TypeConstructor;
import org.jetbrains.jet.lang.types.TypeUtils;
import org.jetbrains.jet.lang.types.checker.JetTypeChecker;
import org.jetbrains.jet.lang.types.lang.KotlinBuiltIns;

public class CompileTimeConstantResolver {
    private final KotlinBuiltIns builtIns = KotlinBuiltIns.getInstance();

    @Nullable
    public Diagnostic checkConstantExpressionType(@NotNull JetConstantExpression expression, @NotNull JetType expectedType) {
        Diagnostic diagnostic;
        CompileTimeConstant<?> compileTimeConstant = this.getCompileTimeConstant(expression, expectedType);
        HashSet<AbstractDiagnosticFactory> errorsThatDependOnExpectedType = Sets.newHashSet(Errors.CONSTANT_EXPECTED_TYPE_MISMATCH, Errors.NULL_FOR_NONNULL_TYPE);
        if (compileTimeConstant instanceof ErrorValue.ErrorValueWithDiagnostic && errorsThatDependOnExpectedType.contains((diagnostic = ((ErrorValue.ErrorValueWithDiagnostic)compileTimeConstant).getDiagnostic()).getFactory())) {
            return diagnostic;
        }
        return null;
    }

    @NotNull
    public CompileTimeConstant<?> getCompileTimeConstant(@NotNull JetConstantExpression expression, @NotNull JetType expectedType) {
        CompileTimeConstant<?> value;
        IElementType elementType = expression.getNode().getElementType();
        if (elementType == JetNodeTypes.INTEGER_CONSTANT) {
            value = this.getIntegerValue(expression, expectedType);
        } else if (elementType == JetNodeTypes.FLOAT_CONSTANT) {
            value = this.getFloatValue(expression, expectedType);
        } else if (elementType == JetNodeTypes.BOOLEAN_CONSTANT) {
            value = this.getBooleanValue(expression, expectedType);
        } else if (elementType == JetNodeTypes.CHARACTER_CONSTANT) {
            value = this.getCharValue(expression, expectedType);
        } else if (elementType == JetNodeTypes.NULL) {
            value = this.getNullValue(expression, expectedType);
        } else {
            throw new IllegalArgumentException("Unsupported constant: " + expression);
        }
        return value;
    }

    @NotNull
    public CompileTimeConstant<?> getIntegerValue(@NotNull JetConstantExpression expression, @NotNull JetType expectedType) {
        String text = expression.getText();
        return this.getIntegerValue(CompileTimeConstantResolver.parseLongValue(text), expectedType, expression);
    }

    @NotNull
    public CompileTimeConstant<?> getIntegerValue(@Nullable Long value, @NotNull JetType expectedType, @NotNull JetConstantExpression expression) {
        long upperBound;
        long lowerBound;
        Function<Long, CompileTimeConstant<Integer>> create;
        if (value == null) {
            return ErrorValue.create(Errors.INT_LITERAL_OUT_OF_RANGE.on(expression));
        }
        if (CompileTimeConstantResolver.noExpectedTypeOrError(expectedType)) {
            if (Integer.MIN_VALUE <= value && value <= Integer.MAX_VALUE) {
                return new IntValue(value.intValue());
            }
            return new LongValue(value);
        }
        TypeConstructor constructor = expectedType.getConstructor();
        if (constructor == this.builtIns.getInt().getTypeConstructor()) {
            create = IntValue.CREATE;
            lowerBound = Integer.MIN_VALUE;
            upperBound = Integer.MAX_VALUE;
        } else if (constructor == this.builtIns.getLong().getTypeConstructor()) {
            create = LongValue.CREATE;
            lowerBound = Long.MIN_VALUE;
            upperBound = Long.MAX_VALUE;
        } else if (constructor == this.builtIns.getShort().getTypeConstructor()) {
            create = ShortValue.CREATE;
            lowerBound = -32768L;
            upperBound = 32767L;
        } else if (constructor == this.builtIns.getByte().getTypeConstructor()) {
            create = ByteValue.CREATE;
            lowerBound = -128L;
            upperBound = 127L;
        } else {
            JetTypeChecker typeChecker = JetTypeChecker.INSTANCE;
            JetType intType = this.builtIns.getIntType();
            JetType longType = this.builtIns.getLongType();
            if (typeChecker.isSubtypeOf(intType, expectedType)) {
                return this.getIntegerValue(value, intType, expression);
            }
            if (typeChecker.isSubtypeOf(longType, expectedType)) {
                return this.getIntegerValue(value, longType, expression);
            }
            return ErrorValue.create(Errors.CONSTANT_EXPECTED_TYPE_MISMATCH.on(expression, "integer", expectedType));
        }
        if (value != null && lowerBound <= value && value <= upperBound) {
            return create.apply(value);
        }
        return ErrorValue.create(Errors.CONSTANT_EXPECTED_TYPE_MISMATCH.on(expression, "integer", expectedType));
    }

    @Nullable
    public static Long parseLongValue(String text) {
        try {
            long value;
            if (text.startsWith("0x") || text.startsWith("0X")) {
                String hexString = text.substring(2);
                value = Long.parseLong(hexString, 16);
            } else if (text.startsWith("0b") || text.startsWith("0B")) {
                String binString = text.substring(2);
                value = Long.parseLong(binString, 2);
            } else {
                value = Long.parseLong(text);
            }
            return value;
        }
        catch (NumberFormatException e) {
            return null;
        }
    }

    @Nullable
    public static Double parseDoubleValue(String text) {
        try {
            return Double.parseDouble(text);
        }
        catch (NumberFormatException e) {
            return null;
        }
    }

    @NotNull
    public CompileTimeConstant<?> getFloatValue(@NotNull JetConstantExpression expression, @NotNull JetType expectedType) {
        String text = expression.getText();
        try {
            if (CompileTimeConstantResolver.noExpectedTypeOrError(expectedType) || JetTypeChecker.INSTANCE.isSubtypeOf(this.builtIns.getDoubleType(), expectedType)) {
                return new DoubleValue(Double.parseDouble(text));
            }
            if (JetTypeChecker.INSTANCE.isSubtypeOf(this.builtIns.getFloatType(), expectedType)) {
                return new FloatValue(Float.parseFloat(text));
            }
            return ErrorValue.create(Errors.CONSTANT_EXPECTED_TYPE_MISMATCH.on(expression, "floating-point", expectedType));
        }
        catch (NumberFormatException e) {
            return ErrorValue.create(Errors.FLOAT_LITERAL_OUT_OF_RANGE.on(expression));
        }
    }

    @Nullable
    private CompileTimeConstant<?> checkNativeType(JetType expectedType, String title, JetType nativeType, JetConstantExpression expression) {
        if (!CompileTimeConstantResolver.noExpectedTypeOrError(expectedType) && !JetTypeChecker.INSTANCE.isSubtypeOf(nativeType, expectedType)) {
            return ErrorValue.create(Errors.CONSTANT_EXPECTED_TYPE_MISMATCH.on(expression, title, expectedType));
        }
        return null;
    }

    @NotNull
    public CompileTimeConstant<?> getBooleanValue(@NotNull JetConstantExpression expression, @NotNull JetType expectedType) {
        String text = expression.getText();
        CompileTimeConstant<?> error = this.checkNativeType(expectedType, "boolean", this.builtIns.getBooleanType(), expression);
        if (error != null) {
            return error;
        }
        if ("true".equals(text)) {
            return BooleanValue.TRUE;
        }
        if ("false".equals(text)) {
            return BooleanValue.FALSE;
        }
        throw new IllegalStateException("Must not happen. A boolean literal has text: " + text);
    }

    @NotNull
    public CompileTimeConstant<?> getCharValue(@NotNull JetConstantExpression expression, @NotNull JetType expectedType) {
        String text = expression.getText();
        CompileTimeConstant<?> error = this.checkNativeType(expectedType, "character", this.builtIns.getCharType(), expression);
        if (error != null) {
            return error;
        }
        if (text.length() < 2 || text.charAt(0) != '\'' || text.charAt(text.length() - 1) != '\'') {
            return ErrorValue.create(Errors.INCORRECT_CHARACTER_LITERAL.on(expression));
        }
        if ((text = text.substring(1, text.length() - 1)).length() == 0) {
            return ErrorValue.create(Errors.EMPTY_CHARACTER_LITERAL.on(expression));
        }
        if (text.charAt(0) != '\\') {
            if (text.length() == 1) {
                return new CharValue(text.charAt(0));
            }
            return ErrorValue.create(Errors.TOO_MANY_CHARACTERS_IN_CHARACTER_LITERAL.on(expression, expression));
        }
        return CompileTimeConstantResolver.escapedStringToCharValue(text, expression);
    }

    @NotNull
    public static CompileTimeConstant<?> escapedStringToCharValue(@NotNull String text, @NotNull JetElement expression) {
        assert (text.length() > 0 && text.charAt(0) == '\\') : "Only escaped sequences must be passed to this routine: " + text;
        String escape = text.substring(1);
        switch (escape.length()) {
            case 0: {
                return CompileTimeConstantResolver.illegalEscape(expression);
            }
            case 1: {
                Character escaped = CompileTimeConstantResolver.translateEscape(escape.charAt(0));
                if (escaped == null) {
                    return CompileTimeConstantResolver.illegalEscape(expression);
                }
                return new CharValue(escaped.charValue());
            }
            case 5: {
                if (escape.charAt(0) != 'u') break;
                try {
                    Integer intValue = Integer.valueOf(escape.substring(1), 16);
                    return new CharValue((char)intValue.intValue());
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
        }
        return CompileTimeConstantResolver.illegalEscape(expression);
    }

    @NotNull
    private static CompileTimeConstant<?> illegalEscape(@NotNull JetElement expression) {
        return ErrorValue.create(Errors.ILLEGAL_ESCAPE.on(expression, expression));
    }

    @Nullable
    public static Character translateEscape(char c) {
        switch (c) {
            case 't': {
                return Character.valueOf('\t');
            }
            case 'b': {
                return Character.valueOf('\b');
            }
            case 'n': {
                return Character.valueOf('\n');
            }
            case 'r': {
                return Character.valueOf('\r');
            }
            case '\'': {
                return Character.valueOf('\'');
            }
            case '\"': {
                return Character.valueOf('\"');
            }
            case '\\': {
                return Character.valueOf('\\');
            }
            case '$': {
                return Character.valueOf('$');
            }
        }
        return null;
    }

    @NotNull
    public CompileTimeConstant<?> getNullValue(@NotNull JetConstantExpression expression, @NotNull JetType expectedType) {
        if (CompileTimeConstantResolver.noExpectedTypeOrError(expectedType) || expectedType.isNullable()) {
            return NullValue.NULL;
        }
        return ErrorValue.create(Errors.NULL_FOR_NONNULL_TYPE.on(expression, expectedType));
    }

    private static boolean noExpectedTypeOrError(JetType expectedType) {
        return TypeUtils.noExpectedType(expectedType) || ErrorUtils.isErrorType(expectedType);
    }
}

