/*
 * The MIT License
 *
 * Copyright (c) 2004-2009, Sun Microsystems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.jvnet.hudson.test;

import hudson.PluginManager.FailedPlugin;
import hudson.PluginWrapper;
import hudson.cli.CLICommand;
import java.io.File;
import java.util.Map;
import jenkins.model.Jenkins;
import junit.framework.TestCase;
import junit.framework.TestSuite;

/**
 * Called by the code generated by maven-hpi-plugin to build tests for plugins.
 *
 * @author Kohsuke Kawaguchi
 */
public class PluginAutomaticTestBuilder {

    /** {@code InjectedTest} mistakenly calls {@link #build} as an instance method, and uses this constructor. */
    public PluginAutomaticTestBuilder() {}

    /**
     * @param params
     *      Various information about the plugin that maven-hpi-plugin adds.
     *      As of 1.119, this includes the followings:
     *
     *      basedir (String) : directory that contains pom.xml
     *      artifactId (String) : artifact ID from the pom.xml
     *      packaging (String) : packaging of from the pom.xml (used to determine if the pom is a plugin or not)
     *      outputDirectory (String) : target/classes dir where class files and resources can be found
     *      testOutputDirectory (String) : target/test-classes.
     *      requirePI (String) : either {@code true} to verify that all the jelly scripts have the Jelly XSS PI in
     *      them or {@code false} to ignore any missing ones.
     */
    public static TestSuite build(Map<String,?> params) throws Exception {
        TestSuite master = new TestSuite();
        if (params.containsKey("outputDirectory")) { // shouldn't happen, but be defensive
            File outputDirectory = new File((String)params.get("outputDirectory"));
            TestSuite inJenkins = JellyTestSuiteBuilder.build(outputDirectory,toBoolean(params.get("requirePI")));
            inJenkins.addTest(new OtherTests("testCliSanity", params));
            String packaging = (String) params.get("packaging");
            if (packaging == null || packaging.trim().isEmpty() || "hpi".equals(packaging)) {
                inJenkins.addTest(new OtherTests("testPluginActive", params));
            }
            master.addTest(inJenkins);
            master.addTest(new PropertiesTestSuite(outputDirectory));
        }
        return master;
    }

    private static boolean toBoolean(Object requirePI) {
        if (requirePI == null) {
            return false;
        }
        if (requirePI instanceof Boolean) {
            return (Boolean) requirePI;
        }
        return Boolean.parseBoolean(requirePI.toString());
    }

    public static class OtherTests extends TestCase {

        private final Map<String,?> params;

        public OtherTests(String name, Map<String,?> params) {
            super(name);
            this.params = params;
        }

        public void testCliSanity() {
            CLICommand.clone("help");
        }

        public void testPluginActive() {
            String plugin = (String) params.get("artifactId");
            if (plugin != null) {
                // did any plugin fail to start?
                for (FailedPlugin fp : Jenkins.get().getPluginManager().getFailedPlugins()) {
                    throw new AssertionError("While testing " + plugin + ", " + fp.name + " failed to start", fp.cause);
                }

                PluginWrapper pw = Jenkins.get().getPluginManager().getPlugin(plugin);

                assertNotNull(plugin+" failed to start", pw);
                assertTrue(plugin + " was not active", pw.isActive());
            }
        }

    }
}
