/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server.integration.spec;

import java.util.ArrayList;
import java.util.List;

import javax.management.ObjectName;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.api.model.Node;
import org.jboss.bpm.api.model.ObjectNameFactory;
import org.jboss.bpm.api.model.ProcessInstance;
import org.jboss.bpm.api.model.ProcessDefinition;
import org.jboss.bpm.api.runtime.Token;
import org.jboss.bpm.api.service.TokenService;
import org.jboss.bpm.api.service.ProcessEngine;
import org.jboss.bpm.console.client.model.TaskRef;
import org.jboss.bpm.console.server.integration.TaskManagement;
import org.jboss.bpm.incubator.model.SingleOutFlowSupport;
import org.jboss.bpm.incubator.service.TaskInstanceService;
import org.jboss.bpm.incubator.task.TaskInstance;

/**
 * An implementation that delegates to a BPM Spec.
 * 
 * https://jira.jboss.org/jira/browse/JBPM-1892
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 28-Nov-2008
 */
public class TaskManagementImpl implements TaskManagement
{
  private static final Log log = LogFactory.getLog(TaskManagementImpl.class);

  public TaskRef getTaskById(long taskId)
  {
    log.debug("getTaskById: " + taskId);
    
    ObjectName taskKey = getTaskKey(taskId);
    TaskInstanceService taskService = getTaskService();
    TaskInstance task = taskService.getTask(taskKey);
    
    return adaptTask(task);
  }

  public List<TaskRef> getTasksByActor(String actor)
  {
    log.debug("getTasksByActor: " + actor);
    
    List<TaskRef> taskRefs = new ArrayList<TaskRef>();
    TaskInstanceService taskService = getTaskService();
    for (TaskInstance task : taskService.getTasksByActor(actor))
    {
      log.debug(task);
      taskRefs.add(adaptTask(task));
    }
    return taskRefs;
  }

  public void closeTask(long taskId, String signalName)
  {
    log.debug("closeTask: " + taskId + "," + signalName);
    
    ObjectName taskKey = getTaskKey(taskId);
    TaskInstanceService taskService = getTaskService();
    taskService.closeTask(taskKey, signalName);
  }

  public void reassignTask(long taskId, String actor)
  {
    log.debug("reassignTask: " + taskId + "," + actor);
    
    ObjectName taskKey = getTaskKey(taskId);
    TaskInstanceService taskService = getTaskService();
    taskService.reassignTask(taskKey, actor);
  }

  private TaskRef adaptTask(TaskInstance task)
  {
    Long taskId = adaptKey(task.getKey());
    
    ObjectName tokenKey = task.getCorrelationKey();
    Token token = getExecutionService().getToken(tokenKey);
    Long tokenId = adaptKey(tokenKey);
    
    ProcessInstance proc = token.getProcess();
    Long procId = adaptKey(proc.getKey());
    
    ProcessDefinition procDef = proc.getProcessDefinition();
    Long procDefId = adaptKey(procDef.getKey());
    
    TaskRef taskRef = new TaskRef(taskId, tokenId, procId, procDefId, task.getName(), task.getActor(), task.isBlocking(), task.isSignalling());

    for (String pa : task.getPooledActors())
    {
      taskRef.addPooledActor(pa);
    }

    Node currentNode = token.getNode();
    if (currentNode instanceof SingleOutFlowSupport)
    {
      SingleOutFlowSupport sofs = (SingleOutFlowSupport)currentNode;
      String targetName = sofs.getOutFlow().getName();
      if (targetName == null)
        targetName = sofs.getOutFlow().getTargetRef();
      
      taskRef.getTransitionNames().add(targetName);
    }

    return taskRef;
  }

  private Long adaptKey(ObjectName key)
  {
    String id = key.getKeyProperty("id");
    if (id == null)
      throw new IllegalStateException("Cannot obtain id property from: " + key);

    return new Long(id);
  }
  
  private ObjectName getTaskKey(long taskId)
  {
    return ObjectNameFactory.create("Task:id=" + taskId);
  }

  private TaskInstanceService getTaskService()
  {
    ProcessEngine engine = ProcessEngineFactory.getProcessEngine();
    TaskInstanceService taskService = engine.getService(TaskInstanceService.class);
    return taskService;
  }
  
  private TokenService getExecutionService()
  {
    ProcessEngine engine = ProcessEngineFactory.getProcessEngine();
    TokenService exService = engine.getService(TokenService.class);
    return exService;
  }
}
