/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server.integration.spec;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.management.ObjectName;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.api.model.Node;
import org.jboss.bpm.api.model.ProcessDefinition;
import org.jboss.bpm.api.model.ProcessInstance;
import org.jboss.bpm.api.model.ProcessInstance.ProcessStatus;
import org.jboss.bpm.api.runtime.Token;
import org.jboss.bpm.api.runtime.Token.TokenStatus;
import org.jboss.bpm.api.service.ProcessDefinitionService;
import org.jboss.bpm.api.service.ProcessEngine;
import org.jboss.bpm.api.service.ProcessInstanceService;
import org.jboss.bpm.console.client.model.ProcessDefinitionRef;
import org.jboss.bpm.console.client.model.ProcessInstanceRef;
import org.jboss.bpm.console.client.model.ProcessInstanceRef.STATE;
import org.jboss.bpm.console.client.model.jbpm3.TokenReference;
import org.jboss.bpm.console.server.integration.ProcessManagement;
import org.jboss.bpm.incubator.model.Gateway;
import org.jboss.bpm.incubator.model.SequenceFlow;
import org.jboss.bpm.incubator.model.SingleOutFlowSupport;

/**
 * An implementation that delegates to a BPM Spec.
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 25-Nov-2008
 */
public class ProcessManagementImpl implements ProcessManagement
{
  private static final Log log = LogFactory.getLog(ProcessManagementImpl.class);

  public List<ProcessDefinitionRef> getProcessDefinitions()
  {
    log.debug("getProcessDefinitions");

    List<ProcessDefinitionRef> results = new ArrayList<ProcessDefinitionRef>();
    ProcessDefinitionService pdService = getProcessDefinitionService();
    Iterator<ObjectName> itKey = pdService.getProcessDefinitions().iterator();
    while (itKey.hasNext())
    {
      ObjectName procDefKey = itKey.next();
      ProcessDefinition procDef = pdService.getProcessDefinition(procDefKey);

      log.debug(procDef);

      ProcessDefinitionRef pdRef = adaptProcessDefinition(procDef);
      results.add(pdRef);
    }
    return results;
  }

  public ProcessDefinitionRef getProcessDefinition(long procDefId)
  {
    ProcessDefinition procDef = getProcessDefinitionInternal(procDefId);
    log.debug("getProcessDefinition: " + procDef);

    return procDef != null ? new ProcessDefinitionRef(procDefId, procDef.getName(), procDef.getVersion()) : null;
  }

  public List<ProcessDefinitionRef> removeProcessDefinition(long procDefId)
  {
    ProcessDefinition procDef = getProcessDefinitionInternal(procDefId);
    log.debug("removeDefinition: " + procDef);

    if (procDef != null)
    {
      ProcessDefinitionService pdService = getProcessDefinitionService();
      pdService.unregisterProcessDefinition(procDef.getKey());
    }
    return getProcessDefinitions();
  }

  public ProcessInstanceRef newInstance(long procDefId)
  {
    ProcessDefinition procDef = getProcessDefinitionInternal(procDefId);
    log.debug("newInstance: " + procDef);

    if (procDef == null)
      throw new IllegalStateException("Cannot obtain process definition: " + procDefId);

    ProcessInstance proc = procDef.newInstance();
    proc.startProcess();

    log.debug("Started: " + proc);

    ProcessInstanceRef procRef = adaptProcess(proc);
    return procRef;
  }

  public List<ProcessInstanceRef> getProcessInstances(long procDefId)
  {
    List<ProcessInstanceRef> results = new ArrayList<ProcessInstanceRef>();
    ProcessDefinition procDef = getProcessDefinitionInternal(procDefId);
    if (procDef == null)
      throw new IllegalStateException("Cannot obtain process definition: " + procDefId);

    log.debug("getProcessInstances: " + procDef);

    ProcessInstanceService procService = getProcessInstanceService();
    Iterator<ObjectName> itProc = procService.getInstance(procDef.getKey(), null).iterator();
    while (itProc.hasNext())
    {
      ObjectName procID = itProc.next();
      ProcessInstance proc = procService.getInstance(procID);
      
      log.debug(proc);

      ProcessStatus status = proc.getProcessStatus();
      if (status == ProcessStatus.Active || status == ProcessStatus.Suspended)
      {
        results.add(adaptProcess(proc));
      }
    }

    return results;
  }

  public ProcessInstanceRef getProcessInstance(long procID)
  {
    ProcessInstance proc = getProcessById(procID);
    log.debug("getProcessInstance: " + proc);

    ProcessInstanceRef procRef = adaptProcess(proc);
    return procRef;
  }

  public void setProcessState(long procID, STATE nextState)
  {
    ProcessInstance proc = getProcessById(procID);
    log.debug("setProcessState " + nextState + " on: " + proc);

    ProcessInstanceRef procRef = adaptProcess(proc);
    procRef.setState(nextState);

    switch (procRef.getState())
    {
      case ENDED:
        proc.cancel();
        break;
      case SUSPENDED:
        proc.suspend();
        break;
      case RUNNING:
        proc.resume();
    }

  }

  public void signalToken(long tokenId, String name)
  {
    Token token = null;
    ProcessInstanceService procService = getProcessInstanceService();
    Iterator<ObjectName> itProc = procService.getInstance().iterator();
    while (token == null && itProc.hasNext())
    {
      ObjectName procID = itProc.next();
      ProcessInstance proc = procService.getInstance(procID);
      for (Token aux : proc.getTokens())
      {
        Long auxId = adaptKey(aux.getKey());
        if (auxId == tokenId)
        {
          token = aux;
          break;
        }
      }
    }

    if (token == null)
      throw new IllegalStateException("Cannot obtain token: " + tokenId);

    log.debug("signalToken: " + token);

    // Signal the token
    token.signal(name);
  }

  // Private

  private ProcessDefinitionService getProcessDefinitionService()
  {
    ProcessEngine engine = ProcessEngineFactory.getProcessEngine();
    ProcessDefinitionService pdService = engine.getService(ProcessDefinitionService.class);
    return pdService;
  }

  private ProcessDefinition getProcessDefinitionInternal(long procDefID)
  {
    ProcessDefinition procDef = null;
    ProcessDefinitionService pdService = getProcessDefinitionService();
    Iterator<ObjectName> it = pdService.getProcessDefinitions().iterator();
    while (it.hasNext())
    {
      ObjectName auxKey = it.next();
      if (procDefID == adaptKey(auxKey))
      {
        procDef = pdService.getProcessDefinition(auxKey);
        break;
      }
    }
    return procDef;
  }

  private ProcessDefinitionRef adaptProcessDefinition(ProcessDefinition procDef)
  {
    ObjectName procDefKey = procDef.getKey();
    Long procDefID = adaptKey(procDefKey);
    return new ProcessDefinitionRef(procDefID, procDef.getName(), procDef.getVersion());
  }

  private ProcessInstanceService getProcessInstanceService()
  {
    ProcessEngine engine = ProcessEngineFactory.getProcessEngine();
    ProcessInstanceService pService = engine.getService(ProcessInstanceService.class);
    return pService;
  }

  private ProcessInstance getProcessById(long procID)
  {
    ProcessInstance proc = null;
    ProcessInstanceService procService = getProcessInstanceService();
    Iterator<ObjectName> it = procService.getInstance().iterator();
    while (it.hasNext())
    {
      ObjectName auxKey = it.next();
      if (procID == adaptKey(auxKey))
      {
        proc = procService.getInstance(auxKey);
        break;
      }
    }
    return proc;
  }

  private ProcessInstanceRef adaptProcess(ProcessInstance proc)
  {
    Long procDefID = adaptKey(proc.getProcessDefinition().getKey());
    Long procID = adaptKey(proc.getKey());

    boolean suspended = proc.getProcessStatus() == ProcessStatus.Suspended;
    ProcessInstanceRef procRef = new ProcessInstanceRef(procID, procDefID, proc.getStartDate(), proc.getEndDate(), suspended);

    Token rootToken = proc.getRootToken();
    procRef.setRootToken(adaptToken(rootToken));

    return procRef;
  }

  private TokenReference adaptToken(Token token)
  {
    Long tokenId = adaptKey(token.getKey());
    token.getKey();

    Node currNode = token.getNode();
    TokenReference tokenRef = new TokenReference(tokenId, null, currNode.getName());

    TokenStatus tokenStatus = token.getTokenStatus();
    tokenRef.setCanBeSignaled(tokenStatus == TokenStatus.Suspended);

    for (Token child : token.getChildTokens())
      tokenRef.getChildren().add(adaptToken(child));

    if (currNode instanceof SingleOutFlowSupport)
    {
      SingleOutFlowSupport flowSupport = (SingleOutFlowSupport)currNode;
      String targetRef = flowSupport.getOutFlow().getTargetRef();
      tokenRef.getAvailableSignals().add(targetRef);
    }
    if (currNode instanceof Gateway)
    {
      Gateway gateway = (Gateway)currNode;
      for (SequenceFlow gate : gateway.getGates())
      {
        String sigName = gate.getName();
        if (sigName == null)
          sigName = gate.getTargetRef();

        tokenRef.getAvailableSignals().add(sigName);
      }
    }

    return tokenRef;
  }

  private Long adaptKey(ObjectName key)
  {
    String id = key.getKeyProperty("id");
    if (id == null)
      throw new IllegalStateException("Cannot obtain id property from: " + key);

    return new Long(id);
  }
}
