/*
 * Decompiled with CFR 0.152.
 */
package org.janusgraph.graphdb.log;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalCause;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.SetMultimap;
import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.janusgraph.core.JanusGraphElement;
import org.janusgraph.core.JanusGraphException;
import org.janusgraph.core.JanusGraphTransaction;
import org.janusgraph.core.RelationType;
import org.janusgraph.core.log.TransactionRecovery;
import org.janusgraph.diskstorage.BackendTransaction;
import org.janusgraph.diskstorage.ReadBuffer;
import org.janusgraph.diskstorage.StaticBuffer;
import org.janusgraph.diskstorage.indexing.IndexTransaction;
import org.janusgraph.diskstorage.log.Log;
import org.janusgraph.diskstorage.log.Message;
import org.janusgraph.diskstorage.log.MessageReader;
import org.janusgraph.diskstorage.log.ReadMarker;
import org.janusgraph.diskstorage.log.kcvs.KCVSLog;
import org.janusgraph.diskstorage.util.BackendOperation;
import org.janusgraph.diskstorage.util.time.TimestampProvider;
import org.janusgraph.graphdb.configuration.GraphDatabaseConfiguration;
import org.janusgraph.graphdb.database.StandardJanusGraph;
import org.janusgraph.graphdb.database.log.LogTxMeta;
import org.janusgraph.graphdb.database.log.LogTxStatus;
import org.janusgraph.graphdb.database.log.TransactionLogHeader;
import org.janusgraph.graphdb.database.serialize.Serializer;
import org.janusgraph.graphdb.internal.ElementCategory;
import org.janusgraph.graphdb.internal.InternalRelationType;
import org.janusgraph.graphdb.log.ModificationDeserializer;
import org.janusgraph.graphdb.log.StandardTransactionId;
import org.janusgraph.graphdb.relations.RelationIdentifier;
import org.janusgraph.graphdb.transaction.StandardJanusGraphTx;
import org.janusgraph.graphdb.types.IndexType;
import org.janusgraph.graphdb.types.MixedIndexType;
import org.janusgraph.graphdb.types.SchemaSource;
import org.janusgraph.graphdb.types.indextype.IndexTypeWrapper;
import org.janusgraph.graphdb.types.vertices.JanusGraphSchemaVertex;
import org.janusgraph.util.system.BackgroundThread;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StandardTransactionLogProcessor
implements TransactionRecovery {
    private static final Logger logger = LoggerFactory.getLogger(StandardTransactionLogProcessor.class);
    private static final Duration CLEAN_SLEEP_TIME = Duration.ofSeconds(5L);
    private static final Duration MIN_TX_LENGTH = Duration.ofSeconds(5L);
    private final StandardJanusGraph graph;
    private final Serializer serializer;
    private final TimestampProvider times;
    private final Duration persistenceTime;
    private final Duration readTime = Duration.ofSeconds(1L);
    private final AtomicLong txCounter = new AtomicLong(0L);
    private final BackgroundCleaner cleaner;
    private final boolean verboseLogging;
    private final AtomicLong successTxCounter = new AtomicLong(0L);
    private final AtomicLong failureTxCounter = new AtomicLong(0L);
    private final Cache<StandardTransactionId, TxEntry> txCache;
    private static final Predicate<IndexType> MIXED_INDEX_FILTER = IndexType::isMixedIndex;

    public StandardTransactionLogProcessor(StandardJanusGraph graph, Instant startTime) {
        Preconditions.checkArgument((graph != null && graph.isOpen() ? 1 : 0) != 0);
        Preconditions.checkArgument((startTime != null ? 1 : 0) != 0);
        Preconditions.checkArgument((boolean)graph.getConfiguration().hasLogTransactions(), (Object)"Transaction logging must be enabled for recovery to work");
        Duration maxTxLength = graph.getConfiguration().getMaxCommitTime();
        if (maxTxLength.compareTo(MIN_TX_LENGTH) < 0) {
            maxTxLength = MIN_TX_LENGTH;
        }
        Preconditions.checkArgument((maxTxLength != null && !maxTxLength.isZero() ? 1 : 0) != 0, (Object)"Max transaction time cannot be 0");
        this.graph = graph;
        this.serializer = graph.getDataSerializer();
        this.times = graph.getConfiguration().getTimestampProvider();
        KCVSLog txLog = graph.getBackend().getSystemTxLog();
        this.persistenceTime = graph.getConfiguration().getMaxWriteTime();
        this.verboseLogging = graph.getConfiguration().getConfiguration().get(GraphDatabaseConfiguration.VERBOSE_TX_RECOVERY, new String[0]);
        this.txCache = CacheBuilder.newBuilder().concurrencyLevel(2).initialCapacity(100).expireAfterWrite(maxTxLength.toNanos(), TimeUnit.NANOSECONDS).removalListener(notification -> {
            RemovalCause cause = notification.getCause();
            Preconditions.checkArgument((cause == RemovalCause.EXPIRED ? 1 : 0) != 0, (String)"Unexpected removal cause [%s] for transaction [%s]", (Object[])new Object[]{cause, notification.getKey()});
            TxEntry entry = (TxEntry)notification.getValue();
            if (entry.status == LogTxStatus.SECONDARY_FAILURE || entry.status == LogTxStatus.PRIMARY_SUCCESS) {
                this.failureTxCounter.incrementAndGet();
                this.fixSecondaryFailure((StandardTransactionId)notification.getKey(), entry);
            } else {
                this.successTxCounter.incrementAndGet();
            }
        }).build();
        ReadMarker start = ReadMarker.fromTime(startTime);
        txLog.registerReader(start, new TxLogMessageReader());
        this.cleaner = new BackgroundCleaner();
        this.cleaner.start();
    }

    public long[] getStatistics() {
        return new long[]{this.successTxCounter.get(), this.failureTxCounter.get()};
    }

    @Override
    public synchronized void shutdown() throws JanusGraphException {
        this.cleaner.close(CLEAN_SLEEP_TIME);
    }

    private void logRecoveryMsg(String message, Object ... args) {
        if (logger.isInfoEnabled() || this.verboseLogging) {
            String msg = String.format(message, args);
            logger.info(msg);
            if (this.verboseLogging) {
                System.out.println(msg);
            }
        }
    }

    private void fixSecondaryFailure(StandardTransactionId txId, TxEntry entry) {
        Predicate isFailedIndex;
        this.logRecoveryMsg("Attempting to repair partially failed transaction [%s]", txId);
        if (entry.entry == null) {
            this.logRecoveryMsg("Trying to repair expired or unpersisted transaction [%s] (Ignore in startup)", txId);
            return;
        }
        boolean userLogFailure = true;
        boolean secIndexFailure = true;
        TransactionLogHeader.Entry commitEntry = entry.entry;
        TransactionLogHeader.SecondaryFailures secFail = entry.failures;
        if (secFail != null) {
            userLogFailure = secFail.userLogFailure;
            secIndexFailure = !secFail.failedIndexes.isEmpty();
            isFailedIndex = secFail.failedIndexes::contains;
        } else {
            isFailedIndex = Predicates.alwaysTrue();
        }
        if (secIndexFailure) {
            this.restoreExternalIndexes((Predicate<String>)isFailedIndex, commitEntry);
        }
        String logTxIdentifier = (String)commitEntry.getMetadata().get((Object)LogTxMeta.LOG_ID);
        if (userLogFailure && logTxIdentifier != null) {
            TransactionLogHeader txHeader = new TransactionLogHeader(this.txCounter.incrementAndGet(), this.times.getTime(), this.times);
            StaticBuffer userLogContent = txHeader.serializeUserLog(this.serializer, commitEntry, txId);
            BackendOperation.execute(() -> {
                Log userLog = this.graph.getBackend().getUserLog(logTxIdentifier);
                Future<Message> env = userLog.add(userLogContent);
                if (env.isDone()) {
                    env.get();
                }
                return true;
            }, this.persistenceTime);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void restoreExternalIndexes(Predicate<String> isFailedIndex, TransactionLogHeader.Entry entry) {
        HashMultimap indexRestores = HashMultimap.create();
        BackendOperation.execute(() -> this.lambda$restoreExternalIndexes$4(entry, isFailedIndex, (SetMultimap)indexRestores), this.readTime);
        for (String indexName : indexRestores.keySet()) {
            StandardJanusGraphTx tx = (StandardJanusGraphTx)this.graph.newTransaction();
            try {
                BackendTransaction btx = tx.getTxHandle();
                IndexTransaction indexTx = btx.getIndexTransaction(indexName);
                BackendOperation.execute(new Callable<Boolean>((SetMultimap)indexRestores, indexName, tx, indexTx){
                    final /* synthetic */ SetMultimap val$indexRestores;
                    final /* synthetic */ String val$indexName;
                    final /* synthetic */ StandardJanusGraphTx val$tx;
                    final /* synthetic */ IndexTransaction val$indexTx;
                    {
                        this.val$indexRestores = setMultimap;
                        this.val$indexName = string;
                        this.val$tx = standardJanusGraphTx;
                        this.val$indexTx = indexTransaction;
                    }

                    @Override
                    public Boolean call() throws Exception {
                        HashMap restoredDocs = Maps.newHashMap();
                        this.val$indexRestores.get((Object)this.val$indexName).forEach(restore -> {
                            JanusGraphSchemaVertex indexV = (JanusGraphSchemaVertex)this.val$tx.getVertex(((IndexRestore)restore).indexId);
                            MixedIndexType index = (MixedIndexType)indexV.asIndexType();
                            JanusGraphElement element = restore.retrieve(this.val$tx);
                            if (element != null) {
                                StandardTransactionLogProcessor.this.graph.getIndexSerializer().reindexElement(element, index, restoredDocs);
                            } else {
                                StandardTransactionLogProcessor.this.graph.getIndexSerializer().removeElement(((IndexRestore)restore).elementId, index, restoredDocs);
                            }
                        });
                        this.val$indexTx.restore(restoredDocs);
                        this.val$indexTx.commit();
                        return true;
                    }

                    public String toString() {
                        return "IndexMutation";
                    }
                }, this.persistenceTime);
            }
            finally {
                if (!tx.isOpen()) continue;
                tx.rollback();
            }
        }
    }

    private static long getIndexId(IndexType index) {
        SchemaSource base = ((IndexTypeWrapper)index).getSchemaBase();
        assert (base instanceof JanusGraphSchemaVertex);
        return base.longId();
    }

    private static Iterable<MixedIndexType> getMixedIndexes(RelationType type) {
        if (!type.isPropertyKey()) {
            return Collections.emptyList();
        }
        return Iterables.filter((Iterable)Iterables.filter(((InternalRelationType)type).getKeyIndexes(), MIXED_INDEX_FILTER), MixedIndexType.class);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private /* synthetic */ Boolean lambda$restoreExternalIndexes$4(TransactionLogHeader.Entry entry, Predicate isFailedIndex, SetMultimap indexRestores) throws Exception {
        StandardJanusGraphTx tx = (StandardJanusGraphTx)this.graph.newTransaction();
        try {
            entry.getContentAsModifications(this.serializer).stream().map(m -> ModificationDeserializer.parseRelation(m, tx)).forEach(rel -> {
                for (MixedIndexType mixedIndexType : StandardTransactionLogProcessor.getMixedIndexes(rel.getType())) {
                    if (mixedIndexType.getElement() != ElementCategory.VERTEX || !isFailedIndex.apply((Object)mixedIndexType.getBackingIndexName())) continue;
                    assert (rel.isProperty());
                    indexRestores.put((Object)mixedIndexType.getBackingIndexName(), (Object)new IndexRestore(rel.getVertex(0).longId(), ElementCategory.VERTEX, StandardTransactionLogProcessor.getIndexId(mixedIndexType)));
                }
                for (RelationType relationType : rel.getPropertyKeysDirect()) {
                    for (MixedIndexType index : StandardTransactionLogProcessor.getMixedIndexes(relationType)) {
                        if (!index.getElement().isInstance((JanusGraphElement)rel) || !isFailedIndex.apply((Object)index.getBackingIndexName())) continue;
                        assert (rel.id() instanceof RelationIdentifier);
                        indexRestores.put((Object)index.getBackingIndexName(), (Object)new IndexRestore(rel.id(), ElementCategory.getByClazz(rel.getClass()), StandardTransactionLogProcessor.getIndexId(index)));
                    }
                }
            });
        }
        finally {
            if (tx.isOpen()) {
                tx.rollback();
            }
        }
        return true;
    }

    private class BackgroundCleaner
    extends BackgroundThread {
        private Instant lastInvocation;

        public BackgroundCleaner() {
            super("TxLogProcessorCleanup", false);
            this.lastInvocation = null;
        }

        @Override
        protected void waitCondition() throws InterruptedException {
            if (this.lastInvocation != null) {
                StandardTransactionLogProcessor.this.times.sleepPast(this.lastInvocation.plus(CLEAN_SLEEP_TIME));
            }
        }

        @Override
        protected void action() {
            this.lastInvocation = StandardTransactionLogProcessor.this.times.getTime();
            StandardTransactionLogProcessor.this.txCache.cleanUp();
        }

        @Override
        protected void cleanup() {
            StandardTransactionLogProcessor.this.txCache.cleanUp();
        }
    }

    private class TxEntry {
        LogTxStatus status;
        TransactionLogHeader.Entry entry;
        TransactionLogHeader.SecondaryFailures failures;

        private TxEntry() {
        }

        synchronized void update(TransactionLogHeader.Entry e) {
            switch (e.getStatus()) {
                case PRECOMMIT: {
                    this.entry = e;
                    if (this.status != null) break;
                    this.status = LogTxStatus.PRECOMMIT;
                    break;
                }
                case PRIMARY_SUCCESS: {
                    if (this.status != null && this.status != LogTxStatus.PRECOMMIT) break;
                    this.status = LogTxStatus.PRIMARY_SUCCESS;
                    break;
                }
                case COMPLETE_SUCCESS: {
                    if (this.status != null && this.status != LogTxStatus.PRECOMMIT) break;
                    this.status = LogTxStatus.COMPLETE_SUCCESS;
                    break;
                }
                case SECONDARY_SUCCESS: {
                    this.status = LogTxStatus.SECONDARY_SUCCESS;
                    break;
                }
                case SECONDARY_FAILURE: {
                    this.status = LogTxStatus.SECONDARY_FAILURE;
                    this.failures = e.getContentAsSecondaryFailures(StandardTransactionLogProcessor.this.serializer);
                    break;
                }
                default: {
                    throw new AssertionError((Object)("Unexpected status: " + (Object)((Object)e.getStatus())));
                }
            }
        }
    }

    private class TxLogMessageReader
    implements MessageReader {
        private final Callable<TxEntry> entryFactory = () -> new TxEntry();

        private TxLogMessageReader() {
        }

        @Override
        public void read(Message message) {
            TxEntry entry;
            ReadBuffer content = message.getContent().asReadBuffer();
            String senderId = message.getSenderId();
            TransactionLogHeader.Entry txentry = TransactionLogHeader.parse(content, StandardTransactionLogProcessor.this.serializer, StandardTransactionLogProcessor.this.times);
            TransactionLogHeader txheader = txentry.getHeader();
            StandardTransactionId transactionId = new StandardTransactionId(senderId, txheader.getId(), txheader.getTimestamp());
            try {
                entry = (TxEntry)StandardTransactionLogProcessor.this.txCache.get((Object)transactionId, this.entryFactory);
            }
            catch (ExecutionException e) {
                throw new AssertionError("Unexpected exception", e);
            }
            entry.update(txentry);
        }

        @Override
        public void updateState() {
        }
    }

    private static class IndexRestore {
        private final Object elementId;
        private final long indexId;
        private final ElementCategory elementCategory;

        private IndexRestore(Object elementId, ElementCategory category, long indexId) {
            this.elementId = elementId;
            this.indexId = indexId;
            this.elementCategory = category;
        }

        public JanusGraphElement retrieve(JanusGraphTransaction tx) {
            return this.elementCategory.retrieve(this.elementId, tx);
        }

        public int hashCode() {
            return Objects.hash(this.elementId, this.indexId);
        }

        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (other == null || !this.getClass().isInstance(other)) {
                return false;
            }
            IndexRestore r = (IndexRestore)other;
            return r.elementId.equals(this.elementId) && this.indexId == r.indexId;
        }
    }
}

