/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.kotlin.dsl.plugins.dsl

import org.gradle.api.HasImplicitReceiver
import org.gradle.api.SupportsKotlinAssignmentOverloading
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.internal.deprecation.DeprecationLogger
import org.gradle.internal.logging.slf4j.ContextAwareTaskLogger
import org.gradle.kotlin.dsl.*
import org.gradle.kotlin.dsl.assignment.internal.KotlinDslAssignment
import org.gradle.kotlin.dsl.provider.KotlinDslPluginSupport
import org.jetbrains.kotlin.gradle.dsl.JvmTarget
import org.jetbrains.kotlin.gradle.dsl.KotlinVersion
import org.jetbrains.kotlin.gradle.tasks.KotlinCompile
import org.jetbrains.kotlin.assignment.plugin.gradle.AssignmentSubplugin
import org.jetbrains.kotlin.assignment.plugin.gradle.AssignmentExtension
import org.jetbrains.kotlin.samWithReceiver.gradle.SamWithReceiverExtension
import org.jetbrains.kotlin.samWithReceiver.gradle.SamWithReceiverGradleSubplugin


/**
 * Configures the Kotlin compiler to recognise Gradle functional interface
 * annotated with [HasImplicitReceiver].
 */
abstract class KotlinDslCompilerPlugins : Plugin<Project> {

    override fun apply(project: Project): Unit = project.run {

        plugins.apply(SamWithReceiverGradleSubplugin::class.java)
        extensions.configure(SamWithReceiverExtension::class.java) { samWithReceiver ->
            samWithReceiver.annotation(HasImplicitReceiver::class.qualifiedName!!)
        }

        if (KotlinDslAssignment.isAssignmentOverloadEnabled()) {
            plugins.apply(AssignmentSubplugin::class.java)
            extensions.configure(AssignmentExtension::class.java) { assignment ->
                assignment.annotation(SupportsKotlinAssignmentOverloading::class.qualifiedName!!)
            }
        }

        afterEvaluate {
            kotlinDslPluginOptions {
                tasks.withType<KotlinCompile>().configureEach {
                    it.compilerOptions {
                        DeprecationLogger.whileDisabled {
                            @Suppress("DEPRECATION")
                            if (this@kotlinDslPluginOptions.jvmTarget.isPresent) {
                                jvmTarget.set(JvmTarget.fromTarget(this@kotlinDslPluginOptions.jvmTarget.get()))
                            }
                        }
                        apiVersion.set(KotlinVersion.KOTLIN_1_8)
                        languageVersion.set(KotlinVersion.KOTLIN_1_8)
                        freeCompilerArgs.addAll(KotlinDslPluginSupport.kotlinCompilerArgs)
                    }
                    it.setWarningRewriter(ExperimentalCompilerWarningSilencer(listOf(
                        "-XXLanguage:+DisableCompatibilityModeForNewInference",
                        "-XXLanguage:-TypeEnhancementImprovementsInStrictMode",
                        "Assign Kotlin compiler plugin is an experimental feature",
                        // Kotlin 1.8.0 has wrong warning message for assign plugin, remove once we update Kotlin to 1.8.20.
                        // https://github.com/JetBrains/kotlin/commit/0eb34983cb38064684cfc76dacb8d4460fcc6573
                        "Lombok Kotlin compiler plugin is an experimental feature",
                    )))
                }
            }
        }
    }

    private
    fun KotlinCompile.setWarningRewriter(rewriter: ContextAwareTaskLogger.MessageRewriter) {
        (logger as ContextAwareTaskLogger).setMessageRewriter(rewriter)
    }
}
