/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.metadata;

import com.oracle.svm.configure.ConfigurationTypeDescriptor;
import com.oracle.svm.configure.JsonFileWriter;
import com.oracle.svm.configure.NamedConfigurationTypeDescriptor;
import com.oracle.svm.configure.ProxyConfigurationTypeDescriptor;
import com.oracle.svm.configure.UnresolvedConfigurationCondition;
import com.oracle.svm.configure.config.ConfigurationFileCollection;
import com.oracle.svm.configure.config.ConfigurationMemberInfo;
import com.oracle.svm.configure.config.ConfigurationSet;
import com.oracle.svm.configure.config.ConfigurationType;
import com.oracle.svm.core.AlwaysInline;
import com.oracle.svm.core.jdk.RuntimeSupport;
import com.oracle.svm.core.log.Log;
import com.oracle.svm.core.metadata.TraceOptions;
import com.oracle.svm.core.option.HostedOptionKey;
import com.oracle.svm.core.option.RuntimeOptionKey;
import com.oracle.svm.core.option.SubstrateOptionsParser;
import com.oracle.svm.core.thread.VMOperation;
import com.oracle.svm.core.util.VMError;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import org.graalvm.collections.EconomicMap;
import org.graalvm.nativeimage.ImageSingletons;

public final class MetadataTracer {
    private TraceOptions options;
    private JsonFileWriter debugWriter;
    private final ThreadLocal<String> disableTracingReason = new ThreadLocal();
    private volatile ConfigurationSet config;

    @AlwaysInline(value="avoid null check on singleton")
    public static MetadataTracer singleton() {
        return (MetadataTracer)ImageSingletons.lookup(MetadataTracer.class);
    }

    private void initialize(TraceOptions parsedOptions) {
        this.options = parsedOptions;
        this.debugWriter = MetadataTracer.initializeDebugWriter(parsedOptions);
        this.config = MetadataTracer.initializeConfigurationSet(parsedOptions);
    }

    private void shutdown() {
        ConfigurationSet finalConfig = this.config;
        this.config = null;
        if (finalConfig != null) {
            try {
                finalConfig.writeConfiguration(configFile -> this.options.path().resolve(configFile.getFileName()));
            }
            catch (IOException ex) {
                Log log = Log.log();
                log.string("Failed to write out reachability metadata to directory ").string(this.options.path().toString());
                log.string(":").string(ex.getMessage());
                log.newline();
            }
        }
        if (this.debugWriter != null) {
            this.debugWriter.close();
        }
    }

    @AlwaysInline(value="tracing should fold away when disabled")
    public static boolean enabled() {
        return Options.MetadataTracingSupport.getValue() != false && MetadataTracer.singleton().enabledAtRunTime();
    }

    private boolean enabledAtRunTime() {
        VMError.guarantee(Options.MetadataTracingSupport.getValue());
        return this.options != null;
    }

    private ConfigurationSet getConfigurationSetForTracing() {
        if (this.disableTracingReason.get() != null || VMOperation.isInProgress()) {
            return null;
        }
        return this.config;
    }

    public void traceReflectionType(String typeName) {
        this.traceReflectionTypeImpl((ConfigurationTypeDescriptor)new NamedConfigurationTypeDescriptor(typeName));
    }

    public void traceReflectionType(Class<?> clazz) {
        this.traceReflectionTypeImpl(ConfigurationTypeDescriptor.fromClass(clazz));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void traceReflectionArrayType(Class<?> componentClazz) {
        ConfigurationTypeDescriptor typeDescriptor = ConfigurationTypeDescriptor.fromClass(componentClazz);
        if (!(typeDescriptor instanceof NamedConfigurationTypeDescriptor)) {
            this.debug("array type not registered for reflection (component type is not a named type)", typeDescriptor);
            return;
        }
        NamedConfigurationTypeDescriptor namedConfigurationTypeDescriptor = (NamedConfigurationTypeDescriptor)typeDescriptor;
        try {
            String string;
            String name = string = namedConfigurationTypeDescriptor.name();
            this.traceReflectionType(name + "[]");
            return;
        }
        catch (Throwable throwable) {
            throw new MatchException(throwable.toString(), throwable);
        }
    }

    public void traceFieldAccess(Class<?> declaringClass, String fieldName, ConfigurationMemberInfo.ConfigurationMemberDeclaration declaration) {
        ConfigurationTypeDescriptor typeDescriptor = ConfigurationTypeDescriptor.fromClass(declaringClass);
        ConfigurationType type = this.traceReflectionTypeImpl(typeDescriptor);
        if (type != null) {
            this.debugField(typeDescriptor, fieldName);
            type.addField(fieldName, declaration, false);
        }
    }

    public void traceMethodAccess(Class<?> declaringClass, String methodName, String internalSignature, ConfigurationMemberInfo.ConfigurationMemberDeclaration declaration) {
        ConfigurationTypeDescriptor typeDescriptor = ConfigurationTypeDescriptor.fromClass(declaringClass);
        ConfigurationType type = this.traceReflectionTypeImpl(typeDescriptor);
        if (type != null) {
            this.debugMethod(typeDescriptor, methodName, internalSignature);
            type.addMethod(methodName, internalSignature, declaration, ConfigurationMemberInfo.ConfigurationMemberAccessibility.ACCESSED);
        }
    }

    public void traceUnsafeAllocatedType(Class<?> clazz) {
        ConfigurationTypeDescriptor typeDescriptor = ConfigurationTypeDescriptor.fromClass(clazz);
        ConfigurationType type = this.traceReflectionTypeImpl(typeDescriptor);
        if (type != null) {
            this.debug("type marked as unsafely allocated", clazz.getTypeName());
            type.setUnsafeAllocated();
        }
    }

    public void traceProxyType(Class<?>[] interfaces) {
        List<String> interfaceNames = Arrays.stream(interfaces).map(Class::getTypeName).toList();
        ProxyConfigurationTypeDescriptor descriptor = new ProxyConfigurationTypeDescriptor(interfaceNames);
        this.traceReflectionTypeImpl((ConfigurationTypeDescriptor)descriptor);
    }

    private ConfigurationType traceReflectionTypeImpl(ConfigurationTypeDescriptor typeDescriptor) {
        assert (this.enabledAtRunTime());
        if (MetadataTracer.isInternal(typeDescriptor)) {
            this.debug("type not registered for reflection (uses an internal interface)", typeDescriptor);
            return null;
        }
        ConfigurationSet configurationSet = this.getConfigurationSetForTracing();
        if (configurationSet != null) {
            this.debugReflectionType(typeDescriptor, configurationSet);
            return configurationSet.getReflectionConfiguration().getOrCreateType(UnresolvedConfigurationCondition.alwaysTrue(), typeDescriptor);
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static boolean isInternal(ConfigurationTypeDescriptor typeDescriptor) {
        if (typeDescriptor instanceof NamedConfigurationTypeDescriptor) {
            NamedConfigurationTypeDescriptor namedConfigurationTypeDescriptor = (NamedConfigurationTypeDescriptor)typeDescriptor;
            try {
                String string;
                String name = string = namedConfigurationTypeDescriptor.name();
                return MetadataTracer.isInternal(name);
            }
            catch (Throwable throwable) {
                throw new MatchException(throwable.toString(), throwable);
            }
        }
        if (!(typeDescriptor instanceof ProxyConfigurationTypeDescriptor)) return false;
        ProxyConfigurationTypeDescriptor proxyType = (ProxyConfigurationTypeDescriptor)typeDescriptor;
        for (String interfaceName : proxyType.interfaceNames()) {
            if (!MetadataTracer.isInternal(interfaceName)) continue;
            return true;
        }
        return false;
    }

    private static boolean isInternal(String typeName) {
        return typeName.startsWith("com.oracle.svm.core");
    }

    public void traceJNIType(String typeName) {
        this.traceJNITypeImpl((ConfigurationTypeDescriptor)new NamedConfigurationTypeDescriptor(typeName));
    }

    public void traceJNIType(Class<?> clazz) {
        this.traceJNITypeImpl(ConfigurationTypeDescriptor.fromClass(clazz));
    }

    private void traceJNITypeImpl(ConfigurationTypeDescriptor typeDescriptor) {
        assert (this.enabledAtRunTime());
        ConfigurationType type = this.traceReflectionTypeImpl(typeDescriptor);
        if (type != null && !type.isJniAccessible()) {
            this.debug("type registered for jni", typeDescriptor);
            type.setJniAccessible();
        }
    }

    public void traceResource(String resourceName, String moduleName) {
        assert (this.enabledAtRunTime());
        ConfigurationSet configurationSet = this.getConfigurationSetForTracing();
        if (configurationSet != null) {
            this.debugResourceGlob(resourceName, moduleName);
            configurationSet.getResourceConfiguration().addGlobPattern(UnresolvedConfigurationCondition.alwaysTrue(), resourceName, moduleName);
        }
    }

    public void traceResourceBundle(String baseName) {
        assert (this.enabledAtRunTime());
        ConfigurationSet configurationSet = this.getConfigurationSetForTracing();
        if (configurationSet != null) {
            this.debug("resource bundle registered", baseName);
            configurationSet.getResourceConfiguration().addBundle(UnresolvedConfigurationCondition.alwaysTrue(), baseName, List.of());
        }
    }

    public void traceSerializationType(Class<?> clazz) {
        assert (this.enabledAtRunTime());
        ConfigurationTypeDescriptor typeDescriptor = ConfigurationTypeDescriptor.fromClass(clazz);
        ConfigurationType result = this.traceReflectionTypeImpl(typeDescriptor);
        if (result != null && !result.isSerializable()) {
            this.debug("type registered for serialization", typeDescriptor);
            result.setSerializable();
        }
    }

    private void debug(String message, Object element) {
        if (this.debugWriter == null) {
            return;
        }
        assert (this.enabledAtRunTime());
        try (DisableTracingImpl ignored = new DisableTracingImpl("debug logging");){
            EconomicMap entry = EconomicMap.create();
            entry.put((Object)"message", (Object)message);
            entry.put((Object)"element", element);
            entry.put((Object)"stacktrace", (Object)MetadataTracer.debugStackTrace());
            this.debugWriter.printObject(entry);
        }
    }

    private static StackTraceElement[] debugStackTrace() {
        int i;
        StackTraceElement[] trace = Thread.currentThread().getStackTrace();
        for (i = 0; i < trace.length && (trace[i].getMethodName().contains("getStackTrace") || trace[i].getMethodName().startsWith("debug")); ++i) {
        }
        return Arrays.copyOfRange(trace, i, trace.length);
    }

    private void debugResourceGlob(String resourceName, String moduleName) {
        if (this.debugWriter == null) {
            return;
        }
        String element = moduleName == null ? resourceName : String.format("%s:%s", moduleName, resourceName);
        this.debug("resource glob registered", element);
    }

    private void debugReflectionType(ConfigurationTypeDescriptor typeDescriptor, ConfigurationSet configurationSet) {
        if (this.debugWriter == null) {
            return;
        }
        if (configurationSet.getReflectionConfiguration().get(UnresolvedConfigurationCondition.alwaysTrue(), typeDescriptor) == null) {
            this.debug("type registered for reflection", typeDescriptor);
        }
    }

    private void debugField(ConfigurationTypeDescriptor typeDescriptor, String fieldName) {
        if (this.debugWriter == null) {
            return;
        }
        this.debug("field registered for reflection", String.valueOf(typeDescriptor) + "." + fieldName);
    }

    private void debugMethod(ConfigurationTypeDescriptor typeDescriptor, String methodName, String internalSignature) {
        if (this.debugWriter == null) {
            return;
        }
        this.debug("method registered for reflection", String.valueOf(typeDescriptor) + "." + methodName + internalSignature);
    }

    public static DisableTracing disableTracing(String reason) {
        if (Options.MetadataTracingSupport.getValue().booleanValue() && MetadataTracer.singleton().enabledAtRunTime()) {
            MetadataTracer metadataTracer = MetadataTracer.singleton();
            Objects.requireNonNull(metadataTracer);
            return metadataTracer.new DisableTracingImpl(reason);
        }
        return DisableTracingNoOp.INSTANCE;
    }

    private static void initializeSingleton(String recordMetadataValue) {
        assert (Options.MetadataTracingSupport.getValue().booleanValue());
        MetadataTracer.singleton().initialize(TraceOptions.parse(recordMetadataValue));
    }

    private static JsonFileWriter initializeDebugWriter(TraceOptions options) {
        if (options.debugLog() == null) {
            return null;
        }
        try {
            Path parentDir = options.debugLog().getParent();
            if (parentDir == null) {
                throw new IllegalArgumentException("Invalid debug-log path '" + String.valueOf(options.debugLog()) + "'.");
            }
            Files.createDirectories(parentDir, new FileAttribute[0]);
            return new JsonFileWriter(options.debugLog());
        }
        catch (IOException ex) {
            throw new IllegalArgumentException("Exception occurred preparing the debug log file (" + String.valueOf(options.debugLog()) + ")", ex);
        }
    }

    private static ConfigurationSet initializeConfigurationSet(TraceOptions options) {
        try {
            Files.createDirectories(options.path(), new FileAttribute[0]);
        }
        catch (IOException ex) {
            throw new IllegalArgumentException("Exception occurred creating the output directory for tracing (" + String.valueOf(options.path()) + ")", ex);
        }
        if (options.merge() && Files.exists(options.path(), new LinkOption[0])) {
            ConfigurationFileCollection mergeConfigs = new ConfigurationFileCollection();
            mergeConfigs.addDirectory(options.path());
            try {
                return mergeConfigs.loadConfigurationSet(ioexception -> ioexception, null, null);
            }
            catch (Exception ex) {
                Log.log().string("An exception occurred when loading merge metadata from path " + String.valueOf(options.path()) + ". ").string("Any existing metadata may be overwritten.").newline().string("Exception: ").exception(ex).newline();
            }
        }
        return new ConfigurationSet();
    }

    private static void shutdownSingleton() {
        assert (Options.MetadataTracingSupport.getValue().booleanValue());
        MetadataTracer.singleton().shutdown();
    }

    static RuntimeSupport.Hook initializeMetadataTracingHook() {
        return isFirstIsolate -> {
            if (!isFirstIsolate) {
                return;
            }
            VMError.guarantee(Options.MetadataTracingSupport.getValue());
            if (Options.TraceMetadata.hasBeenSet()) {
                MetadataTracer.initializeSingleton(Options.TraceMetadata.getValue());
            }
        };
    }

    static RuntimeSupport.Hook shutDownMetadataTracingHook() {
        return isFirstIsolate -> {
            if (!isFirstIsolate) {
                return;
            }
            VMError.guarantee(Options.MetadataTracingSupport.getValue());
            if (Options.TraceMetadata.hasBeenSet()) {
                MetadataTracer.shutdownSingleton();
            }
        };
    }

    static RuntimeSupport.Hook checkImproperOptionUsageHook() {
        String hostedOptionCommandArgument = SubstrateOptionsParser.commandArgument(Options.MetadataTracingSupport, "+");
        return isFirstIsolate -> {
            if (!isFirstIsolate) {
                return;
            }
            VMError.guarantee(Options.MetadataTracingSupport.getValue() == false);
            if (Options.TraceMetadata.hasBeenSet()) {
                throw new IllegalArgumentException("The option " + Options.TraceMetadata.getName() + " can only be used if metadata tracing is enabled at build time (using " + hostedOptionCommandArgument + ").");
            }
        };
    }

    public static class Options {
        public static final HostedOptionKey<Boolean> MetadataTracingSupport = new HostedOptionKey<Boolean>(false);
        static final String TRACE_METADATA_HELP = "Enables metadata tracing at run time. This option is only supported if -H:+MetadataTracingSupport is set when building the image.\nThe value of this option is a comma-separated list of arguments specified as key-value pairs. The following arguments are supported:\n\n- path=<trace-output-directory> (required): Specifies the directory to write traced metadata to.\n- merge=<boolean> (optional): Specifies whether to merge or overwrite metadata with existing files at the output path (default: true).\n- debug-log=<path> (optional): Specifies a path to write debug output to. This option is meant for debugging; the option name and its\n  output format may change at any time.\n\nExample usage:\n    -H:TraceMetadata=path=trace_output_directory\n    -H:TraceMetadata=path=trace_output_directory,merge=false\n";
        public static final RuntimeOptionKey<String> TraceMetadata = new RuntimeOptionKey<Object>(null, new RuntimeOptionKey.RuntimeOptionKeyFlag[0]);
    }

    private final class DisableTracingImpl
    implements DisableTracing {
        final String oldReason;

        private DisableTracingImpl(String reason) {
            this.oldReason = MetadataTracer.this.disableTracingReason.get();
            MetadataTracer.this.disableTracingReason.set(reason);
        }

        @Override
        public void close() {
            MetadataTracer.this.disableTracingReason.set(this.oldReason);
        }
    }

    private static final class DisableTracingNoOp
    implements DisableTracing {
        private static final DisableTracingNoOp INSTANCE = new DisableTracingNoOp();

        private DisableTracingNoOp() {
        }

        @Override
        public void close() {
        }
    }

    public static sealed interface DisableTracing
    extends AutoCloseable
    permits DisableTracingImpl, DisableTracingNoOp {
        @Override
        public void close();
    }
}

