/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.driver;

import com.oracle.svm.core.SubstrateUtil;
import com.oracle.svm.core.option.SubstrateOptionsParser;
import com.oracle.svm.core.util.ClasspathUtils;
import com.oracle.svm.driver.NativeImage;
import com.oracle.svm.driver.NativeImageServerHelper;
import com.oracle.svm.driver.ServerOptionHandler;
import com.oracle.svm.driver.Unistd;
import com.oracle.svm.hosted.server.NativeImageBuildClient;
import com.oracle.svm.hosted.server.SubstrateServerMessage;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.graalvm.nativeimage.ProcessProperties;
import org.graalvm.word.WordFactory;
import sun.misc.Signal;
import sun.misc.SignalHandler;

final class NativeImageServer
extends NativeImage {
    private static final String serverDirPrefix = "server-id-";
    private static final String machineDirPrefix = "machine-id-";
    private static final String sessionDirPrefix = "session-id-";
    private static final String defaultLockFileName = ".lock";
    private static final String pKeyMaxServers = "MaxServers";
    private static final String machineProperties = "machine.properties";
    private boolean useServer = true;
    private boolean verboseServer = false;
    private String sessionName = null;
    private volatile Server building = null;
    private final List<FileChannel> openFileChannels = new ArrayList<FileChannel>();
    private final ServerOptionHandler serverOptionHandler = new ServerOptionHandler(this);

    private NativeImageServer(NativeImage.BuildConfiguration buildConfiguration) {
        super(buildConfiguration);
        this.registerOptionHandler(this.serverOptionHandler);
    }

    static NativeImage create(NativeImage.BuildConfiguration config) {
        if (NativeImageServerHelper.isInConfiguration()) {
            return new NativeImageServer(config);
        }
        return new NativeImage(config);
    }

    private static Consumer<byte[]> byteStreamToByteConsumer(OutputStream stream) {
        return b -> {
            try {
                stream.write((byte[])b);
            }
            catch (IOException e) {
                throw new RuntimeException("Byte stream write failed.");
            }
        };
    }

    private String getSessionID() {
        if (this.sessionName != null) {
            return sessionDirPrefix + this.sessionName;
        }
        return sessionDirPrefix + Long.toHexString(Unistd.getsid(Math.toIntExact(ProcessProperties.getProcessID())));
    }

    private static String getMachineID() {
        try {
            return Files.lines(Paths.get("/etc/machine-id", new String[0])).collect(Collectors.joining("", machineDirPrefix, ""));
        }
        catch (Exception e) {
            return "machine-id-hostid-" + Long.toHexString(Unistd.gethostid());
        }
    }

    private Path getMachineDir() {
        Path machineDir = this.getUserConfigDir().resolve(NativeImageServer.getMachineID());
        NativeImageServer.ensureDirectoryExists(machineDir);
        return machineDir;
    }

    private Path getSessionDir() {
        Path sessionDir = this.getMachineDir().resolve(this.getSessionID());
        NativeImageServer.ensureDirectoryExists(sessionDir);
        return sessionDir;
    }

    private static String getDurationString(Instant since) {
        long seconds = ChronoUnit.SECONDS.between(since, Instant.now());
        long hours = TimeUnit.SECONDS.toHours(seconds);
        long minutes = TimeUnit.SECONDS.toMinutes(seconds -= TimeUnit.HOURS.toSeconds(hours));
        return String.format("%02d:%02d:%02d", hours, minutes, seconds -= TimeUnit.MINUTES.toSeconds(minutes));
    }

    private Server getServerInstance(LinkedHashSet<Path> classpath, LinkedHashSet<Path> bootClasspath, List<String> javaArgs) {
        Server[] result = new Server[]{null};
        this.withFileChannel(this.getMachineDir().resolve("create-server.lock"), lockFileChannel -> {
            try (FileLock lock = NativeImageServer.lockFileChannel(lockFileChannel);){
                List<Server> aliveServers = this.cleanupServers(false, true, true);
                String maxServersStr = NativeImageServer.loadProperties(this.getMachineDir().resolve(machineProperties)).get(pKeyMaxServers);
                if (maxServersStr == null || maxServersStr.isEmpty()) {
                    maxServersStr = this.getUserConfigProperties().get(pKeyMaxServers);
                }
                int maxServers = maxServersStr == null || maxServersStr.isEmpty() ? 2 : Math.max(1, Integer.parseInt(maxServersStr));
                String xmxValueStr = this.getXmxValue(maxServers);
                NativeImageServer.replaceArg(javaArgs, "-Xmx", xmxValueStr);
                String xmsValueStr = this.getXmsValue();
                long xmxValue = SubstrateOptionsParser.parseLong((String)xmxValueStr);
                long xmsValue = SubstrateOptionsParser.parseLong((String)xmsValueStr);
                if (WordFactory.unsigned((long)xmsValue).aboveThan(WordFactory.unsigned((long)xmxValue))) {
                    xmsValueStr = Long.toUnsignedString(xmxValue);
                }
                NativeImageServer.replaceArg(javaArgs, "-Xms", xmsValueStr);
                Path sessionDir = this.getSessionDir();
                ArrayList<Collection<Path>> builderPaths = new ArrayList<Collection<Path>>(Arrays.asList(classpath, bootClasspath));
                if (this.config.useJavaModules()) {
                    builderPaths.addAll(Arrays.asList(this.config.getBuilderModulePath(), this.config.getBuilderUpgradeModulePath()));
                }
                Path javaExePath = this.canonicalize(this.config.getJavaExecutable());
                String serverUID = NativeImageServer.imageServerUID(javaExePath, javaArgs, builderPaths);
                Path serverDir = sessionDir.resolve(serverDirPrefix + serverUID);
                Optional<Server> reusableServer = aliveServers.stream().filter(s -> s.serverDir.equals(serverDir)).findFirst();
                if (reusableServer.isPresent()) {
                    Server server = reusableServer.get();
                    if (!server.isAlive()) {
                        throw NativeImageServer.showError("Found defunct image-build server:" + server.getServerInfo());
                    }
                    this.showVerboseMessage(this.verboseServer, "Reuse existing image-build server: " + server);
                    result[0] = server;
                } else {
                    Server server;
                    if (aliveServers.size() >= maxServers) {
                        this.showVerboseMessage(this.verboseServer, "Image-build server limit reached -> remove least recently used");
                        Server victim = NativeImageServer.findVictim(aliveServers);
                        if (victim != null) {
                            this.showMessage("Shutdown " + victim);
                            victim.shutdown();
                        } else {
                            NativeImageServer.showWarning("Image-build server limit exceeded. Use options --server{-list,-shutdown[-all]} to fix the problem.");
                        }
                    }
                    if ((server = this.startServer(javaExePath, serverDir, 0, classpath, bootClasspath, javaArgs)) == null) {
                        NativeImageServer.showWarning("Creating image-build server failed. Fallback to one-shot image building ...");
                    }
                    result[0] = server;
                }
            }
            catch (IOException e) {
                throw NativeImageServer.showError("ServerInstance-creation locking failed", e);
            }
        });
        return result[0];
    }

    private static Server findVictim(List<Server> aliveServers) {
        return aliveServers.stream().filter(Server::isAlive).min(Comparator.comparing(s -> s.lastBuildRequest)).orElse(null);
    }

    private List<Path> getSessionDirs(boolean machineWide) {
        List<Path> sessionDirs = Collections.singletonList(this.getSessionDir());
        if (machineWide) {
            try {
                sessionDirs = Files.list(this.getMachineDir()).filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).filter(sessionDir -> sessionDir.getFileName().toString().startsWith(sessionDirPrefix)).collect(Collectors.toList());
            }
            catch (IOException e) {
                throw NativeImageServer.showError("Accessing MachineDir " + this.getMachineDir() + " failed", e);
            }
        }
        return sessionDirs;
    }

    private List<Server> findServers(List<Path> sessionDirs) {
        ArrayList<Server> servers = new ArrayList<Server>();
        for (Path sessionDir : sessionDirs) {
            try {
                Files.list(sessionDir).filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(serverDir -> {
                    if (serverDir.getFileName().toString().startsWith(serverDirPrefix)) {
                        try {
                            servers.add(new Server((Path)serverDir));
                        }
                        catch (Exception e) {
                            this.showVerboseMessage(this.verboseServer, "Found corrupt ServerDir " + serverDir);
                            this.deleteAllFiles((Path)serverDir);
                        }
                    }
                });
            }
            catch (IOException e) {
                throw NativeImageServer.showError("Accessing SessionDir " + sessionDir + " failed", e);
            }
        }
        return servers;
    }

    void listServers(boolean machineWide, boolean details) {
        List<Server> servers = this.findServers(this.getSessionDirs(machineWide));
        for (Server server : servers) {
            String sessionInfo = machineWide ? "Session " + server.serverDir.getParent().getFileName() + " " : "";
            this.showMessage(sessionInfo + server + server.getLivenessInfo(server.isAlive()) + server.getLastBuildInfo());
            if (!details) continue;
            this.showMessage("Details:");
            this.showMessage(server.getServerInfo());
        }
    }

    void wipeMachineDir() {
        this.deleteAllFiles(this.getMachineDir());
    }

    List<Server> cleanupServers(boolean serverShutdown, boolean machineWide, boolean quiet) {
        List<Path> sessionDirs = this.getSessionDirs(machineWide);
        for (Path sessionDir : sessionDirs) {
            if (!NativeImageServer.isDeletedPath(sessionDir)) continue;
            this.deleteAllFiles(sessionDir);
        }
        ArrayList<Server> aliveServers = new ArrayList<Server>();
        for (Path sessionDir : sessionDirs) {
            this.withLockDirFileChannel(sessionDir, lockFileChannel -> {
                try (FileLock lock = lockFileChannel.lock();){
                    List<Server> servers = this.findServers(Collections.singletonList(sessionDir));
                    for (Server server : servers) {
                        boolean alive = server.isAlive();
                        if (!alive || serverShutdown) {
                            if (!quiet) {
                                this.showMessage("Cleanup " + server + server.getLivenessInfo(alive));
                            }
                            server.shutdown();
                            continue;
                        }
                        aliveServers.add(server);
                    }
                }
                catch (IOException e) {
                    throw NativeImageServer.showError("Locking SessionDir " + sessionDir + " failed", e);
                }
            });
        }
        return aliveServers;
    }

    private Server startServer(Path javaExePath, Path serverDir, int serverPort, LinkedHashSet<Path> classpath, LinkedHashSet<Path> bootClasspath, List<String> javaArgs) {
        ProcessBuilder pb = new ProcessBuilder(new String[0]);
        pb.directory(serverDir.toFile());
        pb.redirectErrorStream(true);
        List<String> command = pb.command();
        command.add(javaExePath.toString());
        if (!bootClasspath.isEmpty()) {
            command.add(bootClasspath.stream().map(Path::toString).collect(Collectors.joining(File.pathSeparator, "-Xbootclasspath/a:", "")));
        }
        command.addAll(Arrays.asList("-cp", classpath.stream().map(Path::toString).collect(Collectors.joining(File.pathSeparator))));
        command.addAll(javaArgs);
        command.add("-Dgraal.LogFile=%e");
        command.add("com.oracle.svm.hosted.server.NativeImageBuildServer");
        command.add("-port=" + serverPort);
        Path logFilePath = serverDir.resolve("server.log");
        command.add("-logFile=" + logFilePath);
        this.showVerboseMessage(this.isVerbose(), "StartServer [");
        this.showVerboseMessage(this.isVerbose(), SubstrateUtil.getShellCommandString(command, (boolean)true));
        this.showVerboseMessage(this.isVerbose(), "]");
        int childPid = NativeImageServerHelper.daemonize(() -> {
            try {
                NativeImageServer.ensureDirectoryExists(serverDir);
                this.showVerboseMessage(this.verboseServer, "Starting new server ...");
                Process process = pb.start();
                long serverPID = ProcessProperties.getProcessID((Process)process);
                this.showVerboseMessage(this.verboseServer, "New image-build server pid: " + serverPID);
                int selectedPort = serverPort;
                if (selectedPort == 0) {
                    try (BufferedReader serverStdout = new BufferedReader(new InputStreamReader(process.getInputStream(), StandardCharsets.UTF_8));){
                        String line;
                        int readLineTries = 60;
                        ArrayList<String> lines = new ArrayList<String>(readLineTries);
                        while ((line = serverStdout.readLine()) != null && --readLineTries > 0) {
                            lines.add(line);
                            if (!line.startsWith("Started image build server on port: ")) continue;
                            String portStr = line.substring("Started image build server on port: ".length());
                            try {
                                selectedPort = Integer.parseInt(portStr);
                                break;
                            }
                            catch (NumberFormatException numberFormatException) {
                            }
                        }
                        if (selectedPort == 0) {
                            String serverOutputMessage = "";
                            if (!lines.isEmpty()) {
                                serverOutputMessage = "\nServer stdout/stderr:\n" + String.join((CharSequence)"\n", lines);
                            }
                            throw NativeImageServer.showError("Could not determine port for sending image-build requests." + serverOutputMessage);
                        }
                        this.showVerboseMessage(this.verboseServer, "Image-build server selected port " + selectedPort);
                    }
                }
                NativeImageServer.writeServerFile(serverDir, selectedPort, serverPID, classpath, bootClasspath, javaArgs);
            }
            catch (Throwable e) {
                this.deleteAllFiles(serverDir);
                throw NativeImageServer.showError("Starting image-build server instance failed", e);
            }
        });
        int exitStatus = ProcessProperties.waitForProcessExit((long)childPid);
        this.showVerboseMessage(this.verboseServer, "Exit status forked child process: " + exitStatus);
        if (exitStatus == 0) {
            Server server;
            try {
                server = new Server(serverDir);
            }
            catch (Exception e) {
                this.showVerboseMessage(this.verboseServer, "Image-build server unusable.");
                return null;
            }
            for (int i = 0; i < 6; ++i) {
                if (server.isAlive()) {
                    this.showVerboseMessage(this.verboseServer, "Image-build server found.");
                    return server;
                }
                try {
                    Thread.sleep(200L);
                    continue;
                }
                catch (InterruptedException e) {
                    break;
                }
            }
            this.showVerboseMessage(this.verboseServer, "Image-build server not responding.");
            server.shutdown();
        }
        return null;
    }

    private static void writeServerFile(Path serverDir, int port, long pid, LinkedHashSet<Path> classpath, LinkedHashSet<Path> bootClasspath, List<String> javaArgs) throws Exception {
        Properties sp = new Properties();
        sp.setProperty("Port", String.valueOf(port));
        sp.setProperty("PID", String.valueOf(pid));
        sp.setProperty("JavaArgs", String.join((CharSequence)" ", javaArgs));
        sp.setProperty("BootClasspath", bootClasspath.stream().map(ClasspathUtils::classpathToString).collect(Collectors.joining(" ")));
        sp.setProperty("Classpath", classpath.stream().map(ClasspathUtils::classpathToString).collect(Collectors.joining(" ")));
        Path serverPropertiesPath = serverDir.resolve("server.properties");
        try (OutputStream os = Files.newOutputStream(serverPropertiesPath, new OpenOption[0]);){
            sp.store(os, "");
        }
    }

    private void withLockDirFileChannel(Path lockDir, Consumer<FileChannel> consumer) {
        this.withFileChannel(lockDir.resolve(defaultLockFileName), consumer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void withFileChannel(Path filePath, Consumer<FileChannel> consumer) {
        try {
            FileChannel fileChannel;
            List<FileChannel> list = this.openFileChannels;
            synchronized (list) {
                fileChannel = FileChannel.open(filePath, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
                this.openFileChannels.add(fileChannel);
            }
            consumer.accept(fileChannel);
            list = this.openFileChannels;
            synchronized (list) {
                this.openFileChannels.remove(fileChannel);
                fileChannel.close();
            }
        }
        catch (IOException e) {
            throw NativeImageServer.showError("Using FileChannel for " + filePath + " failed", e);
        }
    }

    private static FileLock lockFileChannel(FileChannel channel) throws IOException {
        Thread lockWatcher = new Thread(() -> {
            try {
                Thread.sleep(TimeUnit.MINUTES.toMillis(10L));
                try {
                    NativeImageServer.showWarning("Timeout while waiting for FileChannel.lock");
                    channel.close();
                }
                catch (IOException e) {
                    throw NativeImageServer.showError("LockWatcher closing FileChannel of LockFile failed", e);
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        });
        lockWatcher.start();
        FileLock lock = channel.lock();
        lockWatcher.interrupt();
        return lock;
    }

    @Override
    protected int buildImage(List<String> javaArgs, LinkedHashSet<Path> bcp, LinkedHashSet<Path> cp, LinkedHashSet<String> imageArgs, LinkedHashSet<Path> imagecp) {
        boolean printFlags = imageArgs.stream().anyMatch(arg -> arg.contains(this.enablePrintFlags));
        if (this.useServer && !printFlags && !this.useDebugAttach()) {
            AbortBuildSignalHandler signalHandler = new AbortBuildSignalHandler();
            Signal.handle(new Signal("TERM"), signalHandler);
            Signal.handle(new Signal("INT"), signalHandler);
            Server server = this.getServerInstance(cp, bcp, javaArgs);
            if (server != null) {
                this.showVerboseMessage(this.verboseServer, "\n" + server.getServerInfo() + "\n");
                this.showMessage("Build on " + server);
                this.building = server;
                int status = server.sendBuildRequest(imagecp, imageArgs);
                if (!server.isAlive()) {
                    this.cleanupServers(false, false, true);
                }
                return status;
            }
        }
        return super.buildImage(javaArgs, bcp, cp, imageArgs, imagecp);
    }

    private static String imageServerUID(Path javaExecutable, List<String> vmArgs, List<Collection<Path>> builderPaths) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance("SHA-512");
        }
        catch (NoSuchAlgorithmException e) {
            throw NativeImageServer.showError("SHA-512 digest is not available", e);
        }
        digest.update(javaExecutable.toString().getBytes());
        for (Collection<Path> paths : builderPaths) {
            for (Path path : paths) {
                digest.update(path.toString().getBytes());
                NativeImageServer.updateHash(digest, path);
            }
        }
        for (String string : vmArgs) {
            digest.update(string.getBytes());
        }
        byte[] digestBytes = digest.digest();
        StringBuilder sb = new StringBuilder(digestBytes.length * 2);
        for (Object b : (Object)digestBytes) {
            sb.append(String.format("%02x", b & 0xFF));
        }
        return sb.toString();
    }

    private static void updateHash(MessageDigest md, Path pathElement) {
        try {
            if (!Files.isReadable(pathElement) || !pathElement.getFileName().toString().endsWith(".jar")) {
                throw NativeImageServer.showError("Build server classpath must only contain valid jar-files: " + pathElement);
            }
            md.update(Files.readAllBytes(pathElement));
        }
        catch (IOException e) {
            throw NativeImageServer.showError("Problem reading classpath entries: " + e.getMessage());
        }
    }

    void setUseServer(boolean val) {
        this.useServer = val;
    }

    boolean useServer() {
        return this.useServer;
    }

    @Override
    protected void setDryRun(boolean val) {
        super.setDryRun(val);
        this.useServer = !val;
    }

    void setVerboseServer(boolean val) {
        this.verboseServer = val;
    }

    boolean verboseServer() {
        return this.verboseServer;
    }

    void setSessionName(String val) {
        if (val != null && val.isEmpty()) {
            throw NativeImageServer.showError("Empty string not allowed as session-name");
        }
        this.sessionName = val;
    }

    private final class AbortBuildSignalHandler
    implements SignalHandler {
        private int attemptCount = 0;

        private AbortBuildSignalHandler() {
        }

        @Override
        public void handle(Signal signal) {
            Server current = NativeImageServer.this.building;
            if (this.attemptCount >= 3 || current == null) {
                this.killServer();
                this.closeFileChannels();
                System.exit(1);
            } else {
                ++this.attemptCount;
                current.abortTask();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void closeFileChannels() {
            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.isVerbose(), "CleanupHandler Begin");
            List list = NativeImageServer.this.openFileChannels;
            synchronized (list) {
                for (FileChannel fileChannel : NativeImageServer.this.openFileChannels) {
                    try {
                        NativeImageServer.this.showVerboseMessage(NativeImageServer.this.isVerbose(), "Closing open FileChannel: " + fileChannel);
                        fileChannel.close();
                    }
                    catch (Exception e) {
                        throw NativeImage.showError("Closing FileChannel failed", e);
                    }
                }
            }
            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.isVerbose(), "CleanupHandler End");
        }

        private void killServer() {
            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.isVerbose(), "Caught interrupt. Kill Server.");
            Server current = NativeImageServer.this.building;
            if (current != null) {
                current.shutdown();
            }
        }
    }

    private final class Server {
        private static final String serverProperties = "server.properties";
        private static final String buildRequestLog = "build-request.log";
        private static final String pKeyPort = "Port";
        private static final String pKeyPID = "PID";
        private static final String pKeyJavaArgs = "JavaArgs";
        private static final String pKeyBCP = "BootClasspath";
        private static final String pKeyCP = "Classpath";
        final Path serverDir;
        final Instant since;
        Instant lastBuildRequest;
        final int port;
        final int pid;
        final LinkedHashSet<String> serverJavaArgs;
        final LinkedHashSet<Path> serverBootClasspath;
        final LinkedHashSet<Path> serverClasspath;

        private Server(Path serverDir) throws Exception {
            this.serverDir = serverDir;
            Path serverPropertiesPath = serverDir.resolve(serverProperties);
            Map<String, String> properties = NativeImage.loadProperties(serverPropertiesPath);
            this.pid = Integer.parseInt(properties.get(pKeyPID));
            this.port = Integer.parseInt(properties.get(pKeyPort));
            if (this.port == 0) {
                ProcessProperties.destroyForcibly((long)this.pid);
                NativeImageServer.this.deleteAllFiles(this.serverDir);
                throw new ServerInstanceError();
            }
            this.serverJavaArgs = new LinkedHashSet<String>(Arrays.asList(properties.get(pKeyJavaArgs).split(" ")));
            this.serverBootClasspath = this.readClasspath(properties.get(pKeyBCP));
            this.serverClasspath = this.readClasspath(properties.get(pKeyCP));
            BasicFileAttributes sinceAttrs = Files.readAttributes(serverPropertiesPath, BasicFileAttributes.class, new LinkOption[0]);
            this.since = sinceAttrs.creationTime().toInstant();
            this.updateLastBuildRequest();
        }

        private void updateLastBuildRequest() throws IOException {
            Path buildRequestLogPath = this.serverDir.resolve(buildRequestLog);
            if (Files.isReadable(buildRequestLogPath)) {
                BasicFileAttributes buildAttrs = Files.readAttributes(buildRequestLogPath, BasicFileAttributes.class, new LinkOption[0]);
                this.lastBuildRequest = buildAttrs.lastModifiedTime().toInstant().plusSeconds(1L);
            } else {
                this.lastBuildRequest = this.since;
            }
        }

        private LinkedHashSet<Path> readClasspath(String rawClasspathString) {
            LinkedHashSet<Path> result = new LinkedHashSet<Path>();
            for (String pathStr : rawClasspathString.split(" ")) {
                result.add(ClasspathUtils.stringToClasspath((String)pathStr));
            }
            return result;
        }

        private int sendRequest(Consumer<byte[]> out, Consumer<byte[]> err, SubstrateServerMessage.ServerCommand serverCommand, String ... args) {
            List<String> argList = Arrays.asList(args);
            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.verboseServer, "Sending to server [");
            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.verboseServer, serverCommand.toString());
            if (argList.size() > 0) {
                NativeImageServer.this.showVerboseMessage(NativeImageServer.this.verboseServer, String.join((CharSequence)" \\\n", argList));
            }
            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.verboseServer, "]");
            int exitCode = NativeImageBuildClient.sendRequest((SubstrateServerMessage.ServerCommand)serverCommand, (byte[])String.join((CharSequence)"\n", argList).getBytes(), (int)this.port, out, err);
            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.verboseServer, "Server returns: " + exitCode);
            return exitCode;
        }

        int sendBuildRequest(LinkedHashSet<Path> imageCP, LinkedHashSet<String> imageArgs) {
            int[] requestStatus = new int[]{1};
            NativeImageServer.this.withLockDirFileChannel(this.serverDir, lockFileChannel -> {
                boolean abortedOnce = false;
                boolean finished = false;
                while (!finished) {
                    try {
                        FileLock lock = lockFileChannel.tryLock();
                        Throwable throwable = null;
                        try {
                            if (lock != null) {
                                finished = true;
                                if (abortedOnce) {
                                    NativeImageServer.this.showMessage("DONE.");
                                }
                            } else {
                                if (!abortedOnce) {
                                    NativeImageServer.this.showMessagePart("A previous build is in progress. Aborting previous build...");
                                    this.abortTask();
                                    abortedOnce = true;
                                }
                                try {
                                    Thread.sleep(3000L);
                                    continue;
                                }
                                catch (InterruptedException e) {
                                    throw NativeImage.showError("Woke up from waiting for previous build to abort", e);
                                }
                            }
                            ArrayList<String> command = new ArrayList<String>();
                            command.add("-task=com.oracle.svm.hosted.NativeImageGeneratorRunner");
                            LinkedHashSet<Path> imagecp = new LinkedHashSet<Path>(this.serverClasspath);
                            imagecp.addAll(imageCP);
                            command.addAll(NativeImage.createImageBuilderArgs(imageArgs, imagecp));
                            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.isVerbose(), "SendBuildRequest [");
                            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.isVerbose(), String.join((CharSequence)"\n", command));
                            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.isVerbose(), "]");
                            try {
                                String logFileEntry = command.stream().collect(Collectors.joining(" ", Instant.now().toString() + ": ", "\n"));
                                Files.write(this.serverDir.resolve(buildRequestLog), logFileEntry.getBytes(), StandardOpenOption.CREATE, StandardOpenOption.APPEND);
                                this.updateLastBuildRequest();
                            }
                            catch (IOException e) {
                                throw NativeImage.showError("Could not read/write into build-request log file", e);
                            }
                            requestStatus[0] = this.sendRequest(NativeImageServer.byteStreamToByteConsumer(System.out), NativeImageServer.byteStreamToByteConsumer(System.err), SubstrateServerMessage.ServerCommand.BUILD_IMAGE, command.toArray(new String[command.size()]));
                        }
                        catch (Throwable throwable2) {
                            throwable = throwable2;
                            throw throwable2;
                        }
                        finally {
                            if (lock == null) continue;
                            if (throwable != null) {
                                try {
                                    lock.close();
                                }
                                catch (Throwable e) {
                                    throwable.addSuppressed(e);
                                }
                                continue;
                            }
                            lock.close();
                        }
                    }
                    catch (IOException e) {
                        throw NativeImage.showError("Error while trying to lock ServerDir " + this.serverDir, e);
                    }
                }
            });
            return requestStatus[0];
        }

        boolean isAlive() {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            boolean alive = this.sendRequest(NativeImageServer.byteStreamToByteConsumer(baos), NativeImageServer.byteStreamToByteConsumer(baos), SubstrateServerMessage.ServerCommand.GET_VERSION, new String[0]) == 0;
            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.verboseServer, "Server version response: " + new String(baos.toByteArray()));
            return alive;
        }

        void abortTask() {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            this.sendRequest(NativeImageServer.byteStreamToByteConsumer(baos), NativeImageServer.byteStreamToByteConsumer(baos), SubstrateServerMessage.ServerCommand.ABORT_BUILD, new String[0]);
            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.verboseServer, "Server abort response:" + new String(baos.toByteArray()));
        }

        synchronized void shutdown() {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            this.sendRequest(NativeImageServer.byteStreamToByteConsumer(baos), NativeImageServer.byteStreamToByteConsumer(baos), SubstrateServerMessage.ServerCommand.STOP_SERVER, new String[0]);
            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.verboseServer, "Server stop response:" + new String(baos.toByteArray()));
            long terminationTimeout = System.currentTimeMillis() + 20000L;
            long killTimeout = terminationTimeout + 40000L;
            long killedTimeout = killTimeout + 2000L;
            NativeImageServer.this.showVerboseMessage(NativeImageServer.this.verboseServer, "Waiting for " + this + " to shutdown");
            boolean sentSIGTERM = false;
            boolean sentSIGKILL = false;
            do {
                try {
                    Thread.sleep(500L);
                }
                catch (InterruptedException e) {
                    throw NativeImage.showError("Woke up from waiting for " + this + " to shutdown", e);
                }
                long now = System.currentTimeMillis();
                if (!sentSIGTERM && terminationTimeout < now) {
                    NativeImage.showWarning(this + " keeps responding to port " + this.port + " even after sending STOP_SERVER");
                    ProcessProperties.destroy((long)this.pid);
                    sentSIGTERM = true;
                    continue;
                }
                if (!sentSIGKILL && killTimeout < now) {
                    NativeImage.showWarning(this + " keeps responding to port " + this.port + " even after destroying");
                    ProcessProperties.destroyForcibly((long)this.pid);
                    sentSIGKILL = true;
                    continue;
                }
                if (killedTimeout >= now) continue;
                throw NativeImage.showError(this + " keeps responding to port " + this.port + " even after destroying forcefully");
            } while (this.isAlive());
            NativeImageServer.this.deleteAllFiles(this.serverDir);
        }

        String getServerInfo() {
            StringBuilder sb = new StringBuilder();
            sb.append(this.getClass().getName());
            sb.append("\nServerDir: ").append(this.serverDir);
            sb.append("\nRunning for: ").append(NativeImageServer.getDurationString(this.since));
            sb.append("\nLast build: ");
            if (this.since.equals(this.lastBuildRequest)) {
                sb.append("None");
            } else {
                sb.append(NativeImageServer.getDurationString(this.lastBuildRequest));
            }
            sb.append("\nPID: ").append(this.pid);
            sb.append("\nPort: ").append(this.port);
            sb.append("\nJavaArgs: ").append(String.join((CharSequence)" ", this.serverJavaArgs));
            sb.append("\nBootClasspath: ").append(this.serverBootClasspath.stream().map(Path::toString).collect(Collectors.joining(File.pathSeparator)));
            sb.append("\nClasspath: ").append(this.serverClasspath.stream().map(Path::toString).collect(Collectors.joining(File.pathSeparator)));
            return sb.append('\n').toString();
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append(this.getClass().getSimpleName());
            sb.append("(");
            sb.append("pid: ").append(this.pid);
            sb.append(", ");
            sb.append("port: ").append(this.port);
            sb.append(")");
            if (this.since.equals(this.lastBuildRequest)) {
                sb.append('*');
            }
            return sb.toString();
        }

        private String getLivenessInfo(boolean alive) {
            if (alive) {
                return " running for: " + NativeImageServer.getDurationString(this.since);
            }
            return " DEAD";
        }

        private String getLastBuildInfo() {
            if (this.lastBuildRequest.equals(this.since)) {
                return " <No builds>";
            }
            return " last build: " + NativeImageServer.getDurationString(this.lastBuildRequest);
        }
    }

    private static final class ServerInstanceError
    extends RuntimeException {
        private ServerInstanceError() {
        }
    }
}

