/*
 * Decompiled with CFR 0.152.
 */
package org.geolatte.geom.codec;

import org.geolatte.geom.ByteBuffer;
import org.geolatte.geom.ByteOrder;
import org.geolatte.geom.DimensionalFlag;
import org.geolatte.geom.Geometry;
import org.geolatte.geom.GeometryCollection;
import org.geolatte.geom.GeometryVisitor;
import org.geolatte.geom.LineString;
import org.geolatte.geom.LinearRing;
import org.geolatte.geom.Point;
import org.geolatte.geom.PointSequence;
import org.geolatte.geom.PolyHedralSurface;
import org.geolatte.geom.Polygon;
import org.geolatte.geom.codec.UnsupportedConversionException;
import org.geolatte.geom.codec.WkbGeometryType;

class PostgisWkbEncoder {
    PostgisWkbEncoder() {
    }

    public ByteBuffer encode(Geometry geom, ByteOrder wbo) {
        ByteBuffer output = ByteBuffer.allocate(this.calculateSize(geom, true));
        if (wbo != null) {
            output.setWKBByteOrder(wbo);
        }
        this.writeGeometry(geom, output);
        output.rewind();
        return output;
    }

    private void writeGeometry(Geometry geom, ByteBuffer output) {
        geom.accept(new WkbVisitor(output));
    }

    private int calculateSize(Geometry geom, boolean includeSrid) {
        int size = 5;
        if (geom.getSRID() > 0 && includeSrid) {
            size += 4;
        }
        if (geom instanceof GeometryCollection) {
            size += this.sizeOfGeometryCollection((GeometryCollection)geom);
        } else if (geom instanceof Polygon) {
            size += this.getPolygonSize((Polygon)geom);
        } else if (geom instanceof Point) {
            size += this.getPointByteSize(geom);
        } else if (geom instanceof PolyHedralSurface) {
            size += this.getPolyHedralSurfaceSize((PolyHedralSurface)geom);
        } else {
            size += 4;
            size += this.getPointByteSize(geom) * geom.getNumPoints();
        }
        return size;
    }

    private int getPointByteSize(Geometry geom) {
        return geom.getCoordinateDimension() * 8;
    }

    private int getPolyHedralSurfaceSize(PolyHedralSurface geom) {
        int size = 4;
        for (int i = 0; i < geom.getNumPatches(); ++i) {
            size += this.getPolygonSize(geom.getPatchN(i));
        }
        return size;
    }

    private int getPolygonSize(Polygon geom) {
        int size = 4;
        size += geom.isEmpty() ? 0 : 4 * (geom.getNumInteriorRing() + 1);
        return size += this.getPointByteSize(geom) * geom.getNumPoints();
    }

    private int sizeOfGeometryCollection(GeometryCollection collection) {
        int size = 4;
        for (Geometry g : collection) {
            size += this.calculateSize(g, false);
        }
        return size;
    }

    private static class WkbVisitor
    implements GeometryVisitor {
        private final ByteBuffer output;
        private boolean hasWrittenSrid = false;

        WkbVisitor(ByteBuffer byteBuffer) {
            this.output = byteBuffer;
        }

        @Override
        public void visit(Point geom) {
            this.writeByteOrder(this.output);
            DimensionalFlag dimension = DimensionalFlag.valueOf(geom.is3D(), geom.isMeasured());
            this.writeTypeCodeAndSrid(geom, dimension, this.output);
            this.writePoints(geom.getPoints(), geom.getCoordinateDimension(), this.output);
        }

        @Override
        public void visit(LineString geom) {
            this.writeByteOrder(this.output);
            DimensionalFlag dimension = DimensionalFlag.valueOf(geom.is3D(), geom.isMeasured());
            this.writeTypeCodeAndSrid(geom, dimension, this.output);
            this.output.putUInt(geom.getNumPoints());
            this.writePoints(geom.getPoints(), geom.getCoordinateDimension(), this.output);
        }

        @Override
        public void visit(Polygon geom) {
            this.writeByteOrder(this.output);
            DimensionalFlag dimension = DimensionalFlag.valueOf(geom.is3D(), geom.isMeasured());
            this.writeTypeCodeAndSrid(geom, dimension, this.output);
            this.writeNumRings(geom, this.output);
            for (LinearRing ring : geom) {
                this.writeRing(ring);
            }
        }

        @Override
        public void visit(PolyHedralSurface geom) {
            this.writeByteOrder(this.output);
            DimensionalFlag dimension = DimensionalFlag.valueOf(geom.is3D(), geom.isMeasured());
            this.writeTypeCodeAndSrid(geom, dimension, this.output);
            this.output.putUInt(geom.getNumPatches());
            for (Polygon pg : geom) {
                pg.accept(this);
            }
        }

        @Override
        public void visit(GeometryCollection geom) {
            this.writeByteOrder(this.output);
            DimensionalFlag dimension = DimensionalFlag.valueOf(geom.is3D(), geom.isMeasured());
            this.writeTypeCodeAndSrid(geom, dimension, this.output);
            this.output.putUInt(geom.getNumGeometries());
        }

        @Override
        public void visit(LinearRing geom) {
            this.writeByteOrder(this.output);
            DimensionalFlag dimension = DimensionalFlag.valueOf(geom.is3D(), geom.isMeasured());
            this.writeTypeCodeAndSrid(geom, dimension, this.output);
            this.writeRing(geom);
        }

        private void writeRing(LinearRing geom) {
            this.output.putUInt(geom.getNumPoints());
            this.writePoints(geom.getPoints(), geom.getCoordinateDimension(), this.output);
        }

        private void writeNumRings(Polygon geom, ByteBuffer byteBuffer) {
            byteBuffer.putUInt(geom.isEmpty() ? 0L : (long)(geom.getNumInteriorRing() + 1));
        }

        protected void writePoint(double[] coordinates, ByteBuffer output) {
            for (double coordinate : coordinates) {
                output.putDouble(coordinate);
            }
        }

        protected void writePoints(PointSequence points, int coordinateDimension, ByteBuffer output) {
            double[] coordinates = new double[coordinateDimension];
            for (int i = 0; i < points.size(); ++i) {
                points.getCoordinates(coordinates, i);
                this.writePoint(coordinates, output);
            }
        }

        protected void writeByteOrder(ByteBuffer output) {
            output.put(output.getWKBByteOrder().byteValue());
        }

        protected void writeTypeCodeAndSrid(Geometry geometry, DimensionalFlag dimension, ByteBuffer output) {
            boolean hasSrid;
            int typeCode = this.getGeometryType(geometry);
            boolean bl = hasSrid = geometry.getSRID() > 0;
            if (hasSrid && !this.hasWrittenSrid) {
                typeCode |= 0x20000000;
            }
            if (dimension.isMeasured()) {
                typeCode |= 0x40000000;
            }
            if (dimension.is3D()) {
                typeCode |= Integer.MIN_VALUE;
            }
            output.putUInt(typeCode);
            if (hasSrid && !this.hasWrittenSrid) {
                output.putInt(geometry.getSRID());
                this.hasWrittenSrid = true;
            }
        }

        protected int getGeometryType(Geometry geometry) {
            WkbGeometryType type = WkbGeometryType.forClass(geometry.getClass());
            if (type == null) {
                throw new UnsupportedConversionException(String.format("Can't convert geometries of type %s", geometry.getClass().getCanonicalName()));
            }
            return type.getTypeCode();
        }
    }
}

