/*
 * Decompiled with CFR 0.152.
 */
package io.fury.memory;

import com.google.common.base.Preconditions;
import io.fury.memory.BoundsChecking;
import io.fury.util.Platform;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.ReadOnlyBufferException;
import java.util.Arrays;
import sun.misc.Unsafe;

public final class MemoryBuffer {
    private static final Unsafe UNSAFE = Platform.UNSAFE;
    private static final long BYTE_ARRAY_BASE_OFFSET = UNSAFE.arrayBaseOffset(byte[].class);
    private static final boolean LITTLE_ENDIAN = ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN;
    private byte[] heapMemory;
    private int heapOffset;
    private ByteBuffer offHeapBuffer;
    private long address;
    private long addressLimit;
    private int size;
    private int readerIndex;
    private int writerIndex;
    private static final long HALF_MAX_INT_VALUE = 0x3FFFFFFFL;
    private static final long HALF_MIN_INT_VALUE = -1073741824L;
    private static final byte BIG_LONG_FLAG = 1;

    private MemoryBuffer(byte[] buffer, int offset, int length) {
        Preconditions.checkArgument((offset >= 0 && length >= 0 ? 1 : 0) != 0);
        if (offset + length > buffer.length) {
            throw new IllegalArgumentException(String.format("%d exceeds buffer size %d", offset + length, buffer.length));
        }
        this.initHeapBuffer(buffer, offset, length);
    }

    private void initHeapBuffer(byte[] buffer, int offset, int length) {
        long startPos;
        if (buffer == null) {
            throw new NullPointerException("buffer");
        }
        this.heapMemory = buffer;
        this.heapOffset = offset;
        this.address = startPos = BYTE_ARRAY_BASE_OFFSET + (long)offset;
        this.size = length;
        this.addressLimit = startPos + (long)length;
    }

    private MemoryBuffer(long offHeapAddress, int size, ByteBuffer offHeapBuffer) {
        this.offHeapBuffer = offHeapBuffer;
        if (offHeapAddress <= 0L) {
            throw new IllegalArgumentException("negative pointer or size");
        }
        if (offHeapAddress >= 9223372034707292160L) {
            throw new IllegalArgumentException("Buffer initialized with too large address: " + offHeapAddress + " ; Max allowed address is " + 0x7FFFFFFF7FFFFFFFL);
        }
        this.heapMemory = null;
        this.address = offHeapAddress;
        this.addressLimit = this.address + (long)size;
        this.size = size;
    }

    public int size() {
        return this.size;
    }

    public boolean isOffHeap() {
        return this.heapMemory == null;
    }

    public byte[] getHeapMemory() {
        return this.heapMemory;
    }

    public ByteBuffer getOffHeapBuffer() {
        if (this.offHeapBuffer != null) {
            return this.offHeapBuffer;
        }
        throw new IllegalStateException("Memory buffer does not represent off heap ByteBuffer");
    }

    public byte[] getArray() {
        if (this.heapMemory != null) {
            return this.heapMemory;
        }
        throw new IllegalStateException("Memory buffer does not represent heap memory");
    }

    public long getAddress() {
        if (this.heapMemory == null) {
            return this.address;
        }
        throw new IllegalStateException("Memory buffer does not represent off heap memory");
    }

    public long getUnsafeAddress() {
        return this.address;
    }

    private void checkPosition(long index, long pos, long length) {
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && (index < 0L || pos > this.addressLimit - length)) {
            throw new IndexOutOfBoundsException();
        }
    }

    public static byte unsafeGet(Object o, long offset) {
        return UNSAFE.getByte(o, offset);
    }

    public byte unsafeGet(int index) {
        long pos = this.address + (long)index;
        return UNSAFE.getByte(this.heapMemory, pos);
    }

    public byte get(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 1L);
        return UNSAFE.getByte(this.heapMemory, pos);
    }

    public void get(int index, byte[] dst) {
        this.get(index, dst, 0, dst.length);
    }

    public void get(int index, byte[] dst, int offset, int length) {
        if ((offset | length | offset + length | dst.length - (offset + length)) < 0) {
            throw new IndexOutOfBoundsException();
        }
        long pos = this.address + (long)index;
        if (index < 0 || pos > this.addressLimit - (long)length) {
            throw new IndexOutOfBoundsException();
        }
        long arrayAddress = BYTE_ARRAY_BASE_OFFSET + (long)offset;
        Platform.copyMemory(this.heapMemory, pos, dst, arrayAddress, length);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void get(int offset, ByteBuffer target, int numBytes) {
        if ((offset | numBytes | offset + numBytes) < 0) {
            throw new IndexOutOfBoundsException();
        }
        int targetOffset = target.position();
        int remaining = target.remaining();
        if (remaining < numBytes) {
            throw new BufferOverflowException();
        }
        if (target.isDirect()) {
            if (target.isReadOnly()) {
                throw new ReadOnlyBufferException();
            }
            long targetPointer = Platform.getAddress(target) + (long)targetOffset;
            long sourcePointer = this.address + (long)offset;
            if (sourcePointer > this.addressLimit - (long)numBytes) throw new IndexOutOfBoundsException();
            Platform.copyMemory(this.heapMemory, sourcePointer, null, targetPointer, numBytes);
            target.position(targetOffset + numBytes);
            return;
        } else if (target.hasArray()) {
            this.get(offset, target.array(), targetOffset + target.arrayOffset(), numBytes);
            target.position(targetOffset + numBytes);
            return;
        } else {
            while (target.hasRemaining()) {
                target.put(this.get(offset++));
            }
        }
    }

    public static void unsafePut(Object o, long offset, byte b) {
        UNSAFE.putByte(o, offset, b);
    }

    public void unsafePut(int index, byte b) {
        long pos = this.address + (long)index;
        UNSAFE.putByte(this.heapMemory, pos, b);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void put(int offset, ByteBuffer source, int numBytes) {
        if ((offset | numBytes | offset + numBytes) < 0) {
            throw new IndexOutOfBoundsException();
        }
        int sourceOffset = source.position();
        int remaining = source.remaining();
        if (remaining < numBytes) {
            throw new BufferUnderflowException();
        }
        if (source.isDirect()) {
            long sourcePointer = Platform.getAddress(source) + (long)sourceOffset;
            long targetPointer = this.address + (long)offset;
            if (targetPointer > this.addressLimit - (long)numBytes) throw new IndexOutOfBoundsException();
            Platform.copyMemory(null, sourcePointer, this.heapMemory, targetPointer, numBytes);
            source.position(sourceOffset + numBytes);
            return;
        } else if (source.hasArray()) {
            this.put(offset, source.array(), sourceOffset + source.arrayOffset(), numBytes);
            source.position(sourceOffset + numBytes);
            return;
        } else {
            while (source.hasRemaining()) {
                this.put(offset++, source.get());
            }
        }
    }

    public void put(int index, byte b) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 1L);
        UNSAFE.putByte(this.heapMemory, pos, b);
    }

    public void put(int index, byte[] src) {
        this.put(index, src, 0, src.length);
    }

    public void put(int index, byte[] src, int offset, int length) {
        if ((offset | length | offset + length | src.length - (offset + length)) < 0) {
            throw new IndexOutOfBoundsException();
        }
        long pos = this.address + (long)index;
        if (index < 0 || pos > this.addressLimit - (long)length) {
            throw new IndexOutOfBoundsException();
        }
        long arrayAddress = BYTE_ARRAY_BASE_OFFSET + (long)offset;
        Platform.copyMemory(src, arrayAddress, this.heapMemory, pos, length);
    }

    public static boolean unsafeGetBoolean(Object o, long offset) {
        return UNSAFE.getBoolean(o, offset);
    }

    public boolean unsafeGetBoolean(int index) {
        long pos = this.address + (long)index;
        return UNSAFE.getByte(this.heapMemory, pos) != 0;
    }

    public boolean getBoolean(int index) {
        return this.get(index) != 0;
    }

    public void putBoolean(int index, boolean value) {
        this.put(index, (byte)(value ? 1 : 0));
    }

    public void unsafePutBoolean(int index, boolean value) {
        this.unsafePut(index, (byte)(value ? 1 : 0));
    }

    public static void unsafePutBoolean(Object o, long offset, boolean value) {
        UNSAFE.putBoolean(o, offset, value);
    }

    public char getCharN(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 2L);
        return UNSAFE.getChar(this.heapMemory, pos);
    }

    public char getCharB(int index) {
        if (LITTLE_ENDIAN) {
            return Character.reverseBytes(this.getCharN(index));
        }
        return this.getCharN(index);
    }

    public char getChar(int index) {
        if (LITTLE_ENDIAN) {
            return this.getCharN(index);
        }
        return Character.reverseBytes(this.getCharN(index));
    }

    public void putCharN(int index, char value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 2L);
        UNSAFE.putChar(this.heapMemory, pos, value);
    }

    public void putChar(int index, char value) {
        if (LITTLE_ENDIAN) {
            this.putCharN(index, value);
        } else {
            this.putCharN(index, Character.reverseBytes(value));
        }
    }

    public char unsafeGetCharN(int index) {
        long pos = this.address + (long)index;
        return UNSAFE.getChar(this.heapMemory, pos);
    }

    public char unsafeGetChar(int index) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            return UNSAFE.getChar(this.heapMemory, pos);
        }
        return Character.reverseBytes(UNSAFE.getChar(this.heapMemory, pos));
    }

    public static char unsafeGetChar(Object o, long pos) {
        if (LITTLE_ENDIAN) {
            return UNSAFE.getChar(o, pos);
        }
        return Character.reverseBytes(UNSAFE.getChar(o, pos));
    }

    public void unsafePutCharN(int index, char value) {
        long pos = this.address + (long)index;
        UNSAFE.putChar(this.heapMemory, pos, value);
    }

    public void unsafePutChar(int index, char value) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            UNSAFE.putChar(this.heapMemory, pos, value);
        } else {
            UNSAFE.putChar(this.heapMemory, pos, Character.reverseBytes(value));
        }
    }

    public static void unsafePutChar(Object o, long offset, char value) {
        if (LITTLE_ENDIAN) {
            UNSAFE.putChar(o, offset, value);
        } else {
            UNSAFE.putChar(o, offset, Character.reverseBytes(value));
        }
    }

    public short getShortN(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 2L);
        return UNSAFE.getShort(this.heapMemory, pos);
    }

    public static short getShortB(byte[] b, int off) {
        return (short)((b[off + 1] & 0xFF) + (b[off] << 8));
    }

    public short getShortB(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 2L);
        if (LITTLE_ENDIAN) {
            return Short.reverseBytes(UNSAFE.getShort(this.heapMemory, pos));
        }
        return UNSAFE.getShort(this.heapMemory, pos);
    }

    public short getShort(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 2L);
        if (LITTLE_ENDIAN) {
            return UNSAFE.getShort(this.heapMemory, pos);
        }
        return Short.reverseBytes(UNSAFE.getShort(this.heapMemory, pos));
    }

    public void putShortN(int index, short value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 2L);
        UNSAFE.putShort(this.heapMemory, pos, value);
    }

    public void putShort(int index, short value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 2L);
        if (LITTLE_ENDIAN) {
            UNSAFE.putShort(this.heapMemory, pos, value);
        } else {
            UNSAFE.putShort(this.heapMemory, pos, Short.reverseBytes(value));
        }
    }

    public short unsafeGetShortN(int index) {
        long pos = this.address + (long)index;
        return UNSAFE.getShort(this.heapMemory, pos);
    }

    public short unsafeGetShort(int index) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            return UNSAFE.getShort(this.heapMemory, pos);
        }
        return Short.reverseBytes(UNSAFE.getShort(this.heapMemory, pos));
    }

    public static short unsafeGetShort(Object o, long offset) {
        if (LITTLE_ENDIAN) {
            return UNSAFE.getShort(o, offset);
        }
        return Short.reverseBytes(UNSAFE.getShort(o, offset));
    }

    public void unsafePutShortN(int index, short value) {
        long pos = this.address + (long)index;
        UNSAFE.putShort(this.heapMemory, pos, value);
    }

    public void unsafePutShort(int index, short value) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            UNSAFE.putShort(this.heapMemory, pos, value);
        } else {
            UNSAFE.putShort(this.heapMemory, pos, Short.reverseBytes(value));
        }
    }

    public static void unsafePutShort(Object o, long pos, short value) {
        if (LITTLE_ENDIAN) {
            UNSAFE.putShort(o, pos, value);
        } else {
            UNSAFE.putShort(o, pos, Short.reverseBytes(value));
        }
    }

    public int getIntN(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 4L);
        return UNSAFE.getInt(this.heapMemory, pos);
    }

    public int getInt(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 4L);
        if (LITTLE_ENDIAN) {
            return UNSAFE.getInt(this.heapMemory, pos);
        }
        return Integer.reverseBytes(UNSAFE.getInt(this.heapMemory, pos));
    }

    public void putIntN(int index, int value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 4L);
        UNSAFE.putInt(this.heapMemory, pos, value);
    }

    public void putInt(int index, int value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 4L);
        if (LITTLE_ENDIAN) {
            UNSAFE.putInt(this.heapMemory, pos, value);
        } else {
            UNSAFE.putInt(this.heapMemory, pos, Integer.reverseBytes(value));
        }
    }

    public int unsafeGetIntN(int index) {
        long pos = this.address + (long)index;
        return UNSAFE.getInt(this.heapMemory, pos);
    }

    public int unsafeGetInt(int index) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            return UNSAFE.getInt(this.heapMemory, pos);
        }
        return Integer.reverseBytes(UNSAFE.getInt(this.heapMemory, pos));
    }

    public static int unsafeGetInt(Object o, long pos) {
        if (LITTLE_ENDIAN) {
            return UNSAFE.getInt(o, pos);
        }
        return Integer.reverseBytes(UNSAFE.getInt(o, pos));
    }

    public void unsafePutIntN(int index, int value) {
        long pos = this.address + (long)index;
        UNSAFE.putInt(this.heapMemory, pos, value);
    }

    public void unsafePutInt(int index, int value) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            UNSAFE.putInt(this.heapMemory, pos, value);
        } else {
            UNSAFE.putInt(this.heapMemory, pos, Integer.reverseBytes(value));
        }
    }

    public static void unsafePutInt(Object o, long pos, int value) {
        if (LITTLE_ENDIAN) {
            UNSAFE.putInt(o, pos, value);
        } else {
            UNSAFE.putInt(o, pos, Integer.reverseBytes(value));
        }
    }

    public long getLongN(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        return UNSAFE.getLong(this.heapMemory, pos);
    }

    public long getLong(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        if (LITTLE_ENDIAN) {
            return UNSAFE.getLong(this.heapMemory, pos);
        }
        return Long.reverseBytes(UNSAFE.getLong(this.heapMemory, pos));
    }

    public long getLongB(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        if (LITTLE_ENDIAN) {
            return Long.reverseBytes(UNSAFE.getLong(this.heapMemory, pos));
        }
        return UNSAFE.getLong(this.heapMemory, pos);
    }

    public void putLongN(int index, long value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        UNSAFE.putLong(this.heapMemory, pos, value);
    }

    public void putLong(int index, long value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        if (LITTLE_ENDIAN) {
            UNSAFE.putLong(this.heapMemory, pos, value);
        } else {
            UNSAFE.putLong(this.heapMemory, pos, Long.reverseBytes(value));
        }
    }

    public void putLongB(int index, long value) {
        if (LITTLE_ENDIAN) {
            this.putLongN(index, Long.reverseBytes(value));
        } else {
            this.putLongN(index, value);
        }
    }

    public long unsafeGetLongN(int index) {
        long pos = this.address + (long)index;
        return UNSAFE.getLong(this.heapMemory, pos);
    }

    public long unsafeGetLong(int index) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            return UNSAFE.getLong(this.heapMemory, pos);
        }
        return Long.reverseBytes(UNSAFE.getLong(this.heapMemory, pos));
    }

    public static long unsafeGetLong(Object o, long pos) {
        if (LITTLE_ENDIAN) {
            return UNSAFE.getLong(o, pos);
        }
        return Long.reverseBytes(UNSAFE.getLong(o, pos));
    }

    public void unsafePutLongN(int index, long value) {
        long pos = this.address + (long)index;
        UNSAFE.putLong(this.heapMemory, pos, value);
    }

    public void unsafePutLong(int index, long value) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            UNSAFE.putLong(this.heapMemory, pos, value);
        } else {
            UNSAFE.putLong(this.heapMemory, pos, Long.reverseBytes(value));
        }
    }

    public static void unsafePutLong(Object o, long pos, long value) {
        if (LITTLE_ENDIAN) {
            UNSAFE.putLong(o, pos, value);
        } else {
            UNSAFE.putLong(o, pos, Long.reverseBytes(value));
        }
    }

    public float getFloatN(int index) {
        return Float.intBitsToFloat(this.getIntN(index));
    }

    public float getFloat(int index) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            return Float.intBitsToFloat(UNSAFE.getInt(this.heapMemory, pos));
        }
        return Float.intBitsToFloat(Integer.reverseBytes(UNSAFE.getInt(this.heapMemory, pos)));
    }

    public void putFloatN(int index, float value) {
        this.putIntN(index, Float.floatToRawIntBits(value));
    }

    public void putFloat(int index, float value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 4L);
        if (LITTLE_ENDIAN) {
            UNSAFE.putInt(this.heapMemory, pos, Float.floatToRawIntBits(value));
        } else {
            UNSAFE.putInt(this.heapMemory, pos, Integer.reverseBytes(Float.floatToRawIntBits(value)));
        }
    }

    public float unsafeGetFloatN(int index) {
        return Float.intBitsToFloat(this.unsafeGetIntN(index));
    }

    public float unsafeGetFloat(int index) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            return Float.intBitsToFloat(UNSAFE.getInt(this.heapMemory, pos));
        }
        return Float.intBitsToFloat(Integer.reverseBytes(UNSAFE.getInt(this.heapMemory, pos)));
    }

    public static float unsafeGetFloat(Object o, long pos) {
        if (LITTLE_ENDIAN) {
            return Float.intBitsToFloat(UNSAFE.getInt(o, pos));
        }
        return Float.intBitsToFloat(Integer.reverseBytes(UNSAFE.getInt(o, pos)));
    }

    public void unsafePutFloatN(int index, float value) {
        this.unsafePutIntN(index, Float.floatToRawIntBits(value));
    }

    public void unsafePutFloat(int index, float value) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            UNSAFE.putInt(this.heapMemory, pos, Float.floatToRawIntBits(value));
        } else {
            UNSAFE.putInt(this.heapMemory, pos, Integer.reverseBytes(Float.floatToRawIntBits(value)));
        }
    }

    public static void unsafePutFloat(Object o, long pos, float value) {
        if (LITTLE_ENDIAN) {
            UNSAFE.putInt(o, pos, Float.floatToRawIntBits(value));
        } else {
            UNSAFE.putInt(o, pos, Integer.reverseBytes(Float.floatToRawIntBits(value)));
        }
    }

    public double getDoubleN(int index) {
        return Double.longBitsToDouble(this.getLongN(index));
    }

    public double getDouble(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        if (LITTLE_ENDIAN) {
            return Double.longBitsToDouble(UNSAFE.getLong(this.heapMemory, pos));
        }
        return Double.longBitsToDouble(Long.reverseBytes(UNSAFE.getLong(this.heapMemory, pos)));
    }

    public void putDoubleN(int index, double value) {
        this.putLongN(index, Double.doubleToRawLongBits(value));
    }

    public void putDouble(int index, double value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        if (LITTLE_ENDIAN) {
            UNSAFE.putLong(this.heapMemory, pos, Double.doubleToRawLongBits(value));
        } else {
            UNSAFE.putLong(this.heapMemory, pos, Long.reverseBytes(Double.doubleToRawLongBits(value)));
        }
    }

    public double unsafeGetDoubleN(int index) {
        return Double.longBitsToDouble(this.unsafeGetLongN(index));
    }

    public double unsafeGetDouble(int index) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            return Double.longBitsToDouble(UNSAFE.getLong(this.heapMemory, pos));
        }
        return Double.longBitsToDouble(Long.reverseBytes(UNSAFE.getLong(this.heapMemory, pos)));
    }

    public static double unsafeGetDouble(Object o, long pos) {
        if (LITTLE_ENDIAN) {
            return Double.longBitsToDouble(UNSAFE.getLong(o, pos));
        }
        return Double.longBitsToDouble(Long.reverseBytes(UNSAFE.getLong(o, pos)));
    }

    public void unsafePutDoubleN(int index, double value) {
        this.unsafePutLongN(index, Double.doubleToRawLongBits(value));
    }

    public void unsafePutDouble(int index, double value) {
        long pos = this.address + (long)index;
        if (LITTLE_ENDIAN) {
            UNSAFE.putLong(this.heapMemory, pos, Double.doubleToRawLongBits(value));
        } else {
            UNSAFE.putLong(this.heapMemory, pos, Long.reverseBytes(Double.doubleToRawLongBits(value)));
        }
    }

    public static void unsafePutDouble(Object o, long pos, double value) {
        if (LITTLE_ENDIAN) {
            UNSAFE.putLong(o, pos, Double.doubleToRawLongBits(value));
        } else {
            UNSAFE.putLong(o, pos, Long.reverseBytes(Double.doubleToRawLongBits(value)));
        }
    }

    public int readerIndex() {
        return this.readerIndex;
    }

    public MemoryBuffer readerIndex(int readerIndex) {
        if (readerIndex < 0 || readerIndex > this.size) {
            throw new IndexOutOfBoundsException(String.format("readerIndex: %d (expected: 0 <= readerIndex <= size(%d))", readerIndex, this.size));
        }
        this.readerIndex = readerIndex;
        return this;
    }

    public int unsafeHeapReaderIndex() {
        return this.readerIndex + this.heapOffset;
    }

    public void increaseReaderIndexUnsafe(int diff) {
        this.readerIndex += diff;
    }

    public void increaseReaderIndex(int diff) {
        int readerIdx = this.readerIndex + diff;
        if (readerIdx < 0 || readerIdx > this.size) {
            throw new IndexOutOfBoundsException(String.format("readerIndex: %d (expected: 0 <= readerIndex <= size(%d))", readerIdx, this.size));
        }
        this.readerIndex = readerIdx;
    }

    public long getUnsafeReaderAddress() {
        return this.address + (long)this.readerIndex;
    }

    public int remaining() {
        return this.size - this.readerIndex;
    }

    public int writerIndex() {
        return this.writerIndex;
    }

    public void writerIndex(int writerIndex) {
        if (writerIndex < 0 || writerIndex > this.size) {
            throw new IndexOutOfBoundsException(String.format("writerIndex: %d (expected: 0 <= writerIndex <= size(%d))", writerIndex, this.size));
        }
        this.writerIndex = writerIndex;
    }

    public void unsafeWriterIndex(int writerIndex) {
        this.writerIndex = writerIndex;
    }

    public int unsafeHeapWriterIndex() {
        return this.writerIndex + this.heapOffset;
    }

    public long getUnsafeWriterAddress() {
        return this.address + (long)this.writerIndex;
    }

    public void increaseWriterIndexUnsafe(int diff) {
        this.writerIndex += diff;
    }

    public void increaseWriterIndex(int diff) {
        int writerIdx = this.writerIndex + diff;
        this.ensure(writerIdx);
        this.writerIndex = writerIdx;
    }

    public void writeBoolean(boolean value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 1;
        this.ensure(newIdx);
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(value ? 1 : 0));
        this.writerIndex = newIdx;
    }

    public void unsafeWriteByte(byte value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 1;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, value);
        this.writerIndex = newIdx;
    }

    public void writeByte(byte value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 1;
        this.ensure(newIdx);
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, value);
        this.writerIndex = newIdx;
    }

    public void writeByte(int value) {
        this.writeByte((byte)value);
    }

    public void writeChar(char value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 2;
        this.ensure(newIdx);
        long pos = this.address + (long)writerIdx;
        if (LITTLE_ENDIAN) {
            UNSAFE.putChar(this.heapMemory, pos, value);
        } else {
            UNSAFE.putChar(this.heapMemory, pos, Character.reverseBytes(value));
        }
        this.writerIndex = newIdx;
    }

    public void unsafeWriteShort(short value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 2;
        long pos = this.address + (long)writerIdx;
        if (LITTLE_ENDIAN) {
            UNSAFE.putShort(this.heapMemory, pos, value);
        } else {
            UNSAFE.putShort(this.heapMemory, pos, Short.reverseBytes(value));
        }
        this.writerIndex = newIdx;
    }

    public void writeShort(short value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 2;
        this.ensure(newIdx);
        long pos = this.address + (long)writerIdx;
        if (LITTLE_ENDIAN) {
            UNSAFE.putShort(this.heapMemory, pos, value);
        } else {
            UNSAFE.putShort(this.heapMemory, pos, Short.reverseBytes(value));
        }
        this.writerIndex = newIdx;
    }

    public void writeInt(int value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 4;
        this.ensure(newIdx);
        long pos = this.address + (long)writerIdx;
        if (LITTLE_ENDIAN) {
            UNSAFE.putInt(this.heapMemory, pos, value);
        } else {
            UNSAFE.putInt(this.heapMemory, pos, Integer.reverseBytes(value));
        }
        this.writerIndex = newIdx;
    }

    public void unsafeWriteInt(int value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 4;
        long pos = this.address + (long)writerIdx;
        if (LITTLE_ENDIAN) {
            UNSAFE.putInt(this.heapMemory, pos, value);
        } else {
            UNSAFE.putInt(this.heapMemory, pos, Integer.reverseBytes(value));
        }
        this.writerIndex = newIdx;
    }

    public void writeLong(long value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 8;
        this.ensure(newIdx);
        long pos = this.address + (long)writerIdx;
        if (LITTLE_ENDIAN) {
            UNSAFE.putLong(this.heapMemory, pos, value);
        } else {
            UNSAFE.putLong(this.heapMemory, pos, Long.reverseBytes(value));
        }
        this.writerIndex = newIdx;
    }

    public void unsafeWriteLong(long value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 8;
        long pos = this.address + (long)writerIdx;
        if (LITTLE_ENDIAN) {
            UNSAFE.putLong(this.heapMemory, pos, value);
        } else {
            UNSAFE.putLong(this.heapMemory, pos, Long.reverseBytes(value));
        }
        this.writerIndex = newIdx;
    }

    public void writeFloat(float value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 4;
        this.ensure(newIdx);
        long pos = this.address + (long)writerIdx;
        if (LITTLE_ENDIAN) {
            UNSAFE.putInt(this.heapMemory, pos, Float.floatToRawIntBits(value));
        } else {
            UNSAFE.putInt(this.heapMemory, pos, Integer.reverseBytes(Float.floatToRawIntBits(value)));
        }
        this.writerIndex = newIdx;
    }

    public void writeDouble(double value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 8;
        this.ensure(newIdx);
        long pos = this.address + (long)writerIdx;
        if (LITTLE_ENDIAN) {
            UNSAFE.putLong(this.heapMemory, pos, Double.doubleToRawLongBits(value));
        } else {
            UNSAFE.putLong(this.heapMemory, pos, Long.reverseBytes(Double.doubleToRawLongBits(value)));
        }
        this.writerIndex = newIdx;
    }

    public int writeVarInt(int v) {
        this.ensure(this.writerIndex + 8);
        return this.unsafeWriteVarInt(v);
    }

    public int writePositiveVarInt(int v) {
        this.ensure(this.writerIndex + 8);
        return this.unsafeWritePositiveVarInt(v);
    }

    public int unsafeWriteVarInt(int v) {
        v = v << 1 ^ v >> 31;
        return this.unsafeWritePositiveVarInt(v);
    }

    public int readVarInt() {
        int r = this.readPositiveVarInt();
        return r >>> 1 ^ -(r & 1);
    }

    public int unsafeWritePositiveVarInt(int v) {
        long value = v;
        int writerIndex = this.writerIndex;
        long varInt = value & 0x7FL;
        if ((value >>>= 7) == 0L) {
            UNSAFE.putByte(this.heapMemory, this.address + (long)writerIndex, (byte)varInt);
            this.writerIndex = writerIndex + 1;
            return 1;
        }
        varInt |= 0x80L;
        varInt |= (value & 0x7FL) << 8;
        if ((value >>>= 7) == 0L) {
            this.unsafePutInt(writerIndex, (int)varInt);
            this.writerIndex = writerIndex + 2;
            return 2;
        }
        varInt |= 0x8000L;
        varInt |= (value & 0x7FL) << 16;
        if ((value >>>= 7) == 0L) {
            this.unsafePutInt(writerIndex, (int)varInt);
            this.writerIndex = writerIndex + 3;
            return 3;
        }
        varInt |= 0x800000L;
        varInt |= (value & 0x7FL) << 24;
        if ((value >>>= 7) == 0L) {
            this.unsafePutInt(writerIndex, (int)varInt);
            this.writerIndex = writerIndex + 4;
            return 4;
        }
        varInt |= 0x80000000L;
        varInt |= (value & 0x7FL) << 32;
        this.unsafePutLong(writerIndex, varInt &= 0xFFFFFFFFFL);
        this.writerIndex = writerIndex + 5;
        return 5;
    }

    public int unsafePutPositiveVarInt(int index, int v) {
        long value = v;
        long varInt = value & 0x7FL;
        if ((value >>>= 7) == 0L) {
            UNSAFE.putByte(this.heapMemory, this.address + (long)index, (byte)varInt);
            return 1;
        }
        varInt |= 0x80L;
        varInt |= (value & 0x7FL) << 8;
        if ((value >>>= 7) == 0L) {
            this.unsafePutInt(index, (int)varInt);
            return 2;
        }
        varInt |= 0x8000L;
        varInt |= (value & 0x7FL) << 16;
        if ((value >>>= 7) == 0L) {
            this.unsafePutInt(index, (int)varInt);
            return 3;
        }
        varInt |= 0x800000L;
        varInt |= (value & 0x7FL) << 24;
        if ((value >>>= 7) == 0L) {
            this.unsafePutInt(index, (int)varInt);
            return 4;
        }
        varInt |= 0x80000000L;
        varInt |= (value & 0x7FL) << 32;
        this.unsafePutLong(index, varInt &= 0xFFFFFFFFFL);
        return 5;
    }

    public int readPositiveVarInt() {
        int readIdx = this.readerIndex;
        if (this.size - readIdx < 5) {
            return this.readPositiveVarIntSlow();
        }
        int fourByteValue = this.unsafeGetInt(readIdx);
        int b = fourByteValue & 0xFF;
        ++readIdx;
        int result = b & 0x7F;
        if ((b & 0x80) != 0) {
            ++readIdx;
            b = fourByteValue >>> 8 & 0xFF;
            result |= (b & 0x7F) << 7;
            if ((b & 0x80) != 0) {
                ++readIdx;
                b = fourByteValue >>> 16 & 0xFF;
                result |= (b & 0x7F) << 14;
                if ((b & 0x80) != 0) {
                    ++readIdx;
                    b = fourByteValue >>> 24 & 0xFF;
                    result |= (b & 0x7F) << 21;
                    if ((b & 0x80) != 0) {
                        b = this.unsafeGet(readIdx++);
                        result |= (b & 0x7F) << 28;
                    }
                }
            }
        }
        this.readerIndex = readIdx;
        return result;
    }

    private int readPositiveVarIntSlow() {
        byte b = this.readByte();
        int result = b & 0x7F;
        if ((b & 0x80) != 0) {
            b = this.readByte();
            result |= (b & 0x7F) << 7;
            if ((b & 0x80) != 0) {
                b = this.readByte();
                result |= (b & 0x7F) << 14;
                if ((b & 0x80) != 0) {
                    b = this.readByte();
                    result |= (b & 0x7F) << 21;
                    if ((b & 0x80) != 0) {
                        b = this.readByte();
                        result |= (b & 0x7F) << 28;
                    }
                }
            }
        }
        return result;
    }

    public int writePositiveVarIntAligned(int value) {
        if (value >>> 6 == 0) {
            return this.writePositiveVarIntAligned1(value);
        }
        if (value >>> 12 == 0) {
            return this.writePositiveVarIntAligned2(value);
        }
        if (value >>> 18 == 0) {
            return this.writePositiveVarIntAligned3(value);
        }
        if (value >>> 24 == 0) {
            return this.writePositiveVarIntAligned4(value);
        }
        if (value >>> 30 == 0) {
            return this.writePositiveVarIntAligned5(value);
        }
        return this.writePositiveVarIntAligned6(value);
    }

    private int writePositiveVarIntAligned1(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 5);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        if (numPaddingBytes == 1) {
            UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x40));
            this.writerIndex = writerIdx + 1;
            return 1;
        }
        UNSAFE.putByte(this.heapMemory, pos, (byte)first);
        UNSAFE.putInt(this.heapMemory, pos + 1L, 0);
        UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes - 1L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes;
        return numPaddingBytes;
    }

    private int writePositiveVarIntAligned2(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 6);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x80));
        if (numPaddingBytes == 2) {
            UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6 | 0x40));
            this.writerIndex = writerIdx + 2;
            return 2;
        }
        UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6));
        UNSAFE.putInt(this.heapMemory, pos + 2L, 0);
        if (numPaddingBytes > 2) {
            UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes - 1L, (byte)64);
            this.writerIndex = writerIdx + numPaddingBytes;
            return numPaddingBytes;
        }
        UNSAFE.putByte(this.heapMemory, pos + 4L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes + 4;
        return numPaddingBytes + 4;
    }

    private int writePositiveVarIntAligned3(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 7);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6 | 0x80));
        if (numPaddingBytes == 3) {
            UNSAFE.putByte(this.heapMemory, pos + 2L, (byte)(value >>> 12 | 0x40));
            this.writerIndex = writerIdx + 3;
            return 3;
        }
        UNSAFE.putByte(this.heapMemory, pos + 2L, (byte)(value >>> 12));
        UNSAFE.putInt(this.heapMemory, pos + 3L, 0);
        if (numPaddingBytes == 4) {
            UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes - 1L, (byte)64);
            this.writerIndex = writerIdx + numPaddingBytes;
            return numPaddingBytes;
        }
        UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes + 3L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes + 4;
        return numPaddingBytes + 4;
    }

    private int writePositiveVarIntAligned4(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 8);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 2L, (byte)(value >>> 12 | 0x80));
        if (numPaddingBytes == 4) {
            UNSAFE.putByte(this.heapMemory, pos + 3L, (byte)(value >>> 18 | 0x40));
            this.writerIndex = writerIdx + 4;
            return 4;
        }
        UNSAFE.putByte(this.heapMemory, pos + 3L, (byte)(value >>> 18));
        UNSAFE.putInt(this.heapMemory, pos + 4L, 0);
        UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes + 3L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes + 4;
        return numPaddingBytes + 4;
    }

    private int writePositiveVarIntAligned5(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 9);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 2L, (byte)(value >>> 12 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 3L, (byte)(value >>> 18 | 0x80));
        if (numPaddingBytes == 1) {
            UNSAFE.putByte(this.heapMemory, pos + 4L, (byte)(value >>> 24 | 0x40));
            this.writerIndex = writerIdx + 5;
            return 5;
        }
        UNSAFE.putByte(this.heapMemory, pos + 4L, (byte)(value >>> 24));
        UNSAFE.putInt(this.heapMemory, pos + 5L, 0);
        UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes + 3L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes + 4;
        return numPaddingBytes + 4;
    }

    private int writePositiveVarIntAligned6(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 10);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 2L, (byte)(value >>> 12 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 3L, (byte)(value >>> 18 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 4L, (byte)(value >>> 24 | 0x80));
        if (numPaddingBytes == 2) {
            UNSAFE.putByte(this.heapMemory, pos + 5L, (byte)(value >>> 30 | 0x40));
            this.writerIndex = writerIdx + 6;
            return 6;
        }
        UNSAFE.putByte(this.heapMemory, pos + 5L, (byte)(value >>> 30));
        UNSAFE.putInt(this.heapMemory, pos + 6L, 0);
        if (numPaddingBytes == 1) {
            UNSAFE.putByte(this.heapMemory, pos + 8L, (byte)64);
            this.writerIndex = writerIdx + 9;
            return 9;
        }
        UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes + 3L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes + 4;
        return numPaddingBytes + 4;
    }

    public int readPositiveAlignedVarInt() {
        long pos;
        int readerIdx = this.readerIndex;
        if (readerIdx > this.size - 1) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, 1, this.size, this));
        }
        long startPos = pos = this.address + (long)readerIdx;
        byte b = UNSAFE.getByte(this.heapMemory, pos++);
        int result = b & 0x3F;
        if ((b & 0x80) != 0) {
            b = UNSAFE.getByte(this.heapMemory, pos++);
            result |= (b & 0x3F) << 6;
            if ((b & 0x80) != 0) {
                b = UNSAFE.getByte(this.heapMemory, pos++);
                result |= (b & 0x3F) << 12;
                if ((b & 0x80) != 0) {
                    b = UNSAFE.getByte(this.heapMemory, pos++);
                    result |= (b & 0x3F) << 18;
                    if ((b & 0x80) != 0) {
                        b = UNSAFE.getByte(this.heapMemory, pos++);
                        result |= (b & 0x3F) << 24;
                        if ((b & 0x80) != 0) {
                            b = UNSAFE.getByte(this.heapMemory, pos++);
                            result |= (b & 0x3F) << 30;
                        }
                    }
                }
            }
        }
        pos = this.skipPadding(pos, b);
        this.readerIndex = (int)(pos - startPos + (long)readerIdx);
        return result;
    }

    private long skipPadding(long pos, int b) {
        if ((b & 0x40) == 0 && ((b = (int)UNSAFE.getByte(this.heapMemory, pos++)) & 0x40) == 0 && ((b = (int)UNSAFE.getByte(this.heapMemory, pos++)) & 0x40) == 0) {
            Preconditions.checkArgument((((b = (int)UNSAFE.getByte(this.heapMemory, pos++)) & 0x40) != 0 ? 1 : 0) != 0, (Object)"At most 3 padding bytes.");
        }
        return pos;
    }

    public int writeVarLong(long value) {
        this.ensure(this.writerIndex + 9);
        value = value << 1 ^ value >> 63;
        return this.unsafeWritePositiveVarLong(value);
    }

    public int unsafeWriteVarLong(long value) {
        value = value << 1 ^ value >> 63;
        return this.unsafeWritePositiveVarLong(value);
    }

    public int writePositiveVarLong(long value) {
        this.ensure(this.writerIndex + 9);
        return this.unsafeWritePositiveVarLong(value);
    }

    public int unsafeWritePositiveVarLong(long value) {
        int writerIndex = this.writerIndex;
        int varInt = (int)(value & 0x7FL);
        if ((value >>>= 7) == 0L) {
            UNSAFE.putByte(this.heapMemory, this.address + (long)writerIndex, (byte)varInt);
            this.writerIndex = writerIndex + 1;
            return 1;
        }
        varInt |= 0x80;
        varInt = (int)((long)varInt | (value & 0x7FL) << 8);
        if ((value >>>= 7) == 0L) {
            this.unsafePutInt(writerIndex, varInt);
            this.writerIndex = writerIndex + 2;
            return 2;
        }
        varInt |= 0x8000;
        varInt = (int)((long)varInt | (value & 0x7FL) << 16);
        if ((value >>>= 7) == 0L) {
            this.unsafePutInt(writerIndex, varInt);
            this.writerIndex = writerIndex + 3;
            return 3;
        }
        varInt |= 0x800000;
        varInt = (int)((long)varInt | (value & 0x7FL) << 24);
        if ((value >>>= 7) == 0L) {
            this.unsafePutInt(writerIndex, varInt);
            this.writerIndex = writerIndex + 4;
            return 4;
        }
        varInt = (int)((long)varInt | 0x80000000L);
        long varLong = (long)varInt & 0xFFFFFFFFL;
        varLong |= (value & 0x7FL) << 32;
        if ((value >>>= 7) == 0L) {
            this.unsafePutLong(writerIndex, varLong);
            this.writerIndex = writerIndex + 5;
            return 5;
        }
        varLong |= 0x8000000000L;
        varLong |= (value & 0x7FL) << 40;
        if ((value >>>= 7) == 0L) {
            this.unsafePutLong(writerIndex, varLong);
            this.writerIndex = writerIndex + 6;
            return 6;
        }
        varLong |= 0x800000000000L;
        varLong |= (value & 0x7FL) << 48;
        if ((value >>>= 7) == 0L) {
            this.unsafePutLong(writerIndex, varLong);
            this.writerIndex = writerIndex + 7;
            return 7;
        }
        varLong |= 0x80000000000000L;
        varLong |= (value & 0x7FL) << 56;
        if ((value >>>= 7) == 0L) {
            this.unsafePutLong(writerIndex, varLong);
            this.writerIndex = writerIndex + 8;
            return 8;
        }
        this.unsafePutLong(writerIndex, varLong |= Long.MIN_VALUE);
        UNSAFE.putByte(this.heapMemory, this.address + (long)writerIndex + 8L, (byte)(value & 0xFFL));
        this.writerIndex = writerIndex + 9;
        return 9;
    }

    public long readVarLong() {
        long result = this.readPositiveVarLong();
        return result >>> 1 ^ -(result & 1L);
    }

    public long readPositiveVarLong() {
        int readIdx = this.readerIndex;
        if (this.size - readIdx < 9) {
            return this.readPositiveVarLongSlow();
        }
        long eightByteValue = this.unsafeGetLong(readIdx);
        long b = eightByteValue & 0xFFL;
        ++readIdx;
        long result = b & 0x7FL;
        if ((b & 0x80L) != 0L) {
            ++readIdx;
            b = eightByteValue >>> 8 & 0xFFL;
            result |= (b & 0x7FL) << 7;
            if ((b & 0x80L) != 0L) {
                ++readIdx;
                b = eightByteValue >>> 16 & 0xFFL;
                result |= (b & 0x7FL) << 14;
                if ((b & 0x80L) != 0L) {
                    ++readIdx;
                    b = eightByteValue >>> 24 & 0xFFL;
                    result |= (b & 0x7FL) << 21;
                    if ((b & 0x80L) != 0L) {
                        ++readIdx;
                        b = eightByteValue >>> 32 & 0xFFL;
                        result |= (b & 0x7FL) << 28;
                        if ((b & 0x80L) != 0L) {
                            ++readIdx;
                            b = eightByteValue >>> 40 & 0xFFL;
                            result |= (b & 0x7FL) << 35;
                            if ((b & 0x80L) != 0L) {
                                ++readIdx;
                                b = eightByteValue >>> 48 & 0xFFL;
                                result |= (b & 0x7FL) << 42;
                                if ((b & 0x80L) != 0L) {
                                    ++readIdx;
                                    b = eightByteValue >>> 56 & 0xFFL;
                                    result |= (b & 0x7FL) << 49;
                                    if ((b & 0x80L) != 0L) {
                                        b = this.unsafeGet(readIdx++);
                                        result |= b << 56;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        this.readerIndex = readIdx;
        return result;
    }

    private long readPositiveVarLongSlow() {
        long b = this.readByte();
        long result = b & 0x7FL;
        if ((b & 0x80L) != 0L) {
            b = this.readByte();
            result |= (b & 0x7FL) << 7;
            if ((b & 0x80L) != 0L) {
                b = this.readByte();
                result |= (b & 0x7FL) << 14;
                if ((b & 0x80L) != 0L) {
                    b = this.readByte();
                    result |= (b & 0x7FL) << 21;
                    if ((b & 0x80L) != 0L) {
                        b = this.readByte();
                        result |= (b & 0x7FL) << 28;
                        if ((b & 0x80L) != 0L) {
                            b = this.readByte();
                            result |= (b & 0x7FL) << 35;
                            if ((b & 0x80L) != 0L) {
                                b = this.readByte();
                                result |= (b & 0x7FL) << 42;
                                if ((b & 0x80L) != 0L) {
                                    b = this.readByte();
                                    result |= (b & 0x7FL) << 49;
                                    if ((b & 0x80L) != 0L) {
                                        b = this.readByte();
                                        result |= b << 56;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        return result;
    }

    public int writeSliLong(long value) {
        this.ensure(this.writerIndex + 9);
        return this.unsafeWriteSliLong(value);
    }

    public int unsafeWriteSliLong(long value) {
        int writerIndex = this.writerIndex;
        long pos = this.address + (long)writerIndex;
        byte[] heapMemory = this.heapMemory;
        if (value >= -1073741824L && value <= 0x3FFFFFFFL) {
            int v = (int)value << 1;
            if (LITTLE_ENDIAN) {
                UNSAFE.putInt(heapMemory, pos, v);
            } else {
                UNSAFE.putInt(heapMemory, pos, Integer.reverseBytes(v));
            }
            this.writerIndex = writerIndex + 4;
            return 4;
        }
        UNSAFE.putByte(heapMemory, pos, (byte)1);
        if (LITTLE_ENDIAN) {
            UNSAFE.putLong(heapMemory, pos + 1L, value);
        } else {
            UNSAFE.putLong(heapMemory, pos + 1L, Long.reverseBytes(value));
        }
        this.writerIndex = writerIndex + 9;
        return 9;
    }

    public long readSliLong() {
        int readIdx = this.readerIndex;
        long pos = this.address + (long)readIdx;
        int size = this.size;
        byte[] heapMemory = this.heapMemory;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readIdx > size - 4) {
            this.throwIndexOutOfBoundsException(readIdx, size, 4);
        }
        if (LITTLE_ENDIAN) {
            int i = UNSAFE.getInt(heapMemory, pos);
            if ((i & 1) != 1) {
                this.readerIndex = readIdx + 4;
                return i >> 1;
            }
            if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readIdx > size - 9) {
                this.throwIndexOutOfBoundsException(readIdx, size, 9);
            }
            this.readerIndex = readIdx + 9;
            return UNSAFE.getLong(heapMemory, pos + 1L);
        }
        int i = Integer.reverseBytes(UNSAFE.getInt(heapMemory, pos));
        if ((i & 1) != 1) {
            this.readerIndex = readIdx + 4;
            return i >> 1;
        }
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readIdx > size - 9) {
            this.throwIndexOutOfBoundsException(readIdx, size, 9);
        }
        this.readerIndex = readIdx + 9;
        return Long.reverseBytes(UNSAFE.getLong(heapMemory, pos + 1L));
    }

    private void throwIndexOutOfBoundsException(int readIdx, int size, int need) {
        throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readIdx, need, size, this));
    }

    public void writeBytes(byte[] bytes) {
        this.writeBytes(bytes, 0, bytes.length);
    }

    public void writeBytes(byte[] bytes, int offset, int length) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + length;
        this.ensure(newIdx);
        this.put(writerIdx, bytes, offset, length);
        this.writerIndex = newIdx;
    }

    public void write(ByteBuffer source) {
        this.write(source, source.remaining());
    }

    public void write(ByteBuffer source, int numBytes) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + numBytes;
        this.ensure(newIdx);
        this.put(writerIdx, source, numBytes);
        this.writerIndex = newIdx;
    }

    public void writeBytesWithSizeEmbedded(byte[] arr) {
        this.writePrimitiveArrayWithSizeEmbedded(arr, Platform.BYTE_ARRAY_OFFSET, arr.length);
    }

    public void writePrimitiveArrayWithSizeEmbedded(Object arr, int offset, int numBytes) {
        int idx = this.writerIndex;
        this.ensure(idx + 5 + numBytes);
        long destAddr = this.address + (long)(idx += this.unsafeWritePositiveVarInt(numBytes));
        Platform.copyMemory(arr, offset, this.heapMemory, destAddr, numBytes);
        this.writerIndex = idx + numBytes;
    }

    public void writePrimitiveArrayAlignedSizeEmbedded(Object arr, int offset, int numBytes) {
        this.writePositiveVarIntAligned(numBytes);
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + numBytes;
        this.ensure(newIdx);
        long destAddr = this.address + (long)writerIdx;
        Platform.copyMemory(arr, offset, this.heapMemory, destAddr, numBytes);
        this.writerIndex = newIdx;
    }

    public void writePrimitiveArray(Object arr, int offset, int numBytes) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + numBytes;
        this.ensure(newIdx);
        long destAddr = this.address + (long)writerIdx;
        Platform.copyMemory(arr, offset, this.heapMemory, destAddr, numBytes);
        this.writerIndex = newIdx;
    }

    public void grow(int neededSize) {
        this.ensure(this.writerIndex + neededSize);
    }

    public void ensure(int length) {
        if (length > this.size) {
            byte[] data = new byte[length * 2];
            this.copyToUnsafe(0L, data, BYTE_ARRAY_BASE_OFFSET, this.size());
            this.initHeapBuffer(data, 0, data.length);
        }
    }

    public boolean readBoolean() {
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - 1) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, 1, this.size, this));
        }
        this.readerIndex = readerIdx + 1;
        return UNSAFE.getByte(this.heapMemory, this.address + (long)readerIdx) != 0;
    }

    public byte readByte() {
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - 1) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, 1, this.size, this));
        }
        this.readerIndex = readerIdx + 1;
        return UNSAFE.getByte(this.heapMemory, this.address + (long)readerIdx);
    }

    public char readChar() {
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - 2) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, 2, this.size, this));
        }
        this.readerIndex = readerIdx + 2;
        long pos = this.address + (long)readerIdx;
        if (LITTLE_ENDIAN) {
            return UNSAFE.getChar(this.heapMemory, pos);
        }
        return Character.reverseBytes(UNSAFE.getChar(this.heapMemory, pos));
    }

    public short readShort() {
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - 2) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, 2, this.size, this));
        }
        this.readerIndex = readerIdx + 2;
        long pos = this.address + (long)readerIdx;
        if (LITTLE_ENDIAN) {
            return UNSAFE.getShort(this.heapMemory, pos);
        }
        return Short.reverseBytes(UNSAFE.getShort(this.heapMemory, pos));
    }

    public int readInt() {
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - 4) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, 4, this.size, this));
        }
        this.readerIndex = readerIdx + 4;
        long pos = this.address + (long)readerIdx;
        if (LITTLE_ENDIAN) {
            return UNSAFE.getInt(this.heapMemory, pos);
        }
        return Integer.reverseBytes(UNSAFE.getInt(this.heapMemory, pos));
    }

    public long readLong() {
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - 8) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, 8, this.size, this));
        }
        this.readerIndex = readerIdx + 8;
        long pos = this.address + (long)readerIdx;
        if (LITTLE_ENDIAN) {
            return UNSAFE.getLong(this.heapMemory, pos);
        }
        return Long.reverseBytes(UNSAFE.getLong(this.heapMemory, pos));
    }

    public float readFloat() {
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - 4) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, 4, this.size, this));
        }
        this.readerIndex = readerIdx + 4;
        long pos = this.address + (long)readerIdx;
        if (LITTLE_ENDIAN) {
            return Float.intBitsToFloat(UNSAFE.getInt(this.heapMemory, pos));
        }
        return Float.intBitsToFloat(Integer.reverseBytes(UNSAFE.getInt(this.heapMemory, pos)));
    }

    public double readDouble() {
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - 8) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, 8, this.size, this));
        }
        this.readerIndex = readerIdx + 8;
        long pos = this.address + (long)readerIdx;
        if (LITTLE_ENDIAN) {
            return Double.longBitsToDouble(UNSAFE.getLong(this.heapMemory, pos));
        }
        return Double.longBitsToDouble(Long.reverseBytes(UNSAFE.getLong(this.heapMemory, pos)));
    }

    public byte[] readBytes(int length) {
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - length) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, length, this.size, this));
        }
        byte[] heapMemory = this.heapMemory;
        byte[] bytes = new byte[length];
        if (heapMemory != null) {
            System.arraycopy(heapMemory, this.heapOffset + readerIdx, bytes, 0, length);
        } else {
            Platform.copyMemory(null, this.address + (long)readerIdx, bytes, Platform.BYTE_ARRAY_OFFSET, length);
        }
        this.readerIndex = readerIdx + length;
        return bytes;
    }

    public void readBytes(byte[] dst, int dstIndex, int length) {
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - length) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, length, this.size, this));
        }
        if (dstIndex > dst.length - length) {
            throw new IndexOutOfBoundsException();
        }
        this.copyToUnsafe(readerIdx, dst, BYTE_ARRAY_BASE_OFFSET + (long)dstIndex, length);
        this.readerIndex = readerIdx + length;
    }

    public void readBytes(byte[] dst) {
        this.readBytes(dst, 0, dst.length);
    }

    public void read(ByteBuffer dst) {
        int readerIdx = this.readerIndex;
        int len = Math.min(dst.remaining(), this.size - readerIdx);
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - len) {
            throw new IndexOutOfBoundsException(String.format("readerIdx(%d) + length(%d) exceeds size(%d): %s", readerIdx, len, this.size, this));
        }
        this.readerIndex = readerIdx + len;
        dst.put(this.sliceAsByteBuffer(readerIdx, len));
    }

    public byte[] readBytesWithSizeEmbedded() {
        int numBytes = this.readPositiveVarInt();
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - numBytes) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, numBytes, this.size, this));
        }
        byte[] arr = new byte[numBytes];
        byte[] heapMemory = this.heapMemory;
        if (heapMemory != null) {
            System.arraycopy(heapMemory, this.heapOffset + readerIdx, arr, 0, numBytes);
        } else {
            Platform.UNSAFE.copyMemory(null, this.address + (long)readerIdx, arr, Platform.BYTE_ARRAY_OFFSET, numBytes);
        }
        this.readerIndex = readerIdx + numBytes;
        return arr;
    }

    public byte[] readBytesAlignedSizeEmbedded() {
        int numBytes = this.readPositiveAlignedVarInt();
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - numBytes) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", readerIdx, numBytes, this.size, this));
        }
        byte[] arr = new byte[numBytes];
        Platform.UNSAFE.copyMemory(this.heapMemory, this.address + (long)readerIdx, arr, Platform.BYTE_ARRAY_OFFSET, numBytes);
        this.readerIndex = readerIdx + numBytes;
        return arr;
    }

    public char[] readCharsWithSizeEmbedded() {
        int numBytes = this.readPositiveVarInt();
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - numBytes) {
            throw new IndexOutOfBoundsException(String.format("readerIdx(%d) + length(%d) exceeds size(%d): %s", readerIdx, numBytes, this.size, this));
        }
        char[] chars = new char[numBytes / 2];
        Platform.copyMemory(this.heapMemory, this.address + (long)readerIdx, chars, Platform.CHAR_ARRAY_OFFSET, numBytes);
        this.readerIndex = readerIdx + numBytes;
        return chars;
    }

    public char[] readCharsAlignedSizeEmbedded() {
        int numBytes = this.readPositiveAlignedVarInt();
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - numBytes) {
            throw new IndexOutOfBoundsException(String.format("readerIdx(%d) + length(%d) exceeds size(%d): %s", readerIdx, numBytes, this.size, this));
        }
        char[] chars = new char[numBytes / 2];
        Platform.copyMemory(this.heapMemory, this.address + (long)readerIdx, chars, Platform.CHAR_ARRAY_OFFSET, numBytes);
        this.readerIndex = readerIdx + numBytes;
        return chars;
    }

    public long[] readLongsWithSizeEmbedded() {
        int readerIdx = this.readerIndex;
        int numBytes = this.readPositiveVarInt();
        if (readerIdx > this.size - numBytes) {
            throw new IndexOutOfBoundsException(String.format("readerIdx(%d) + length(%d) exceeds size(%d): %s", readerIdx, numBytes, this.size, this));
        }
        long[] longs = new long[numBytes / 8];
        Platform.copyMemory(this.heapMemory, this.address + (long)readerIdx, longs, Platform.LONG_ARRAY_OFFSET, numBytes);
        this.readerIndex = readerIdx + numBytes;
        return longs;
    }

    public void readChars(char[] chars, int offset, int numBytes) {
        int readerIdx = this.readerIndex;
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && readerIdx > this.size - numBytes) {
            throw new IndexOutOfBoundsException(String.format("readerIdx(%d) + length(%d) exceeds size(%d): %s", readerIdx, numBytes, this.size, this));
        }
        Platform.copyMemory(this.heapMemory, this.address + (long)readerIdx, chars, offset, numBytes);
        this.readerIndex = readerIdx + numBytes;
    }

    public void checkReadableBytes(int minimumReadableBytes) {
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && this.readerIndex > this.size - minimumReadableBytes) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", this.readerIndex, minimumReadableBytes, this.size, this));
        }
    }

    public void copyToUnsafe(long offset, Object target, long targetPointer, int numBytes) {
        long thisPointer = this.address + offset;
        Preconditions.checkArgument((thisPointer + (long)numBytes <= this.addressLimit ? 1 : 0) != 0);
        Platform.copyMemory(this.heapMemory, thisPointer, target, targetPointer, numBytes);
    }

    public void copyToUnsafeSmall(long offset, Object target, long targetPointer, int numBytes) {
        long thisPointer = this.address + offset;
        Preconditions.checkArgument((thisPointer + (long)numBytes <= this.addressLimit ? 1 : 0) != 0);
        Platform.UNSAFE.copyMemory(this.heapMemory, thisPointer, target, targetPointer, numBytes);
    }

    public void copyFromUnsafe(long offset, Object source, long sourcePointer, long numBytes) {
        long thisPointer = this.address + offset;
        Preconditions.checkArgument((thisPointer + numBytes <= this.addressLimit ? 1 : 0) != 0);
        Platform.copyMemory(source, sourcePointer, this.heapMemory, thisPointer, numBytes);
    }

    public void copyFromUnsafeSmall(long offset, Object source, long sourcePointer, long numBytes) {
        long thisPointer = this.address + offset;
        Preconditions.checkArgument((thisPointer + numBytes <= this.addressLimit ? 1 : 0) != 0);
        Platform.UNSAFE.copyMemory(source, sourcePointer, this.heapMemory, thisPointer, numBytes);
    }

    public void copyTo(int offset, MemoryBuffer target, int targetOffset, int numBytes) {
        byte[] thisHeapRef = this.heapMemory;
        byte[] otherHeapRef = target.heapMemory;
        long thisPointer = this.address + (long)offset;
        long otherPointer = target.address + (long)targetOffset;
        if ((numBytes | offset | targetOffset) < 0 || thisPointer > this.addressLimit - (long)numBytes || otherPointer > target.addressLimit - (long)numBytes) {
            if (this.address > this.addressLimit) {
                throw new IllegalStateException("this memory buffer has been freed.");
            }
            if (target.address > target.addressLimit) {
                throw new IllegalStateException("target memory buffer has been freed.");
            }
            throw new IndexOutOfBoundsException(String.format("offset=%d, targetOffset=%d, numBytes=%d, address=%d, targetAddress=%d", offset, targetOffset, numBytes, this.address, target.address));
        }
        UNSAFE.copyMemory(thisHeapRef, thisPointer, otherHeapRef, otherPointer, numBytes);
    }

    public void copyFrom(int offset, MemoryBuffer source, int sourcePointer, int numBytes) {
        source.copyTo(sourcePointer, this, offset, numBytes);
    }

    public byte[] getRemainingBytes() {
        int length = this.size - this.readerIndex;
        if (this.heapMemory != null && this.size == length) {
            return this.heapMemory;
        }
        return this.getBytes(this.readerIndex, length);
    }

    public byte[] getAllBytes() {
        if (this.heapMemory != null && this.size == this.heapMemory.length) {
            return this.heapMemory;
        }
        return this.getBytes(0, this.size);
    }

    public byte[] getBytes(int index, int length) {
        if (index == 0 && this.heapMemory != null && this.heapOffset == 0) {
            return Arrays.copyOf(this.heapMemory, length);
        }
        if (index + length > this.size) {
            throw new IllegalArgumentException();
        }
        byte[] data = new byte[length];
        this.copyToUnsafe(index, data, BYTE_ARRAY_BASE_OFFSET, length);
        return data;
    }

    public void getBytes(int index, byte[] dst, int dstIndex, int length) {
        if (dstIndex > dst.length - length) {
            throw new IndexOutOfBoundsException();
        }
        if (index > this.size - length) {
            throw new IndexOutOfBoundsException(String.format("offset(%d) + length(%d) exceeds size(%d): %s", index, length, this.size, this));
        }
        this.copyToUnsafe(index, dst, BYTE_ARRAY_BASE_OFFSET + (long)dstIndex, length);
    }

    public MemoryBuffer slice(int offset) {
        return this.slice(offset, this.size - offset);
    }

    public MemoryBuffer slice(int offset, int length) {
        if (offset + length > this.size) {
            throw new IndexOutOfBoundsException(String.format("offset(%d) + length(%d) exceeds size(%d): %s", offset, length, this.size, this));
        }
        if (this.heapMemory != null) {
            return new MemoryBuffer(this.heapMemory, this.heapOffset + offset, length);
        }
        return new MemoryBuffer(this.address + (long)offset, length, this.offHeapBuffer);
    }

    public ByteBuffer sliceAsByteBuffer() {
        return this.sliceAsByteBuffer(this.readerIndex, this.size - this.readerIndex);
    }

    public ByteBuffer sliceAsByteBuffer(int offset, int length) {
        if (offset + length > this.size) {
            throw new IndexOutOfBoundsException(String.format("offset(%d) + length(%d) exceeds size(%d): %s", offset, length, this.size, this));
        }
        if (this.heapMemory != null) {
            return ByteBuffer.wrap(this.heapMemory, this.heapOffset + offset, length).slice();
        }
        ByteBuffer offHeapBuffer = this.offHeapBuffer;
        if (offHeapBuffer != null) {
            ByteBuffer duplicate = offHeapBuffer.duplicate();
            int start = (int)(this.address - Platform.getAddress(duplicate));
            duplicate.position(start + offset);
            duplicate.limit(start + offset + length);
            return duplicate.slice();
        }
        return Platform.createDirectByteBufferFromNativeAddress(this.address + (long)offset, length);
    }

    public int compare(MemoryBuffer buf2, int offset1, int offset2, int len) {
        while (len >= 8) {
            long l2;
            long l1 = this.getLongB(offset1);
            if (l1 != (l2 = buf2.getLongB(offset2))) {
                return l1 < l2 ^ l1 < 0L ^ l2 < 0L ? -1 : 1;
            }
            offset1 += 8;
            offset2 += 8;
            len -= 8;
        }
        while (len > 0) {
            int b2;
            int b1 = this.get(offset1) & 0xFF;
            int cmp = b1 - (b2 = buf2.get(offset2) & 0xFF);
            if (cmp != 0) {
                return cmp;
            }
            ++offset1;
            ++offset2;
            --len;
        }
        return 0;
    }

    public boolean equalTo(MemoryBuffer buf2, int offset1, int offset2, int len) {
        long pos1 = this.address + (long)offset1;
        long pos2 = buf2.address + (long)offset2;
        Preconditions.checkArgument((pos1 < this.addressLimit ? 1 : 0) != 0);
        Preconditions.checkArgument((pos2 < buf2.addressLimit ? 1 : 0) != 0);
        return Platform.arrayEquals(this.heapMemory, pos1, buf2.heapMemory, pos2, len);
    }

    public MemoryBuffer cloneReference() {
        if (this.heapMemory != null) {
            return new MemoryBuffer(this.heapMemory, this.heapOffset, this.size);
        }
        return new MemoryBuffer(this.address, this.size, this.offHeapBuffer);
    }

    public String toString() {
        return "MemoryBuffer{size=" + this.size + ", readerIndex=" + this.readerIndex + ", writerIndex=" + this.writerIndex + ", heapMemory=" + (this.heapMemory == null ? null : "len(" + this.heapMemory.length + ")") + ", heapOffset=" + this.heapOffset + ", offHeapBuffer=" + this.offHeapBuffer + ", address=" + this.address + ", addressLimit=" + this.addressLimit + '}';
    }

    public void pointTo(byte[] buffer, int offset, int length) {
        this.initHeapBuffer(buffer, offset, length);
    }

    public static MemoryBuffer fromByteArray(byte[] buffer, int offset, int length) {
        return new MemoryBuffer(buffer, offset, length);
    }

    public static MemoryBuffer fromByteArray(byte[] buffer) {
        return new MemoryBuffer(buffer, 0, buffer.length);
    }

    public static MemoryBuffer fromByteBuffer(ByteBuffer buffer) {
        if (buffer.isDirect()) {
            return new MemoryBuffer(Platform.getAddress(buffer) + (long)buffer.position(), buffer.remaining(), buffer);
        }
        int offset = buffer.arrayOffset() + buffer.position();
        return new MemoryBuffer(buffer.array(), offset, buffer.remaining());
    }

    public static MemoryBuffer fromNativeAddress(long address, int size) {
        return new MemoryBuffer(address, size, null);
    }

    public static MemoryBuffer newHeapBuffer(int initialSize) {
        return MemoryBuffer.fromByteArray(new byte[initialSize]);
    }
}

