package cdm.product.asset.floatingrate.functions;

import cdm.base.staticdata.asset.rates.FloatingRateIndexEnum;
import cdm.base.staticdata.asset.rates.metafields.FieldWithMetaFloatingRateIndexEnum;
import cdm.observable.asset.FloatingRateOption;
import cdm.observable.asset.calculatedrate.FloatingRateCalculationParameters;
import cdm.observable.asset.fro.FloatingRateIndexCalculationDefaults;
import cdm.observable.asset.fro.FloatingRateIndexCalculationMethodEnum;
import cdm.observable.asset.fro.FloatingRateIndexCategoryEnum;
import cdm.observable.asset.fro.FloatingRateIndexDefinition;
import cdm.observable.asset.fro.FloatingRateIndexIdentification;
import cdm.observable.asset.fro.FloatingRateIndexStyleEnum;
import cdm.observable.asset.fro.functions.FloatingRateIndexMetadata;
import cdm.observable.asset.metafields.ReferenceWithMetaFloatingRateOption;
import cdm.product.asset.FloatingRateSpecification;
import cdm.product.asset.floatingrate.FloatingRateIndexProcessingTypeEnum;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(GetFloatingRateProcessingType.GetFloatingRateProcessingTypeDefault.class)
public abstract class GetFloatingRateProcessingType implements RosettaFunction {
	
	// RosettaFunction dependencies
	//
	@Inject protected FloatingRateIndexMetadata floatingRateIndexMetadata;

	/**
	* @param rateDef Specification details of the floating rate.
	* @return processingType The processing category for the rate .
	*/
	public FloatingRateIndexProcessingTypeEnum evaluate(FloatingRateSpecification rateDef) {
		FloatingRateIndexProcessingTypeEnum processingType = doEvaluate(rateDef);
		
		return processingType;
	}

	protected abstract FloatingRateIndexProcessingTypeEnum doEvaluate(FloatingRateSpecification rateDef);

	protected abstract Mapper<Boolean> isCalculatedRate(FloatingRateSpecification rateDef);

	protected abstract Mapper<? extends FloatingRateIndexDefinition> floatingRateDefinition(FloatingRateSpecification rateDef);

	protected abstract Mapper<? extends FloatingRateIndexCalculationDefaults> calcDefaults(FloatingRateSpecification rateDef);

	protected abstract Mapper<FloatingRateIndexCategoryEnum> category(FloatingRateSpecification rateDef);

	protected abstract Mapper<FloatingRateIndexStyleEnum> idxStyle(FloatingRateSpecification rateDef);

	protected abstract Mapper<FloatingRateIndexCalculationMethodEnum> method(FloatingRateSpecification rateDef);

	protected abstract Mapper<FloatingRateIndexProcessingTypeEnum> calcProcessingType(FloatingRateSpecification rateDef);

	protected abstract Mapper<FloatingRateIndexProcessingTypeEnum> definitionProcessingType(FloatingRateSpecification rateDef);

	protected abstract Mapper<FloatingRateIndexProcessingTypeEnum> processingCategory(FloatingRateSpecification rateDef);

	public static class GetFloatingRateProcessingTypeDefault extends GetFloatingRateProcessingType {
		@Override
		protected FloatingRateIndexProcessingTypeEnum doEvaluate(FloatingRateSpecification rateDef) {
			FloatingRateIndexProcessingTypeEnum processingType = null;
			return assignOutput(processingType, rateDef);
		}
		
		protected FloatingRateIndexProcessingTypeEnum assignOutput(FloatingRateIndexProcessingTypeEnum processingType, FloatingRateSpecification rateDef) {
			processingType = MapperS.of(processingCategory(rateDef).get()).get();
			
			return processingType;
		}
		
		@Override
		protected Mapper<Boolean> isCalculatedRate(FloatingRateSpecification rateDef) {
			return exists(MapperS.of(rateDef).<FloatingRateCalculationParameters>map("getCalculationParameters", floatingRate -> floatingRate.getCalculationParameters()));
		}
		
		@Override
		protected Mapper<? extends FloatingRateIndexDefinition> floatingRateDefinition(FloatingRateSpecification rateDef) {
			return MapperS.of(floatingRateIndexMetadata.evaluate(MapperS.of(rateDef).<ReferenceWithMetaFloatingRateOption>map("getRateOption", floatingRateBase -> floatingRateBase.getRateOption()).<FloatingRateOption>map("getValue", _f->_f.getValue()).<FieldWithMetaFloatingRateIndexEnum>map("getFloatingRateIndex", floatingRateOption -> floatingRateOption.getFloatingRateIndex()).<FloatingRateIndexEnum>map("getValue", _f->_f.getValue()).get()));
		}
		
		@Override
		protected Mapper<? extends FloatingRateIndexCalculationDefaults> calcDefaults(FloatingRateSpecification rateDef) {
			return MapperS.of(floatingRateDefinition(rateDef).get()).<FloatingRateIndexCalculationDefaults>map("getCalculationDefaults", floatingRateIndexDefinition -> floatingRateIndexDefinition.getCalculationDefaults());
		}
		
		@Override
		protected Mapper<FloatingRateIndexCategoryEnum> category(FloatingRateSpecification rateDef) {
			return MapperS.of(calcDefaults(rateDef).get()).<FloatingRateIndexCategoryEnum>map("getCategory", floatingRateIndexCalculationDefaults -> floatingRateIndexCalculationDefaults.getCategory());
		}
		
		@Override
		protected Mapper<FloatingRateIndexStyleEnum> idxStyle(FloatingRateSpecification rateDef) {
			return MapperS.of(calcDefaults(rateDef).get()).<FloatingRateIndexStyleEnum>map("getIndexStyle", floatingRateIndexCalculationDefaults -> floatingRateIndexCalculationDefaults.getIndexStyle());
		}
		
		@Override
		protected Mapper<FloatingRateIndexCalculationMethodEnum> method(FloatingRateSpecification rateDef) {
			return MapperS.of(calcDefaults(rateDef).get()).<FloatingRateIndexCalculationMethodEnum>map("getMethod", floatingRateIndexCalculationDefaults -> floatingRateIndexCalculationDefaults.getMethod());
		}
		
		@Override
		protected Mapper<FloatingRateIndexProcessingTypeEnum> calcProcessingType(FloatingRateSpecification rateDef) {
			return MapperUtils.runSingle(() -> {
				if (areEqual(MapperS.of(idxStyle(rateDef).get()), MapperS.of(FloatingRateIndexStyleEnum.COMPOUNDED_FRO), CardinalityOperator.All).and(areEqual(MapperS.of(method(rateDef).get()), MapperS.of(FloatingRateIndexCalculationMethodEnum.OIS_COMPOUND), CardinalityOperator.All)).getOrDefault(false)) {
					return MapperS.of(FloatingRateIndexProcessingTypeEnum.OIS);
				}
				else if (areEqual(MapperS.of(idxStyle(rateDef).get()), MapperS.of(FloatingRateIndexStyleEnum.AVERAGE_FRO), CardinalityOperator.All).and(areEqual(MapperS.of(method(rateDef).get()), MapperS.of(FloatingRateIndexCalculationMethodEnum.AVERAGE), CardinalityOperator.All)).getOrDefault(false)) {
					return MapperS.of(FloatingRateIndexProcessingTypeEnum.OVERNIGHT_AVG);
				}
				else {
					return MapperS.<FloatingRateIndexProcessingTypeEnum>ofNull();
				}
			});
		}
		
		@Override
		protected Mapper<FloatingRateIndexProcessingTypeEnum> definitionProcessingType(FloatingRateSpecification rateDef) {
			return MapperUtils.runSingle(() -> {
				if (areEqual(MapperS.of(category(rateDef).get()), MapperS.of(FloatingRateIndexCategoryEnum.SCREEN_RATE), CardinalityOperator.All).getOrDefault(false)) {
					return MapperS.of(FloatingRateIndexProcessingTypeEnum.SCREEN);
				}
				else {
					return MapperS.of(calcProcessingType(rateDef).get());
				}
			});
		}
		
		@Override
		protected Mapper<FloatingRateIndexProcessingTypeEnum> processingCategory(FloatingRateSpecification rateDef) {
			return MapperUtils.runSingle(() -> {
				if (areEqual(MapperS.of(isCalculatedRate(rateDef).get()), MapperS.of(Boolean.valueOf(true)), CardinalityOperator.All).getOrDefault(false)) {
					return MapperS.of(FloatingRateIndexProcessingTypeEnum.MODULAR);
				}
				else if (exists(MapperS.of(floatingRateDefinition(rateDef).get()).<FloatingRateIndexIdentification>map("getFro", floatingRateIndexDefinition -> floatingRateIndexDefinition.getFro())).getOrDefault(false)) {
					return MapperS.of(definitionProcessingType(rateDef).get());
				}
				else {
					return MapperS.of(FloatingRateIndexProcessingTypeEnum.SCREEN);
				}
			});
		}
	}
}
