package cdm.observable.asset.calculatedrate.functions;

import cdm.base.datetime.BusinessCenterEnum;
import cdm.base.datetime.BusinessCenters;
import cdm.base.datetime.functions.GetAllBusinessCenters;
import cdm.observable.asset.calculatedrate.FloatingRateCalculationParameters;
import cdm.observable.asset.calculatedrate.ObservationShiftCalculation;
import cdm.observable.asset.calculatedrate.OffsetCalculation;
import cdm.product.common.schedule.CalculationPeriodBase;
import cdm.product.common.schedule.CalculationPeriodBase.CalculationPeriodBaseBuilder;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import java.util.Optional;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(DetermineObservationPeriod.DetermineObservationPeriodDefault.class)
public abstract class DetermineObservationPeriod implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected GenerateObservationPeriod generateObservationPeriod;
	@Inject protected GetAllBusinessCenters getAllBusinessCenters;

	/**
	* @param adjustedCalculationPeriod The calculation period for which the rate is being computed, after any adjustment.
	* @param calculationParams Floating rate definition for the calculated rate.
	* @return observationPeriod The resulting observation period.
	*/
	public CalculationPeriodBase evaluate(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams) {
		CalculationPeriodBase.CalculationPeriodBaseBuilder observationPeriodBuilder = doEvaluate(adjustedCalculationPeriod, calculationParams);
		
		final CalculationPeriodBase observationPeriod;
		if (observationPeriodBuilder == null) {
			observationPeriod = null;
		} else {
			observationPeriod = observationPeriodBuilder.build();
			objectValidator.validate(CalculationPeriodBase.class, observationPeriod);
		}
		
		return observationPeriod;
	}

	protected abstract CalculationPeriodBase.CalculationPeriodBaseBuilder doEvaluate(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams);

	protected abstract Mapper<? extends ObservationShiftCalculation> obsShift(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams);

	protected abstract Mapper<? extends OffsetCalculation> lookback(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams);

	protected abstract Mapper<? extends BusinessCenters> businessDays(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams);

	protected abstract Mapper<? extends BusinessCenters> additionalBusinessDays(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams);

	protected abstract Mapper<BusinessCenterEnum> allBusinessDays(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams);

	protected abstract Mapper<Integer> shift(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams);

	protected abstract Mapper<Integer> shiftDefaulted(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams);

	public static class DetermineObservationPeriodDefault extends DetermineObservationPeriod {
		@Override
		protected CalculationPeriodBase.CalculationPeriodBaseBuilder doEvaluate(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams) {
			CalculationPeriodBase.CalculationPeriodBaseBuilder observationPeriod = CalculationPeriodBase.builder();
			return assignOutput(observationPeriod, adjustedCalculationPeriod, calculationParams);
		}
		
		protected CalculationPeriodBase.CalculationPeriodBaseBuilder assignOutput(CalculationPeriodBase.CalculationPeriodBaseBuilder observationPeriod, CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams) {
			observationPeriod = toBuilder(MapperS.of(generateObservationPeriod.evaluate(MapperS.of(adjustedCalculationPeriod).get(), MapperC.<BusinessCenterEnum>of(allBusinessDays(adjustedCalculationPeriod, calculationParams).getMulti()).getMulti(), MapperS.of(shiftDefaulted(adjustedCalculationPeriod, calculationParams).get()).get())).get());
			
			return Optional.ofNullable(observationPeriod)
				.map(o -> o.prune())
				.orElse(null);
		}
		
		@Override
		protected Mapper<? extends ObservationShiftCalculation> obsShift(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams) {
			return MapperS.of(calculationParams).<ObservationShiftCalculation>map("getObservationShiftCalculation", floatingRateCalculationParameters -> floatingRateCalculationParameters.getObservationShiftCalculation());
		}
		
		@Override
		protected Mapper<? extends OffsetCalculation> lookback(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams) {
			return MapperS.of(calculationParams).<OffsetCalculation>map("getLookbackCalculation", floatingRateCalculationParameters -> floatingRateCalculationParameters.getLookbackCalculation());
		}
		
		@Override
		protected Mapper<? extends BusinessCenters> businessDays(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams) {
			return MapperS.of(calculationParams).<BusinessCenters>map("getApplicableBusinessDays", floatingRateCalculationParameters -> floatingRateCalculationParameters.getApplicableBusinessDays());
		}
		
		@Override
		protected Mapper<? extends BusinessCenters> additionalBusinessDays(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams) {
			return MapperS.of(obsShift(adjustedCalculationPeriod, calculationParams).get()).<BusinessCenters>map("getAdditionalBusinessDays", observationShiftCalculation -> observationShiftCalculation.getAdditionalBusinessDays());
		}
		
		@Override
		protected Mapper<BusinessCenterEnum> allBusinessDays(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams) {
			return MapperC.<BusinessCenters>of(MapperS.of(businessDays(adjustedCalculationPeriod, calculationParams).get()), MapperS.of(additionalBusinessDays(adjustedCalculationPeriod, calculationParams).get()))
				.mapItemToList(item -> (MapperC<BusinessCenterEnum>)MapperC.<BusinessCenterEnum>of(getAllBusinessCenters.evaluate(item.get())))
				.apply(item -> item
					.flattenList());
		}
		
		@Override
		protected Mapper<Integer> shift(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(obsShift(adjustedCalculationPeriod, calculationParams).get())).getOrDefault(false)) {
					return MapperS.of(obsShift(adjustedCalculationPeriod, calculationParams).get()).<Integer>map("getOffsetDays", observationShiftCalculation -> observationShiftCalculation.getOffsetDays());
				}
				else if (exists(MapperS.of(lookback(adjustedCalculationPeriod, calculationParams).get())).getOrDefault(false)) {
					return MapperS.of(lookback(adjustedCalculationPeriod, calculationParams).get()).<Integer>map("getOffsetDays", offsetCalculation -> offsetCalculation.getOffsetDays());
				}
				else {
					return MapperS.of(Integer.valueOf(0));
				}
			});
		}
		
		@Override
		protected Mapper<Integer> shiftDefaulted(CalculationPeriodBase adjustedCalculationPeriod, FloatingRateCalculationParameters calculationParams) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(shift(adjustedCalculationPeriod, calculationParams).get())).getOrDefault(false)) {
					return MapperS.of(shift(adjustedCalculationPeriod, calculationParams).get());
				}
				else {
					return MapperS.of(Integer.valueOf(5));
				}
			});
		}
	}
}
