/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.job;

import java.io.IOException;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateTaskConfig;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.CheckedConsumer;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.xpack.ml.MlMetadata;
import org.elasticsearch.xpack.ml.action.DeleteJobAction;
import org.elasticsearch.xpack.ml.action.PutJobAction;
import org.elasticsearch.xpack.ml.action.RevertModelSnapshotAction;
import org.elasticsearch.xpack.ml.action.util.QueryPage;
import org.elasticsearch.xpack.ml.job.UpdateJobProcessNotifier;
import org.elasticsearch.xpack.ml.job.config.Job;
import org.elasticsearch.xpack.ml.job.config.JobState;
import org.elasticsearch.xpack.ml.job.config.JobUpdate;
import org.elasticsearch.xpack.ml.job.messages.Messages;
import org.elasticsearch.xpack.ml.job.persistence.AnomalyDetectorsIndex;
import org.elasticsearch.xpack.ml.job.persistence.JobProvider;
import org.elasticsearch.xpack.ml.job.persistence.JobStorageDeletionTask;
import org.elasticsearch.xpack.ml.job.process.autodetect.state.ModelSnapshot;
import org.elasticsearch.xpack.ml.notifications.Auditor;
import org.elasticsearch.xpack.ml.utils.ExceptionsHelper;
import org.elasticsearch.xpack.persistent.PersistentTasksCustomMetaData;

public class JobManager
extends AbstractComponent {
    private final JobProvider jobProvider;
    private final ClusterService clusterService;
    private final Auditor auditor;
    private final Client client;
    private final UpdateJobProcessNotifier updateJobProcessNotifier;

    public JobManager(Settings settings, JobProvider jobProvider, ClusterService clusterService, Auditor auditor, Client client, UpdateJobProcessNotifier updateJobProcessNotifier) {
        super(settings);
        this.jobProvider = Objects.requireNonNull(jobProvider);
        this.clusterService = Objects.requireNonNull(clusterService);
        this.auditor = Objects.requireNonNull(auditor);
        this.client = Objects.requireNonNull(client);
        this.updateJobProcessNotifier = updateJobProcessNotifier;
    }

    public QueryPage<Job> getJob(String jobId, ClusterState clusterState) {
        if (jobId.equals("_all")) {
            return this.getJobs(clusterState);
        }
        MlMetadata mlMetadata = (MlMetadata)clusterState.getMetaData().custom("ml");
        Job job = mlMetadata.getJobs().get(jobId);
        if (job == null) {
            this.logger.debug(String.format(Locale.ROOT, "Cannot find job '%s'", jobId));
            throw ExceptionsHelper.missingJobException(jobId);
        }
        this.logger.debug("Returning job [" + jobId + "]");
        return new QueryPage<Job>(Collections.singletonList(job), 1L, Job.RESULTS_FIELD);
    }

    public QueryPage<Job> getJobs(ClusterState clusterState) {
        MlMetadata mlMetadata = (MlMetadata)clusterState.getMetaData().custom("ml");
        List jobs = mlMetadata.getJobs().entrySet().stream().map(Map.Entry::getValue).collect(Collectors.toList());
        return new QueryPage<Job>(jobs, mlMetadata.getJobs().size(), Job.RESULTS_FIELD);
    }

    public Job getJobOrThrowIfUnknown(String jobId) {
        return JobManager.getJobOrThrowIfUnknown(this.clusterService.state(), jobId);
    }

    public JobState getJobState(String jobId) {
        PersistentTasksCustomMetaData tasks = (PersistentTasksCustomMetaData)this.clusterService.state().getMetaData().custom("persistent_tasks");
        return MlMetadata.getJobState(jobId, tasks);
    }

    public static Job getJobOrThrowIfUnknown(ClusterState clusterState, String jobId) {
        MlMetadata mlMetadata = (MlMetadata)clusterState.metaData().custom("ml");
        Job job = mlMetadata.getJobs().get(jobId);
        if (job == null) {
            throw ExceptionsHelper.missingJobException(jobId);
        }
        return job;
    }

    public void putJob(final PutJobAction.Request request, ClusterState state, final ActionListener<PutJobAction.Response> actionListener) {
        final Job job = request.getJobBuilder().build(new Date());
        this.jobProvider.createJobResultIndex(job, state, new ActionListener<Boolean>(){

            public void onResponse(Boolean indicesCreated) {
                JobManager.this.auditor.info(job.getId(), Messages.getMessage("Job created"));
                JobManager.this.clusterService.submitStateUpdateTask("put-job-" + job.getId(), (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<PutJobAction.Response>((AckedRequest)request, actionListener){

                    protected PutJobAction.Response newResponse(boolean acknowledged) {
                        return new PutJobAction.Response(acknowledged, job);
                    }

                    public ClusterState execute(ClusterState currentState) throws Exception {
                        return JobManager.this.updateClusterState(job, false, currentState);
                    }
                });
            }

            public void onFailure(Exception e) {
                if (e instanceof IllegalArgumentException && e.getMessage().matches("mapper \\[.*\\] of different type, current_type \\[.*\\], merged_type \\[.*\\]")) {
                    actionListener.onFailure((Exception)ExceptionsHelper.badRequestException("A field has a different mapping type to an existing field with the same name. Use the 'results_index_name' setting to assign the job to another index", e, new Object[0]));
                } else {
                    actionListener.onFailure(e);
                }
            }
        });
    }

    public void updateJob(String jobId, JobUpdate jobUpdate, AckedRequest request, ActionListener<PutJobAction.Response> actionListener) {
        Job job = this.getJobOrThrowIfUnknown(jobId);
        this.validate(jobUpdate, job, isValid -> {
            if (isValid.booleanValue()) {
                this.internalJobUpdate(jobId, jobUpdate, request, actionListener);
            } else {
                actionListener.onFailure((Exception)new IllegalArgumentException("Invalid update to job [" + jobId + "]"));
            }
        }, arg_0 -> actionListener.onFailure(arg_0));
    }

    private void validate(JobUpdate jobUpdate, Job job, Consumer<Boolean> handler, Consumer<Exception> errorHandler) {
        if (jobUpdate.getModelSnapshotId() != null) {
            this.jobProvider.getModelSnapshot(job.getId(), jobUpdate.getModelSnapshotId(), newModelSnapshot -> {
                if (newModelSnapshot == null) {
                    String message = Messages.getMessage("No model snapshot with id [{0}] exists for job [{1}]", jobUpdate.getModelSnapshotId(), job.getId());
                    errorHandler.accept((Exception)new ResourceNotFoundException(message, new Object[0]));
                }
                this.jobProvider.getModelSnapshot(job.getId(), job.getModelSnapshotId(), oldModelSnapshot -> {
                    if (oldModelSnapshot != null && newModelSnapshot.getTimestamp().before(oldModelSnapshot.getTimestamp())) {
                        String message = "Job [" + job.getId() + "] has a more recent model snapshot [" + oldModelSnapshot.getSnapshotId() + "]";
                        errorHandler.accept(new IllegalArgumentException(message));
                    }
                    handler.accept(true);
                }, errorHandler);
            }, errorHandler);
        } else {
            handler.accept(true);
        }
    }

    private void internalJobUpdate(final String jobId, final JobUpdate jobUpdate, AckedRequest request, ActionListener<PutJobAction.Response> actionListener) {
        this.clusterService.submitStateUpdateTask("update-job-" + jobId, (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<PutJobAction.Response>(request, actionListener){
            private volatile Job updatedJob;

            protected PutJobAction.Response newResponse(boolean acknowledged) {
                return new PutJobAction.Response(acknowledged, this.updatedJob);
            }

            public ClusterState execute(ClusterState currentState) throws Exception {
                Job job = JobManager.this.getJob(jobId, currentState).results().get(0);
                this.updatedJob = jobUpdate.mergeWithJob(job);
                return JobManager.this.updateClusterState(this.updatedJob, true, currentState);
            }

            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                PersistentTasksCustomMetaData persistentTasks = (PersistentTasksCustomMetaData)newState.metaData().custom("persistent_tasks");
                JobState jobState = MlMetadata.getJobState(jobId, persistentTasks);
                if (jobState == JobState.OPENED) {
                    JobManager.this.updateJobProcessNotifier.submitJobUpdate(jobUpdate);
                }
            }
        });
    }

    ClusterState updateClusterState(Job job, boolean overwrite, ClusterState currentState) {
        MlMetadata.Builder builder = JobManager.createMlMetadataBuilder(currentState);
        builder.putJob(job, overwrite);
        return JobManager.buildNewClusterState(currentState, builder);
    }

    public void deleteJob(DeleteJobAction.Request request, JobStorageDeletionTask task, final ActionListener<DeleteJobAction.Response> actionListener) {
        final String jobId = request.getJobId();
        this.logger.debug("Deleting job '" + jobId + "'");
        CheckedConsumer apiResponseHandler = jobDeleted -> {
            if (jobDeleted.booleanValue()) {
                this.logger.info("Job [" + jobId + "] deleted.");
                this.auditor.info(jobId, Messages.getMessage("Job deleted"));
                actionListener.onResponse((Object)new DeleteJobAction.Response(true));
            } else {
                actionListener.onResponse((Object)new DeleteJobAction.Response(false));
            }
        };
        CheckedConsumer deleteJobStateHandler = response -> this.clusterService.submitStateUpdateTask("delete-job-" + jobId, (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<Boolean>((AckedRequest)request, ActionListener.wrap((CheckedConsumer)apiResponseHandler, arg_0 -> ((ActionListener)actionListener).onFailure(arg_0))){

            protected Boolean newResponse(boolean acknowledged) {
                return acknowledged && response != false;
            }

            public ClusterState execute(ClusterState currentState) throws Exception {
                MlMetadata.Builder builder = JobManager.createMlMetadataBuilder(currentState);
                builder.deleteJob(jobId, (PersistentTasksCustomMetaData)currentState.getMetaData().custom("persistent_tasks"));
                return JobManager.buildNewClusterState(currentState, builder);
            }
        });
        final CheckedConsumer updateHandler = response -> {
            if (response.booleanValue()) {
                this.logger.info("Job [" + jobId + "] is successfully marked as deleted");
            } else {
                this.logger.warn("Job [" + jobId + "] marked as deleted wan't acknowledged");
            }
            task.delete(jobId, this.client, this.clusterService.state(), (CheckedConsumer<Boolean, Exception>)((CheckedConsumer)arg_0 -> ((CheckedConsumer)deleteJobStateHandler).accept(arg_0)), arg_0 -> ((ActionListener)actionListener).onFailure(arg_0));
        };
        this.clusterService.submitStateUpdateTask("mark-job-as-deleted", (ClusterStateTaskConfig)new ClusterStateUpdateTask(){

            public ClusterState execute(ClusterState currentState) throws Exception {
                MlMetadata currentMlMetadata = (MlMetadata)currentState.metaData().custom("ml");
                PersistentTasksCustomMetaData tasks = (PersistentTasksCustomMetaData)currentState.metaData().custom("persistent_tasks");
                MlMetadata.Builder builder = new MlMetadata.Builder(currentMlMetadata);
                builder.markJobAsDeleted(jobId, tasks);
                return JobManager.buildNewClusterState(currentState, builder);
            }

            public void onFailure(String source, Exception e) {
                actionListener.onFailure(e);
            }

            public void clusterStatePublished(ClusterChangedEvent clusterChangedEvent) {
                try {
                    updateHandler.accept((Object)true);
                }
                catch (Exception e) {
                    actionListener.onFailure(e);
                }
            }
        });
    }

    public void revertSnapshot(final RevertModelSnapshotAction.Request request, ActionListener<RevertModelSnapshotAction.Response> actionListener, final ModelSnapshot modelSnapshot) {
        this.clusterService.submitStateUpdateTask("revert-snapshot-" + request.getJobId(), (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<RevertModelSnapshotAction.Response>((AckedRequest)request, actionListener){

            protected RevertModelSnapshotAction.Response newResponse(boolean acknowledged) {
                if (acknowledged) {
                    JobManager.this.auditor.info(request.getJobId(), Messages.getMessage("Job model snapshot reverted to ''{0}''", modelSnapshot.getDescription()));
                    return new RevertModelSnapshotAction.Response(modelSnapshot);
                }
                throw new IllegalStateException("Could not revert modelSnapshot on job [" + request.getJobId() + "], not acknowledged by master.");
            }

            public ClusterState execute(ClusterState currentState) throws Exception {
                Job job = JobManager.getJobOrThrowIfUnknown(currentState, request.getJobId());
                Job.Builder builder = new Job.Builder(job);
                builder.setModelSnapshotId(modelSnapshot.getSnapshotId());
                return JobManager.this.updateClusterState(builder.build(), true, currentState);
            }
        });
    }

    public void updateModelSnapshot(ModelSnapshot modelSnapshot, Consumer<Boolean> handler, Consumer<Exception> errorHandler) {
        String index = AnomalyDetectorsIndex.jobResultsAliasedName(modelSnapshot.getJobId());
        IndexRequest indexRequest = new IndexRequest(index, ModelSnapshot.TYPE.getPreferredName(), ModelSnapshot.documentId(modelSnapshot));
        try (XContentBuilder builder = XContentFactory.jsonBuilder();){
            modelSnapshot.toXContent(builder, ToXContent.EMPTY_PARAMS);
            indexRequest.source(builder);
        }
        catch (IOException e) {
            errorHandler.accept(e);
        }
        this.client.index(indexRequest, ActionListener.wrap(r -> handler.accept(true), errorHandler));
    }

    private static MlMetadata.Builder createMlMetadataBuilder(ClusterState currentState) {
        MlMetadata currentMlMetadata = (MlMetadata)currentState.metaData().custom("ml");
        return new MlMetadata.Builder(currentMlMetadata);
    }

    private static ClusterState buildNewClusterState(ClusterState currentState, MlMetadata.Builder builder) {
        ClusterState.Builder newState = ClusterState.builder((ClusterState)currentState);
        newState.metaData(MetaData.builder((MetaData)currentState.getMetaData()).putCustom("ml", (MetaData.Custom)builder.build()).build());
        return newState.build();
    }
}

