/*
 * Copyright (c) 2019 vitasystems GmbH and Hannover Medical School.
 *
 * This file is part of project openEHR_SDK
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ehrbase.openehr.sdk.serialisation.dto;

import com.nedap.archie.rm.RMObject;
import com.nedap.archie.rm.composition.Composition;
import com.nedap.archie.rm.support.identification.ObjectVersionId;
import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;
import java.util.Optional;
import org.ehrbase.openehr.sdk.generator.commons.annotations.Id;
import org.ehrbase.openehr.sdk.generator.commons.annotations.Template;
import org.ehrbase.openehr.sdk.serialisation.walker.defaultvalues.DefaultValues;
import org.ehrbase.openehr.sdk.util.exception.ClientException;
import org.ehrbase.openehr.sdk.util.exception.SdkException;
import org.ehrbase.openehr.sdk.webtemplate.model.WebTemplate;
import org.ehrbase.openehr.sdk.webtemplate.templateprovider.TemplateProvider;
import org.ehrbase.openehr.sdk.webtemplate.webtemplateskeletonbuilder.WebTemplateSkeletonBuilder;

public class GeneratedDtoToRmConverter {

    private final TemplateProvider templateProvider;
    private final DefaultValuesProvider defaultValuesProvider;

    public GeneratedDtoToRmConverter(TemplateProvider templateProvider, DefaultValuesProvider defaultValuesProvider) {
        this.templateProvider = templateProvider;
        this.defaultValuesProvider = defaultValuesProvider;
    }

    public GeneratedDtoToRmConverter(TemplateProvider templateProvider) {
        this.defaultValuesProvider = o -> new DefaultValues();
        this.templateProvider = templateProvider;
    }

    /**
     * Converts a java bean, usually generated by the ehrbase class generator, to a RMObject
     * The template annotation of the dto class must reference a template available to the templateProvider.
     *
     * @param dto
     * @return
     */
    public RMObject toRMObject(Object dto) {
        Template template = dto.getClass().getAnnotation(Template.class);

        WebTemplate introspect = templateProvider
                .buildIntrospect(template.value())
                .orElseThrow(() -> new SdkException(String.format("Can not find Template: %s", template.value())));

        Composition generate = WebTemplateSkeletonBuilder.build(introspect, false);
        new DtoToCompositionWalker()
                .walk(
                        generate,
                        DtoToCompositionWalker.findEntity(dto),
                        introspect,
                        defaultValuesProvider.provide(dto),
                        template.value());
        Optional<ObjectVersionId> versionUid = extractVersionUid(dto);
        if (versionUid.isPresent()) {
            generate.setUid(new ObjectVersionId(versionUid.get().toString()));
        }
        return generate;
    }

    /**
     * Extracts the {@code ObjectVersionId} of the dto from a property annotated with {@code @Id}
     *
     * @param dto
     * @return
     */
    static Optional<ObjectVersionId> extractVersionUid(Object dto) {
        return Arrays.stream(dto.getClass().getDeclaredFields())
                .filter(f -> f.isAnnotationPresent(Id.class))
                .findAny()
                .map(idField -> {
                    try {
                        PropertyDescriptor propertyDescriptor =
                                new PropertyDescriptor(idField.getName(), dto.getClass());
                        return (ObjectVersionId)
                                propertyDescriptor.getReadMethod().invoke(dto);
                    } catch (IllegalAccessException | InvocationTargetException | IntrospectionException e) {
                        throw new ClientException(e.getMessage(), e);
                    }
                });
    }
}
