/*
 * Decompiled with CFR 0.152.
 */
package org.terracotta.runnel.utils;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UTFDataFormatException;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ReadOnlyBufferException;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Objects;

public final class StringTool {
    private static final ThreadLocal<CharsetDecoder> US_ASCII_THREAD_LOCAL = ThreadLocal.withInitial(() -> {
        CharsetDecoder cs = StandardCharsets.US_ASCII.newDecoder().onMalformedInput(CodingErrorAction.REPORT).onUnmappableCharacter(CodingErrorAction.REPORT);
        return cs;
    });
    private static final int MAX_SLICE_LENGTH = 512;

    private StringTool() {
    }

    public static String getUTF(ByteBuffer buffer) throws UTFDataFormatException, BufferUnderflowException {
        Objects.requireNonNull(buffer, "buffer");
        int encodingType = Byte.toUnsignedInt(buffer.get());
        switch (encodingType) {
            case 0: {
                return StringTool.decodeString(buffer, Short.toUnsignedLong(buffer.getShort()));
            }
            case 1: {
                long encodedLength = buffer.getLong();
                if (encodedLength / 3L > Integer.MAX_VALUE) {
                    throw new UTFDataFormatException("Encoded length larger than supported: " + encodedLength);
                }
                return StringTool.decodeString(buffer, encodedLength);
            }
            case 2: {
                long encodedLength = buffer.getLong();
                long charLength = encodedLength / 2L;
                if (charLength != (long)((int)charLength)) {
                    throw new UTFDataFormatException("Encoded length larger than supported: " + encodedLength);
                }
                return StringTool.getCharsAsString(buffer, (int)charLength);
            }
            case 3: {
                return StringTool.getBytesAsString(buffer, Short.toUnsignedInt(buffer.getShort()));
            }
            case 4: {
                return StringTool.getBytesAsString(buffer, buffer.getInt());
            }
        }
        throw new UTFDataFormatException("Unexpected encoding type: 0x" + Integer.toHexString(encodingType));
    }

    public static String decodeString(ByteBuffer buffer, long encodedLength) throws UTFDataFormatException, BufferUnderflowException {
        int initialArrayLength = (int)encodedLength;
        char[] chars = new char[initialArrayLength];
        int charIndex = -1;
        long remaining = encodedLength;
        while (remaining > 0L) {
            int b;
            int f;
            if (++charIndex == chars.length) {
                chars = StringTool.enlargeChars(encodedLength, chars);
            }
            if ((f = (b = Byte.toUnsignedInt(buffer.get())) >>> 4) < 8) {
                chars[charIndex] = (char)b;
                --remaining;
                continue;
            }
            if (f == 14) {
                chars[charIndex] = (char)((b & 0xF) << 12 | (buffer.get() & 0x3F) << 6 | buffer.get() & 0x3F);
                remaining -= 3L;
                continue;
            }
            if (f >= 12) {
                chars[charIndex] = (char)((b & 0x1F) << 6 | buffer.get() & 0x3F);
                remaining -= 2L;
                continue;
            }
            throw new UTFDataFormatException(String.format("Illegal element: %02x", b));
        }
        if (remaining != 0L) {
            throw new UTFDataFormatException(String.format("Decoding string: %d bytes remaining after decoding", remaining));
        }
        return new String(chars, 0, 1 + charIndex);
    }

    private static char[] enlargeChars(long encodedLength, char[] chars) {
        long newLength = (long)chars.length + (long)((int)((encodedLength + 3L) / 4L));
        if (newLength != (long)((int)newLength)) {
            throw new OutOfMemoryError();
        }
        return Arrays.copyOf(chars, (int)newLength);
    }

    private static String getCharsAsString(ByteBuffer buffer, int charLength) {
        char[] chars = new char[charLength];
        for (int i = 0; i < charLength; ++i) {
            chars[i] = buffer.getChar();
        }
        return new String(chars);
    }

    private static String getBytesAsString(ByteBuffer buffer, int charLength) {
        char[] chars = new char[charLength];
        for (int i = 0; i < charLength; ++i) {
            chars[i] = (char)Byte.toUnsignedInt(buffer.get());
        }
        return new String(chars);
    }

    public static void putUTF(ByteBuffer buffer, String str) throws ReadOnlyBufferException, BufferOverflowException {
        Objects.requireNonNull(buffer, "buffer");
        Objects.requireNonNull(str, "str");
        long rawEncodedLength = StringTool.getEncodedLen(str);
        int strLength = str.length();
        if (rawEncodedLength == (long)strLength) {
            StringTool.putASCII(buffer, str, rawEncodedLength, strLength);
            return;
        }
        if (rawEncodedLength <= 65535L) {
            buffer.put((byte)0).putShort((short)rawEncodedLength);
        } else if (rawEncodedLength < (long)strLength * 2L) {
            buffer.put((byte)1).putLong(rawEncodedLength);
        } else {
            StringTool.putNonEncoded(buffer, str, strLength);
            return;
        }
        if (rawEncodedLength > (long)buffer.remaining()) {
            throw new BufferOverflowException();
        }
        StringTool.putEncoded(buffer, str, strLength);
    }

    public static void putEncoded(ByteBuffer buffer, String str, int strLength) throws BufferOverflowException, ReadOnlyBufferException {
        char[] slice = new char[512];
        int sz = 0;
        for (int offset = 0; offset < strLength; offset += 512) {
            int sliceLength = Math.min(512, strLength - offset);
            str.getChars(offset, offset + sliceLength, slice, 0);
            for (int i = 0; i < sliceLength; ++i) {
                char c = slice[i];
                if (c <= '\u007f' && c != '\u0000') {
                    buffer.put((byte)c);
                    ++sz;
                    continue;
                }
                if (c <= '\u07ff') {
                    buffer.put((byte)(0xC0 | c >>> 6)).put((byte)(0x80 | c & 0x3F));
                    sz += 2;
                    continue;
                }
                buffer.put((byte)(0xE0 | c >>> 12)).put((byte)(0x80 | c >>> 6 & 0x3F)).put((byte)(0x80 | c & 0x3F));
                sz += 3;
            }
        }
    }

    public static int putEncoded(OutputStream os, String str, int strLength) throws IOException {
        char[] slice = new char[512];
        byte[] tmp = new byte[1539];
        int sz = 0;
        for (int offset = 0; offset < strLength; offset += 512) {
            int sliceLength = Math.min(512, strLength - offset);
            str.getChars(offset, offset + sliceLength, slice, 0);
            int tmpCount = 0;
            for (int i = 0; i < sliceLength; ++i) {
                char c = slice[i];
                if (c <= '\u007f' && c != '\u0000') {
                    tmp[tmpCount++] = (byte)c;
                    continue;
                }
                if (c <= '\u07ff') {
                    tmp[tmpCount++] = (byte)(0xC0 | c >>> 6);
                    tmp[tmpCount++] = (byte)(0x80 | c & 0x3F);
                    continue;
                }
                tmp[tmpCount++] = (byte)(0xE0 | c >>> 12);
                tmp[tmpCount++] = (byte)(0x80 | c >>> 6 & 0x3F);
                tmp[tmpCount++] = (byte)(0x80 | c & 0x3F);
            }
            if (tmpCount <= 0) continue;
            os.write(tmp, 0, tmpCount);
            sz += tmpCount;
        }
        return sz;
    }

    private static void putNonEncoded(ByteBuffer buffer, String str, int strLength) throws BufferOverflowException, ReadOnlyBufferException {
        int byteLength = strLength * 2;
        buffer.put((byte)2).putLong(byteLength);
        if (byteLength > buffer.remaining()) {
            throw new BufferOverflowException();
        }
        char[] slice = new char[512];
        for (int offset = 0; offset < strLength; offset += 512) {
            int sliceLength = Math.min(512, strLength - offset);
            str.getChars(offset, offset + sliceLength, slice, 0);
            for (int i = 0; i < sliceLength; ++i) {
                buffer.putChar(slice[i]);
            }
        }
    }

    private static void putASCII(ByteBuffer buffer, String str, long rawEncodedLength, int strLength) throws BufferOverflowException, ReadOnlyBufferException {
        if (rawEncodedLength <= 65535L) {
            buffer.put((byte)3).putShort((short)rawEncodedLength);
        } else {
            buffer.put((byte)4).putInt((int)rawEncodedLength);
        }
        if (rawEncodedLength > (long)buffer.remaining()) {
            throw new BufferOverflowException();
        }
        for (int i = 0; i < strLength; ++i) {
            buffer.put((byte)str.charAt(i));
        }
    }

    public static int getLengthAsUTF(String str) {
        Objects.requireNonNull(str, "str");
        long len = StringTool.getEncodedLen(str);
        int strLength = str.length();
        if (len == (long)strLength) {
            len += (long)(1 + (len <= 65535L ? 2 : 4));
        } else if (len <= 65535L) {
            len += 3L;
        } else if (len < (long)strLength * 2L) {
            len += 9L;
        } else {
            len = (long)strLength * 2L;
            len += 9L;
        }
        if (len > Integer.MAX_VALUE) {
            throw new IllegalStateException("Encoded length greater than Integer.MAX_VALUE: " + len);
        }
        return (int)len;
    }

    private static long getEncodedLen(String str) {
        int strLength = str.length();
        long len = strLength;
        for (int i = 0; i < strLength; ++i) {
            char c = str.charAt(i);
            if (c < '\u0080' && c != '\u0000') continue;
            if (c < '\u0800') {
                ++len;
                continue;
            }
            len += 2L;
        }
        return len;
    }

    public static int worstCaseByteArraySize(String str) {
        return str.length() * 4 + 8;
    }

    public static String attemptDecodeAsAscii(ByteBuffer binary) {
        int start = binary.position();
        try {
            CharsetDecoder cs = US_ASCII_THREAD_LOCAL.get();
            cs.reset();
            return cs.decode(binary).toString();
        }
        catch (CharacterCodingException e) {
            binary.position(start);
            return null;
        }
    }
}

