/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.rdf4j.sail.nativerdf.datastore;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.WritableByteChannel;
import java.util.Arrays;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.eclipse.rdf4j.common.io.NioFile;

public class HashFile
implements Closeable {
    private static final int ITEM_SIZE = 8;
    private static final byte[] MAGIC_NUMBER = new byte[]{110, 104, 102};
    private static final byte FILE_FORMAT_VERSION = 1;
    private static final long HEADER_LENGTH = 16L;
    private static final int INIT_BUCKET_COUNT = 64;
    private static final int INIT_BUCKET_SIZE = 8;
    private final NioFile nioFile;
    private final boolean forceSync;
    private volatile int bucketCount;
    private final int bucketSize;
    private volatile int itemCount;
    private final float loadFactor = 0.75f;
    private final int recordSize;
    private final ReentrantReadWriteLock structureLock = new ReentrantReadWriteLock();

    public HashFile(File file) throws IOException {
        this(file, false);
    }

    public HashFile(File file, boolean forceSync) throws IOException {
        this.nioFile = new NioFile(file);
        this.forceSync = forceSync;
        try {
            if (this.nioFile.size() == 0L) {
                this.bucketCount = 64;
                this.bucketSize = 8;
                this.itemCount = 0;
                this.recordSize = 8 * this.bucketSize + 4;
                this.writeEmptyBuckets(16L, this.bucketCount);
                this.sync();
            } else {
                ByteBuffer buf = ByteBuffer.allocate(16);
                this.nioFile.read(buf, 0L);
                buf.rewind();
                if ((long)buf.remaining() < 16L) {
                    throw new IOException("File too short to be a compatible hash file");
                }
                byte[] magicNumber = new byte[MAGIC_NUMBER.length];
                buf.get(magicNumber);
                byte version = buf.get();
                this.bucketCount = buf.getInt();
                this.bucketSize = buf.getInt();
                this.itemCount = buf.getInt();
                if (!Arrays.equals(MAGIC_NUMBER, magicNumber)) {
                    throw new IOException("File doesn't contain compatible hash file data");
                }
                if (version > 1) {
                    throw new IOException("Unable to read hash file; it uses a newer file format");
                }
                if (version != 1) {
                    throw new IOException("Unable to read hash file; invalid file format version: " + version);
                }
                this.recordSize = 8 * this.bucketSize + 4;
            }
        }
        catch (IOException e) {
            this.nioFile.close();
            throw e;
        }
    }

    public File getFile() {
        return this.nioFile.getFile();
    }

    public int getItemCount() {
        return this.itemCount;
    }

    public IDIterator getIDIterator(int hash) throws IOException {
        return new IDIterator(hash);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void storeID(int hash, int id) throws IOException {
        int n;
        this.structureLock.readLock().lock();
        try {
            long bucketOffset = this.getBucketOffset(hash);
            this.storeID(bucketOffset, hash, id);
        }
        finally {
            this.structureLock.readLock().unlock();
        }
        ++this.itemCount;
        if ((float)n >= 0.75f * (float)this.bucketCount * (float)this.bucketSize) {
            this.structureLock.writeLock().lock();
            try {
                this.increaseHashTable();
            }
            finally {
                this.structureLock.writeLock().unlock();
            }
        }
    }

    private void storeID(long bucketOffset, int hash, int id) throws IOException {
        boolean idStored = false;
        ByteBuffer bucket = ByteBuffer.allocate(this.recordSize);
        while (!idStored) {
            this.nioFile.read(bucket, bucketOffset);
            int slotID = this.findEmptySlotInBucket(bucket);
            if (slotID >= 0) {
                bucket.putInt(8 * slotID, hash);
                bucket.putInt(8 * slotID + 4, id);
                bucket.rewind();
                this.nioFile.write(bucket, bucketOffset);
                idStored = true;
                continue;
            }
            int overflowID = bucket.getInt(8 * this.bucketSize);
            if (overflowID == 0) {
                overflowID = this.createOverflowBucket();
                bucket.putInt(8 * this.bucketSize, overflowID);
                bucket.rewind();
                this.nioFile.write(bucket, bucketOffset);
            }
            bucketOffset = this.getOverflowBucketOffset(overflowID);
            bucket.clear();
        }
    }

    public void clear() throws IOException {
        this.structureLock.writeLock().lock();
        try {
            this.nioFile.truncate(16L + (long)this.bucketCount * (long)this.recordSize);
            this.writeEmptyBuckets(16L, this.bucketCount);
            this.itemCount = 0;
        }
        finally {
            this.structureLock.writeLock().unlock();
        }
    }

    public void sync() throws IOException {
        this.structureLock.readLock().lock();
        try {
            this.writeFileHeader();
        }
        finally {
            this.structureLock.readLock().unlock();
        }
        if (this.forceSync) {
            this.nioFile.force(false);
        }
    }

    @Override
    public void close() throws IOException {
        this.nioFile.close();
    }

    private RandomAccessFile createEmptyFile(File file) throws IOException {
        boolean created;
        if (!file.exists() && !(created = file.createNewFile())) {
            throw new IOException("Failed to create file " + file);
        }
        RandomAccessFile raf = new RandomAccessFile(file, "rw");
        raf.setLength(0L);
        return raf;
    }

    private void writeFileHeader() throws IOException {
        ByteBuffer buf = ByteBuffer.allocate(16);
        buf.put(MAGIC_NUMBER);
        buf.put((byte)1);
        buf.putInt(this.bucketCount);
        buf.putInt(this.bucketSize);
        buf.putInt(this.itemCount);
        buf.rewind();
        this.nioFile.write(buf, 0L);
    }

    private long getBucketOffset(int hash) {
        int bucketNo = hash % this.bucketCount;
        if (bucketNo < 0) {
            bucketNo += this.bucketCount;
        }
        return 16L + (long)bucketNo * (long)this.recordSize;
    }

    private long getOverflowBucketOffset(int bucketID) {
        return 16L + ((long)this.bucketCount + (long)bucketID - 1L) * (long)this.recordSize;
    }

    private int createOverflowBucket() throws IOException {
        long offset = this.nioFile.size();
        this.writeEmptyBuckets(offset, 1);
        return (int)((offset - 16L) / (long)this.recordSize) - this.bucketCount + 1;
    }

    private void writeEmptyBuckets(long fileOffset, int bucketCount) throws IOException {
        ByteBuffer emptyBucket = ByteBuffer.allocate(this.recordSize);
        for (int i = 0; i < bucketCount; ++i) {
            this.nioFile.write(emptyBucket, fileOffset + (long)i * (long)this.recordSize);
            emptyBucket.rewind();
        }
    }

    private int findEmptySlotInBucket(ByteBuffer bucket) {
        for (int slotNo = 0; slotNo < this.bucketSize; ++slotNo) {
            if (bucket.getInt(8 * slotNo + 4) != 0) continue;
            return slotNo;
        }
        return -1;
    }

    private void increaseHashTable() throws IOException {
        long oldTableSize = 16L + (long)this.bucketCount * (long)this.recordSize;
        long newTableSize = 16L + (long)this.bucketCount * (long)this.recordSize * 2L;
        long oldFileSize = this.nioFile.size();
        File tmpFile = new File(this.getFile().getParentFile(), "rehash_" + this.getFile().getName());
        try (RandomAccessFile tmpRaf = this.createEmptyFile(tmpFile);){
            FileChannel tmpChannel = tmpRaf.getChannel();
            this.nioFile.transferTo(oldTableSize, oldFileSize - oldTableSize, (WritableByteChannel)tmpChannel);
            this.writeEmptyBuckets(oldTableSize, this.bucketCount);
            this.bucketCount *= 2;
            this.nioFile.truncate(newTableSize);
            ByteBuffer bucket = ByteBuffer.allocate(this.recordSize);
            ByteBuffer newBucket = ByteBuffer.allocate(this.recordSize);
            for (long bucketOffset = 16L; bucketOffset < oldTableSize; bucketOffset += (long)this.recordSize) {
                this.nioFile.read(bucket, bucketOffset);
                boolean bucketChanged = false;
                long newBucketOffset = 0L;
                for (int slotNo = 0; slotNo < this.bucketSize; ++slotNo) {
                    int hash;
                    long newOffset;
                    int id = bucket.getInt(8 * slotNo + 4);
                    if (id == 0 || (newOffset = this.getBucketOffset(hash = bucket.getInt(8 * slotNo))) == bucketOffset) continue;
                    newBucket.putInt(hash);
                    newBucket.putInt(id);
                    bucket.putInt(8 * slotNo, 0);
                    bucket.putInt(8 * slotNo + 4, 0);
                    bucketChanged = true;
                    newBucketOffset = newOffset;
                }
                if (bucketChanged) {
                    newBucket.flip();
                    this.nioFile.write(newBucket, newBucketOffset);
                    newBucket.clear();
                }
                if (bucket.getInt(8 * this.bucketSize) != 0) {
                    bucket.putInt(8 * this.bucketSize, 0);
                    bucketChanged = true;
                }
                if (bucketChanged) {
                    bucket.rewind();
                    this.nioFile.write(bucket, bucketOffset);
                }
                bucket.clear();
            }
            long tmpFileSize = tmpChannel.size();
            for (long bucketOffset = 0L; bucketOffset < tmpFileSize; bucketOffset += (long)this.recordSize) {
                tmpChannel.read(bucket, bucketOffset);
                for (int slotNo = 0; slotNo < this.bucketSize; ++slotNo) {
                    int id = bucket.getInt(8 * slotNo + 4);
                    if (id == 0) continue;
                    int hash = bucket.getInt(8 * slotNo);
                    long newBucketOffset = this.getBucketOffset(hash);
                    this.storeID(newBucketOffset, hash, id);
                }
                bucket.clear();
            }
        }
        tmpFile.delete();
    }

    public class IDIterator {
        private final int queryHash;
        private ByteBuffer bucketBuffer;
        private int slotNo;

        private IDIterator(int hash) throws IOException {
            this.queryHash = hash;
            this.bucketBuffer = ByteBuffer.allocate(HashFile.this.recordSize);
            HashFile.this.structureLock.readLock().lock();
            try {
                long bucketOffset = HashFile.this.getBucketOffset(hash);
                HashFile.this.nioFile.read(this.bucketBuffer, bucketOffset);
                this.slotNo = -1;
            }
            catch (IOException | RuntimeException e) {
                HashFile.this.structureLock.readLock().unlock();
                throw e;
            }
        }

        public void close() {
            this.bucketBuffer = null;
            HashFile.this.structureLock.readLock().unlock();
        }

        public int next() throws IOException {
            while (this.bucketBuffer != null) {
                while (++this.slotNo < HashFile.this.bucketSize) {
                    if (this.bucketBuffer.getInt(8 * this.slotNo) != this.queryHash) continue;
                    return this.bucketBuffer.getInt(8 * this.slotNo + 4);
                }
                int overflowID = this.bucketBuffer.getInt(8 * HashFile.this.bucketSize);
                if (overflowID == 0) {
                    this.bucketBuffer = null;
                    break;
                }
                this.bucketBuffer.clear();
                long bucketOffset = HashFile.this.getOverflowBucketOffset(overflowID);
                HashFile.this.nioFile.read(this.bucketBuffer, bucketOffset);
                this.slotNo = -1;
            }
            return -1;
        }
    }
}

