/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.server;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import javax.servlet.ServletException;
import javax.servlet.ServletInputStream;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.eclipse.jetty.io.NetworkTrafficListener;
import org.eclipse.jetty.server.Connector;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.NetworkTrafficServerConnector;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.handler.AbstractHandler;
import org.eclipse.jetty.util.BufferUtil;
import org.junit.After;
import org.junit.Assert;
import org.junit.Test;

public class NetworkTrafficListenerTest {
    private static final byte END_OF_CONTENT = 126;
    private Server server;
    private NetworkTrafficServerConnector connector;

    public void initConnector(Handler handler) throws Exception {
        this.server = new Server();
        this.connector = new NetworkTrafficServerConnector(this.server);
        ((HttpConfiguration.ConnectionFactory)this.connector.getConnectionFactory(HttpConfiguration.ConnectionFactory.class)).getHttpConfiguration().setSendDateHeader(false);
        ((HttpConfiguration.ConnectionFactory)this.connector.getConnectionFactory(HttpConfiguration.ConnectionFactory.class)).getHttpConfiguration().setSendServerVersion(false);
        this.server.addConnector((Connector)this.connector);
        this.server.setHandler(handler);
        this.server.start();
    }

    @After
    public void destroyConnector() throws Exception {
        if (this.server != null) {
            this.server.stop();
            this.server.join();
        }
    }

    @Test
    public void testOpenedClosedAreInvoked() throws Exception {
        this.initConnector(null);
        final CountDownLatch openedLatch = new CountDownLatch(1);
        final CountDownLatch closedLatch = new CountDownLatch(1);
        this.connector.addNetworkTrafficListener((NetworkTrafficListener)new NetworkTrafficListener.Adapter(){
            public volatile Socket socket;

            public void opened(Socket socket) {
                this.socket = socket;
                openedLatch.countDown();
            }

            public void closed(Socket socket) {
                if (this.socket == socket) {
                    closedLatch.countDown();
                }
            }
        });
        int port = this.connector.getLocalPort();
        Socket socket = new Socket("localhost", port);
        Assert.assertTrue((boolean)openedLatch.await(10L, TimeUnit.SECONDS));
        socket.close();
        Assert.assertTrue((boolean)closedLatch.await(10L, TimeUnit.SECONDS));
    }

    @Test
    public void testTrafficWithNoResponseContentOnNonPersistentConnection() throws Exception {
        this.initConnector((Handler)new AbstractHandler(){

            public void handle(String uri, Request request, HttpServletRequest servletRequest, HttpServletResponse servletResponse) throws IOException, ServletException {
                request.setHandled(true);
            }
        });
        final AtomicReference incomingData = new AtomicReference();
        final CountDownLatch incomingLatch = new CountDownLatch(1);
        final AtomicReference<String> outgoingData = new AtomicReference<String>("");
        final CountDownLatch outgoingLatch = new CountDownLatch(1);
        this.connector.addNetworkTrafficListener((NetworkTrafficListener)new NetworkTrafficListener.Adapter(){

            public void incoming(Socket socket, ByteBuffer bytes) {
                incomingData.set(BufferUtil.toString((ByteBuffer)bytes, (Charset)StandardCharsets.UTF_8));
                incomingLatch.countDown();
            }

            public void outgoing(Socket socket, ByteBuffer bytes) {
                outgoingData.set((String)outgoingData.get() + BufferUtil.toString((ByteBuffer)bytes, (Charset)StandardCharsets.UTF_8));
                outgoingLatch.countDown();
            }
        });
        int port = this.connector.getLocalPort();
        String request = "GET / HTTP/1.1\r\nHost: localhost:" + port + "\r\n" + "Connection: close\r\n" + "\r\n";
        String expectedResponse = "HTTP/1.1 200 OK\r\nConnection: close\r\n\r\n";
        Socket socket = new Socket("localhost", port);
        OutputStream output = socket.getOutputStream();
        output.write(request.getBytes(StandardCharsets.UTF_8));
        output.flush();
        Assert.assertTrue((boolean)incomingLatch.await(1L, TimeUnit.SECONDS));
        Assert.assertEquals((Object)request, incomingData.get());
        Assert.assertTrue((boolean)outgoingLatch.await(1L, TimeUnit.SECONDS));
        Assert.assertEquals((Object)expectedResponse, (Object)outgoingData.get());
        byte[] responseBytes = this.readResponse(socket);
        String response = new String(responseBytes, StandardCharsets.UTF_8);
        Assert.assertEquals((Object)expectedResponse, (Object)response);
        socket.close();
    }

    @Test
    public void testTrafficWithResponseContentOnPersistentConnection() throws Exception {
        String responseContent = "response_content";
        this.initConnector((Handler)new AbstractHandler(){

            public void handle(String uri, Request request, HttpServletRequest servletRequest, HttpServletResponse servletResponse) throws IOException, ServletException {
                request.setHandled(true);
                ServletOutputStream output = servletResponse.getOutputStream();
                output.write("response_content".getBytes(StandardCharsets.UTF_8));
                output.write(126);
            }
        });
        final AtomicReference incomingData = new AtomicReference();
        final CountDownLatch incomingLatch = new CountDownLatch(1);
        final AtomicReference<String> outgoingData = new AtomicReference<String>("");
        final CountDownLatch outgoingLatch = new CountDownLatch(2);
        this.connector.addNetworkTrafficListener((NetworkTrafficListener)new NetworkTrafficListener.Adapter(){

            public void incoming(Socket socket, ByteBuffer bytes) {
                incomingData.set(BufferUtil.toString((ByteBuffer)bytes, (Charset)StandardCharsets.UTF_8));
                incomingLatch.countDown();
            }

            public void outgoing(Socket socket, ByteBuffer bytes) {
                outgoingData.set((String)outgoingData.get() + BufferUtil.toString((ByteBuffer)bytes, (Charset)StandardCharsets.UTF_8));
                outgoingLatch.countDown();
            }
        });
        int port = this.connector.getLocalPort();
        String request = "GET / HTTP/1.1\r\nHost: localhost:" + port + "\r\n" + "\r\n";
        String expectedResponse = "HTTP/1.1 200 OK\r\nContent-Length: " + ("response_content".length() + 1) + "\r\n" + "\r\n" + "" + "response_content" + '~';
        Socket socket = new Socket("localhost", port);
        OutputStream output = socket.getOutputStream();
        output.write(request.getBytes(StandardCharsets.UTF_8));
        output.flush();
        Assert.assertTrue((boolean)incomingLatch.await(1L, TimeUnit.SECONDS));
        Assert.assertEquals((Object)request, incomingData.get());
        Assert.assertTrue((boolean)outgoingLatch.await(1L, TimeUnit.SECONDS));
        Assert.assertEquals((Object)expectedResponse, (Object)outgoingData.get());
        byte[] responseBytes = this.readResponse(socket);
        String response = new String(responseBytes, StandardCharsets.UTF_8);
        Assert.assertEquals((Object)expectedResponse, (Object)response);
        socket.close();
    }

    @Test
    public void testTrafficWithResponseContentChunkedOnPersistentConnection() throws Exception {
        String responseContent = "response_content";
        final String responseChunk1 = "response_content".substring(0, "response_content".length() / 2);
        final String responseChunk2 = "response_content".substring("response_content".length() / 2, "response_content".length());
        this.initConnector((Handler)new AbstractHandler(){

            public void handle(String uri, Request request, HttpServletRequest servletRequest, HttpServletResponse servletResponse) throws IOException, ServletException {
                request.setHandled(true);
                ServletOutputStream output = servletResponse.getOutputStream();
                output.write(responseChunk1.getBytes(StandardCharsets.UTF_8));
                output.flush();
                output.write(responseChunk2.getBytes(StandardCharsets.UTF_8));
                output.flush();
            }
        });
        final AtomicReference incomingData = new AtomicReference();
        final CountDownLatch incomingLatch = new CountDownLatch(1);
        final AtomicReference<String> outgoingData = new AtomicReference<String>("");
        final CountDownLatch outgoingLatch = new CountDownLatch(1);
        this.connector.addNetworkTrafficListener((NetworkTrafficListener)new NetworkTrafficListener.Adapter(){

            public void incoming(Socket socket, ByteBuffer bytes) {
                incomingData.set(BufferUtil.toString((ByteBuffer)bytes, (Charset)StandardCharsets.UTF_8));
                incomingLatch.countDown();
            }

            public void outgoing(Socket socket, ByteBuffer bytes) {
                outgoingData.set((String)outgoingData.get() + BufferUtil.toString((ByteBuffer)bytes, (Charset)StandardCharsets.UTF_8));
                if (((String)outgoingData.get()).endsWith("\r\n0\r\n\r\n")) {
                    outgoingLatch.countDown();
                }
            }
        });
        int port = this.connector.getLocalPort();
        String request = "GET / HTTP/1.1\r\nHost: localhost:" + port + "\r\n" + "\r\n";
        String expectedResponse = "HTTP/1.1 200 OK\r\nTransfer-Encoding: chunked\r\n\r\n" + responseChunk1.length() + "\r\n" + responseChunk1 + "\r\n" + responseChunk2.length() + "\r\n" + responseChunk2 + "\r\n" + "0\r\n" + "\r\n";
        Socket socket = new Socket("localhost", port);
        OutputStream output = socket.getOutputStream();
        output.write(request.getBytes(StandardCharsets.UTF_8));
        output.flush();
        Assert.assertTrue((boolean)incomingLatch.await(1L, TimeUnit.SECONDS));
        Assert.assertEquals((Object)request, incomingData.get());
        Assert.assertTrue((boolean)outgoingLatch.await(1L, TimeUnit.SECONDS));
        Assert.assertEquals((Object)expectedResponse, (Object)outgoingData.get());
        byte[] responseBytes = this.readResponse(socket);
        String response = new String(responseBytes, StandardCharsets.UTF_8);
        Assert.assertEquals((Object)expectedResponse, (Object)response);
        socket.close();
    }

    @Test
    public void testTrafficWithRequestContentWithResponseRedirectOnPersistentConnection() throws Exception {
        String location = "/redirect";
        this.initConnector((Handler)new AbstractHandler(){

            public void handle(String uri, Request request, HttpServletRequest servletRequest, HttpServletResponse servletResponse) throws IOException, ServletException {
                request.setHandled(true);
                servletResponse.sendRedirect("/redirect");
            }
        });
        final AtomicReference incomingData = new AtomicReference();
        final CountDownLatch incomingLatch = new CountDownLatch(1);
        final AtomicReference<String> outgoingData = new AtomicReference<String>("");
        final CountDownLatch outgoingLatch = new CountDownLatch(1);
        this.connector.addNetworkTrafficListener((NetworkTrafficListener)new NetworkTrafficListener.Adapter(){

            public void incoming(Socket socket, ByteBuffer bytes) {
                incomingData.set(BufferUtil.toString((ByteBuffer)bytes, (Charset)StandardCharsets.UTF_8));
                incomingLatch.countDown();
            }

            public void outgoing(Socket socket, ByteBuffer bytes) {
                outgoingData.set((String)outgoingData.get() + BufferUtil.toString((ByteBuffer)bytes, (Charset)StandardCharsets.UTF_8));
                outgoingLatch.countDown();
            }
        });
        int port = this.connector.getLocalPort();
        String requestContent = "a=1&b=2";
        String request = "POST / HTTP/1.1\r\nHost: localhost:" + port + "\r\n" + "Content-Type: application/x-www-form-urlencoded\r\n" + "Content-Length: " + requestContent.length() + "\r\n" + "\r\n" + requestContent;
        String expectedResponse = "HTTP/1.1 302 Found\r\nLocation: http://localhost:" + port + "/redirect" + "\r\n" + "Content-Length: 0\r\n" + "\r\n";
        Socket socket = new Socket("localhost", port);
        OutputStream output = socket.getOutputStream();
        output.write(request.getBytes(StandardCharsets.UTF_8));
        output.flush();
        Assert.assertTrue((boolean)incomingLatch.await(1L, TimeUnit.SECONDS));
        Assert.assertEquals((Object)request, incomingData.get());
        Assert.assertTrue((boolean)outgoingLatch.await(1L, TimeUnit.SECONDS));
        Assert.assertEquals((Object)expectedResponse, (Object)outgoingData.get());
        byte[] responseBytes = this.readResponse(socket);
        String response = new String(responseBytes, StandardCharsets.UTF_8);
        Assert.assertEquals((Object)expectedResponse, (Object)response);
        socket.close();
    }

    @Test
    public void testTrafficWithBigRequestContentOnPersistentConnection() throws Exception {
        this.initConnector((Handler)new AbstractHandler(){

            public void handle(String uri, Request request, HttpServletRequest servletRequest, HttpServletResponse servletResponse) throws IOException, ServletException {
                int read;
                ServletInputStream input = servletRequest.getInputStream();
                byte[] buffer = new byte[4096];
                while ((read = input.read(buffer)) >= 0) {
                }
                request.setHandled(true);
            }
        });
        final AtomicReference<String> incomingData = new AtomicReference<String>("");
        final AtomicReference<String> outgoingData = new AtomicReference<String>("");
        final CountDownLatch outgoingLatch = new CountDownLatch(1);
        this.connector.addNetworkTrafficListener((NetworkTrafficListener)new NetworkTrafficListener.Adapter(){

            public void incoming(Socket socket, ByteBuffer bytes) {
                incomingData.set((String)incomingData.get() + BufferUtil.toString((ByteBuffer)bytes, (Charset)StandardCharsets.UTF_8));
            }

            public void outgoing(Socket socket, ByteBuffer bytes) {
                outgoingData.set((String)outgoingData.get() + BufferUtil.toString((ByteBuffer)bytes, (Charset)StandardCharsets.UTF_8));
                outgoingLatch.countDown();
            }
        });
        int port = this.connector.getLocalPort();
        String requestContent = "0123456789ABCDEF";
        for (int i = 0; i < 11; ++i) {
            requestContent = requestContent + requestContent;
        }
        String request = "POST / HTTP/1.1\r\nHost: localhost:" + port + "\r\n" + "Content-Type: text/plain\r\n" + "Content-Length: " + requestContent.length() + "\r\n" + "\r\n" + requestContent;
        String expectedResponse = "HTTP/1.1 200 OK\r\nContent-Length: 0\r\n\r\n";
        Socket socket = new Socket("localhost", port);
        OutputStream output = socket.getOutputStream();
        output.write(request.getBytes(StandardCharsets.UTF_8));
        output.flush();
        Assert.assertTrue((boolean)outgoingLatch.await(1L, TimeUnit.SECONDS));
        Assert.assertEquals((Object)expectedResponse, (Object)outgoingData.get());
        byte[] responseBytes = this.readResponse(socket);
        String response = new String(responseBytes, StandardCharsets.UTF_8);
        Assert.assertEquals((Object)expectedResponse, (Object)response);
        Assert.assertEquals((Object)request, (Object)incomingData.get());
        socket.close();
    }

    private byte[] readResponse(Socket socket) throws IOException {
        int read;
        socket.setSoTimeout(5000);
        InputStream input = socket.getInputStream();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        while ((read = input.read()) >= 0) {
            String response;
            baos.write(read);
            if (read != 126 && !(response = baos.toString("UTF-8")).endsWith("\r\n0\r\n\r\n") && (!response.contains("Content-Length: 0") || !response.endsWith("\r\n\r\n"))) continue;
            break;
        }
        return baos.toByteArray();
    }
}

