/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.deviceregistry.mongodb.model;

import io.opentracing.Span;
import io.opentracing.SpanContext;
import io.opentracing.Tracer;
import io.opentracing.tag.Tag;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import io.vertx.core.json.JsonObject;
import io.vertx.ext.healthchecks.HealthCheckHandler;
import io.vertx.ext.healthchecks.Status;
import io.vertx.ext.mongo.FindOptions;
import io.vertx.ext.mongo.IndexOptions;
import io.vertx.ext.mongo.MongoClient;
import io.vertx.ext.mongo.UpdateOptions;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import javax.security.auth.x500.X500Principal;
import org.eclipse.hono.client.ClientErrorException;
import org.eclipse.hono.deviceregistry.mongodb.model.MongoDbBasedDao;
import org.eclipse.hono.deviceregistry.mongodb.model.TenantDao;
import org.eclipse.hono.deviceregistry.mongodb.utils.MongoDbDocumentBuilder;
import org.eclipse.hono.service.HealthCheckProvider;
import org.eclipse.hono.service.management.Filter;
import org.eclipse.hono.service.management.SearchResult;
import org.eclipse.hono.service.management.Sort;
import org.eclipse.hono.service.management.tenant.Tenant;
import org.eclipse.hono.service.management.tenant.TenantDto;
import org.eclipse.hono.service.management.tenant.TenantWithId;
import org.eclipse.hono.tracing.TracingHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MongoDbBasedTenantDao
extends MongoDbBasedDao
implements TenantDao,
HealthCheckProvider {
    private static final Logger LOG = LoggerFactory.getLogger(MongoDbBasedTenantDao.class);
    private final AtomicBoolean creatingIndices = new AtomicBoolean(false);
    private final AtomicBoolean indicesCreated = new AtomicBoolean(false);

    public MongoDbBasedTenantDao(MongoClient mongoClient, String collectionName, Tracer tracer) {
        super(mongoClient, collectionName, tracer, null);
    }

    public Future<Void> createIndices() {
        Promise result = Promise.promise();
        if (this.creatingIndices.compareAndSet(false, true)) {
            return this.createIndex(new JsonObject().put("tenant-id", Integer.valueOf(1)), new IndexOptions().unique(true)).compose(ok -> this.createIndex(new JsonObject().put("tenant.trusted-ca.subject-dn", Integer.valueOf(1)), new IndexOptions().unique(true).partialFilterExpression(new JsonObject().put("tenant.trusted-ca", new JsonObject().put("$exists", Boolean.valueOf(true)))))).onSuccess(ok -> this.indicesCreated.set(true)).onComplete(r -> {
                this.creatingIndices.set(false);
                result.handle(r);
            });
        }
        LOG.debug("already trying to create indices");
        return result.future();
    }

    public void registerReadinessChecks(HealthCheckHandler readinessHandler) {
        readinessHandler.register("tenants-indices-created-" + UUID.randomUUID(), status -> {
            if (this.indicesCreated.get()) {
                status.tryComplete((Object)Status.OK());
            } else {
                LOG.debug("tenants-indices not (yet) created");
                status.tryComplete((Object)Status.KO());
                this.createIndices();
            }
        });
    }

    public void registerLivenessChecks(HealthCheckHandler livenessHandler) {
    }

    @Override
    public Future<String> create(TenantDto tenantConfig, SpanContext tracingContext) {
        Objects.requireNonNull(tenantConfig);
        Span span = this.tracer.buildSpan("create Tenant").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_TENANT_ID, (Object)tenantConfig.getTenantId()).start();
        Promise createTenantPromise = Promise.promise();
        JsonObject newTenantDtoJson = JsonObject.mapFrom((Object)tenantConfig);
        this.mongoClient.insert(this.collectionName, newTenantDtoJson, (Handler)createTenantPromise);
        return createTenantPromise.future().map(tenantObjectIdResult -> {
            LOG.debug("successfully created tenant [tenant-id: {}, version: {}]", (Object)tenantConfig.getTenantId(), (Object)tenantConfig.getVersion());
            span.log("successfully created tenant");
            return tenantConfig.getVersion();
        }).recover(error -> {
            if (MongoDbBasedDao.isDuplicateKeyError(error)) {
                TracingHelper.logError((Span)span, (String)"tenant already exists or an existing tenant uses a CA with the same subject DN");
                return Future.failedFuture((Throwable)new ClientErrorException(tenantConfig.getTenantId(), 409, "tenant already exists or an existing tenant uses a CA with the same subject DN"));
            }
            TracingHelper.logError((Span)span, (String)"error creating tenant", (Throwable)error);
            return this.mapError((Throwable)error);
        }).onComplete(r -> span.finish());
    }

    @Override
    public Future<TenantDto> getById(String tenantId, SpanContext tracingContext) {
        Objects.requireNonNull(tenantId);
        Span span = this.tracer.buildSpan("get Tenant by ID").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_TENANT_ID, (Object)tenantId).start();
        return this.getById(tenantId, span).onComplete(r -> span.finish());
    }

    private Future<TenantDto> getById(String tenantId, Span span) {
        Promise findTenantPromise = Promise.promise();
        this.mongoClient.findOne(this.collectionName, MongoDbDocumentBuilder.builder().withTenantId(tenantId).document(), null, (Handler)findTenantPromise);
        return findTenantPromise.future().map(tenantJsonResult -> {
            if (tenantJsonResult == null) {
                throw new ClientErrorException(tenantId, 404, "no such tenant");
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace("tenant from collection:{}{}", (Object)System.lineSeparator(), (Object)tenantJsonResult.encodePrettily());
            }
            return TenantDto.forRead((String)tenantJsonResult.getString("tenant-id"), (Tenant)((Tenant)tenantJsonResult.getJsonObject("tenant").mapTo(Tenant.class)), (Instant)tenantJsonResult.getInstant("created"), (Instant)tenantJsonResult.getInstant("updatedOn"), (String)tenantJsonResult.getString("version"));
        }).onFailure(t -> TracingHelper.logError((Span)span, (String)"error retrieving tenant", (Throwable)t)).recover(this::mapError);
    }

    @Override
    public Future<TenantDto> getBySubjectDn(X500Principal subjectDn, SpanContext tracingContext) {
        Objects.requireNonNull(subjectDn);
        String dn = subjectDn.getName("RFC2253");
        Span span = this.tracer.buildSpan("get Tenant by subject DN").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_SUBJECT_DN, (Object)dn).start();
        Promise findTenantPromise = Promise.promise();
        this.mongoClient.findOne(this.collectionName, MongoDbDocumentBuilder.builder().withCa(dn).document(), new JsonObject(), (Handler)findTenantPromise);
        return findTenantPromise.future().map(tenantJsonResult -> {
            if (tenantJsonResult == null) {
                LOG.debug("could not find tenant [subject DN: {}]", (Object)dn);
                throw new ClientErrorException(404, "no such tenant");
            }
            return TenantDto.forRead((String)tenantJsonResult.getString("tenant-id"), (Tenant)((Tenant)tenantJsonResult.getJsonObject("tenant").mapTo(Tenant.class)), (Instant)tenantJsonResult.getInstant("created"), (Instant)tenantJsonResult.getInstant("updatedOn"), (String)tenantJsonResult.getString("version"));
        }).onFailure(t -> TracingHelper.logError((Span)span, (String)"error retrieving tenant", (Throwable)t)).recover(this::mapError).onComplete(r -> span.finish());
    }

    @Override
    public Future<String> update(TenantDto newTenantConfig, Optional<String> resourceVersion, SpanContext tracingContext) {
        Objects.requireNonNull(newTenantConfig);
        Objects.requireNonNull(resourceVersion);
        Span span = this.tracer.buildSpan("update Tenant").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_TENANT_ID, (Object)newTenantConfig.getTenantId()).start();
        resourceVersion.ifPresent(v -> TracingHelper.TAG_RESOURCE_VERSION.set(span, v));
        JsonObject updateTenantQuery = MongoDbDocumentBuilder.builder().withVersion(resourceVersion).withTenantId(newTenantConfig.getTenantId()).document();
        Promise updateTenantPromise = Promise.promise();
        this.mongoClient.findOneAndReplaceWithOptions(this.collectionName, updateTenantQuery, JsonObject.mapFrom((Object)newTenantConfig), new FindOptions(), new UpdateOptions().setReturningNewDocument(true), (Handler)updateTenantPromise);
        return updateTenantPromise.future().compose(updateResult -> {
            if (updateResult == null) {
                return MongoDbBasedDao.checkForVersionMismatchAndFail(newTenantConfig.getTenantId(), resourceVersion, this.getById(newTenantConfig.getTenantId(), span));
            }
            LOG.debug("successfully updated tenant [tenant-id: {}]", (Object)newTenantConfig.getTenantId());
            span.log("successfully updated tenant");
            return Future.succeededFuture((Object)updateResult.getString("version"));
        }).recover(error -> {
            if (MongoDbBasedDao.isDuplicateKeyError(error)) {
                LOG.debug("conflict updating tenant [{}]. An existing tenant uses a certificate authority with the same Subject DN", (Object)newTenantConfig.getTenantId(), error);
                TracingHelper.logError((Span)span, (String)"an existing tenant uses a certificate authority with the same Subject DN");
                return Future.failedFuture((Throwable)new ClientErrorException(newTenantConfig.getTenantId(), 409, "an existing tenant uses a certificate authority with the same Subject DN"));
            }
            TracingHelper.logError((Span)span, (String)"error updating tenant", (Throwable)error);
            return this.mapError((Throwable)error);
        }).onComplete(r -> span.finish());
    }

    @Override
    public Future<Void> delete(String tenantId, Optional<String> resourceVersion, SpanContext tracingContext) {
        Objects.requireNonNull(tenantId);
        Objects.requireNonNull(resourceVersion);
        Span span = this.tracer.buildSpan("delete Tenant").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_TENANT_ID, (Object)tenantId).start();
        resourceVersion.ifPresent(v -> TracingHelper.TAG_RESOURCE_VERSION.set(span, v));
        JsonObject deleteTenantQuery = MongoDbDocumentBuilder.builder().withVersion(resourceVersion).withTenantId(tenantId).document();
        Promise deleteTenantPromise = Promise.promise();
        this.mongoClient.findOneAndDelete(this.collectionName, deleteTenantQuery, (Handler)deleteTenantPromise);
        return deleteTenantPromise.future().compose(deleteResult -> {
            if (deleteResult == null) {
                return MongoDbBasedDao.checkForVersionMismatchAndFail(tenantId, resourceVersion, this.getById(tenantId, span));
            }
            LOG.debug("successfully deleted tenant [tenant-id: {}]", (Object)tenantId);
            span.log("successfully deleted tenant");
            return Future.succeededFuture((Object)null);
        }).onFailure(t -> TracingHelper.logError((Span)span, (String)"error deleting tenant", (Throwable)t)).recover(this::mapError).onComplete(r -> span.finish());
    }

    @Override
    public Future<SearchResult<TenantWithId>> find(int pageSize, int pageOffset, List<Filter> filters, List<Sort> sortOptions, SpanContext tracingContext) {
        Objects.requireNonNull(filters);
        Objects.requireNonNull(sortOptions);
        if (pageSize <= 0) {
            throw new IllegalArgumentException("page size must be a positive integer");
        }
        if (pageOffset < 0) {
            throw new IllegalArgumentException("page offset must not be negative");
        }
        Span span = this.tracer.buildSpan("find Tenants").addReference("child_of", tracingContext).start();
        JsonObject filterDocument = MongoDbDocumentBuilder.builder().withTenantFilters(filters).document();
        JsonObject sortDocument = MongoDbDocumentBuilder.builder().withTenantSortOptions(sortOptions).document();
        return this.processSearchResource(pageSize, pageOffset, filterDocument, sortDocument, MongoDbBasedTenantDao::getTenantsWithId).onFailure(t -> TracingHelper.logError((Span)span, (String)"error finding tenants", (Throwable)t)).onComplete(r -> span.finish());
    }

    private static List<TenantWithId> getTenantsWithId(JsonObject searchResult) {
        return Optional.ofNullable(searchResult.getJsonArray("result")).map(tenants -> tenants.stream().filter(JsonObject.class::isInstance).map(JsonObject.class::cast).map(json -> (TenantDto)json.mapTo(TenantDto.class)).map(tenantDto -> TenantWithId.from((String)tenantDto.getTenantId(), (Tenant)tenantDto.getData())).collect(Collectors.toList())).orElseGet(ArrayList::new);
    }
}

