/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.deviceregistry.mongodb.model;

import io.opentracing.Span;
import io.opentracing.SpanContext;
import io.opentracing.Tracer;
import io.opentracing.tag.Tag;
import io.vertx.core.Future;
import io.vertx.core.Promise;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.ext.healthchecks.HealthCheckHandler;
import io.vertx.ext.healthchecks.Status;
import io.vertx.ext.mongo.FindOptions;
import io.vertx.ext.mongo.IndexOptions;
import io.vertx.ext.mongo.MongoClient;
import io.vertx.ext.mongo.UpdateOptions;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.eclipse.hono.client.ClientErrorException;
import org.eclipse.hono.deviceregistry.mongodb.model.DeviceDao;
import org.eclipse.hono.deviceregistry.mongodb.model.MongoDbBasedDao;
import org.eclipse.hono.deviceregistry.mongodb.utils.MongoDbDocumentBuilder;
import org.eclipse.hono.service.HealthCheckProvider;
import org.eclipse.hono.service.management.Filter;
import org.eclipse.hono.service.management.SearchResult;
import org.eclipse.hono.service.management.Sort;
import org.eclipse.hono.service.management.device.Device;
import org.eclipse.hono.service.management.device.DeviceDto;
import org.eclipse.hono.service.management.device.DeviceWithId;
import org.eclipse.hono.tracing.TracingHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MongoDbBasedDeviceDao
extends MongoDbBasedDao
implements DeviceDao,
HealthCheckProvider {
    private static final Logger LOG = LoggerFactory.getLogger(MongoDbBasedDeviceDao.class);
    private static final String PROPERTY_DEVICE_MEMBER_OF = String.format("%s.%s", "device", "memberOf");
    private final AtomicBoolean creatingIndices = new AtomicBoolean(false);
    private final AtomicBoolean indicesCreated = new AtomicBoolean(false);

    public MongoDbBasedDeviceDao(MongoClient mongoClient, String collectionName, Tracer tracer) {
        super(mongoClient, collectionName, tracer, null);
    }

    public Future<Void> createIndices() {
        Promise result = Promise.promise();
        if (this.creatingIndices.compareAndSet(false, true)) {
            return this.createIndex(new JsonObject().put("tenant-id", (Object)1).put("device-id", (Object)1), new IndexOptions().unique(true)).onSuccess(ok -> this.indicesCreated.set(true)).onComplete(r -> {
                this.creatingIndices.set(false);
                result.handle(r);
            });
        }
        LOG.debug("already trying to create indices");
        return result.future();
    }

    public void registerReadinessChecks(HealthCheckHandler readinessHandler) {
        readinessHandler.register("devices-indices-created-" + UUID.randomUUID(), status -> {
            if (this.indicesCreated.get()) {
                status.tryComplete((Object)Status.OK());
            } else {
                LOG.debug("devices-indices not (yet) created");
                status.tryComplete((Object)Status.KO());
                this.createIndices();
            }
        });
    }

    public void registerLivenessChecks(HealthCheckHandler livenessHandler) {
    }

    @Override
    public Future<String> create(DeviceDto deviceConfig, SpanContext tracingContext) {
        Objects.requireNonNull(deviceConfig);
        Span span = this.tracer.buildSpan("create Device").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_TENANT_ID, (Object)deviceConfig.getTenantId()).withTag((Tag)TracingHelper.TAG_DEVICE_ID, (Object)deviceConfig.getDeviceId()).start();
        return this.mongoClient.insert(this.collectionName, JsonObject.mapFrom((Object)deviceConfig)).map(success -> {
            span.log("successfully created device");
            LOG.debug("successfully created device [tenant: {}, device-id: {}, resource-version: {}]", new Object[]{deviceConfig.getTenantId(), deviceConfig.getDeviceId(), deviceConfig.getVersion()});
            return deviceConfig.getVersion();
        }).recover(error -> {
            if (MongoDbBasedDao.isDuplicateKeyError(error)) {
                LOG.debug("device [{}] already exists for tenant [{}]", new Object[]{deviceConfig.getDeviceId(), deviceConfig.getTenantId(), error});
                TracingHelper.logError((Span)span, (String)"device already exists");
                throw new ClientErrorException(deviceConfig.getTenantId(), 409, "device already exists");
            }
            TracingHelper.logError((Span)span, (String)"error creating device", (Throwable)error);
            return this.mapError((Throwable)error);
        }).onComplete(r -> span.finish());
    }

    @Override
    public Future<DeviceDto> getById(String tenantId, String deviceId, SpanContext tracingContext) {
        Objects.requireNonNull(tenantId);
        Objects.requireNonNull(deviceId);
        Span span = this.tracer.buildSpan("get Device by ID").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_TENANT_ID, (Object)tenantId).withTag((Tag)TracingHelper.TAG_DEVICE_ID, (Object)deviceId).start();
        return this.getById(tenantId, deviceId, span).map(DeviceDto.class::cast).onComplete(r -> span.finish());
    }

    private Future<DeviceDto> getById(String tenantId, String deviceId, Span span) {
        JsonObject findDeviceQuery = MongoDbDocumentBuilder.builder().withTenantId(tenantId).withDeviceId(deviceId).document();
        return this.mongoClient.findOne(this.collectionName, findDeviceQuery, null).map(result -> {
            if (result == null) {
                throw new ClientErrorException(tenantId, 404, "device not found");
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace("device data from collection;{}{}", (Object)System.lineSeparator(), (Object)result.encodePrettily());
            }
            return (DeviceDto)result.mapTo(DeviceDto.class);
        }).onFailure(t -> TracingHelper.logError((Span)span, (String)"error retrieving device", (Throwable)t)).recover(this::mapError);
    }

    @Override
    public Future<Set<String>> resolveGroupMembers(String tenantId, Set<String> viaGroups, SpanContext tracingContext) {
        Objects.requireNonNull(tenantId);
        Objects.requireNonNull(viaGroups);
        if (viaGroups.isEmpty()) {
            return Future.succeededFuture(Set.of());
        }
        Span span = this.tracer.buildSpan("resolve group members").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_TENANT_ID, (Object)tenantId).start();
        span.log("resolving " + viaGroups.size() + " device groups");
        JsonObject deviceMemberOfSpec = new JsonObject().put("$exists", (Object)true).put("$in", (Object)new JsonArray(List.copyOf(viaGroups)));
        JsonObject resolveGroupMembersQuery = MongoDbDocumentBuilder.builder().withTenantId(tenantId).document().put(PROPERTY_DEVICE_MEMBER_OF, (Object)deviceMemberOfSpec);
        FindOptions findOptions = new FindOptions().setFields(new JsonObject().put("device-id", (Object)true).put("_id", (Object)false));
        return this.mongoClient.findWithOptions(this.collectionName, resolveGroupMembersQuery, findOptions).map(documents -> {
            if (documents == null) {
                Set result = Set.of();
                return result;
            }
            span.log("successfully resolved " + documents.size() + " group members");
            return documents.stream().map(json -> json.getString("device-id")).collect(Collectors.toSet());
        }).onComplete(r -> span.finish());
    }

    @Override
    public Future<SearchResult<DeviceWithId>> find(String tenantId, int pageSize, int pageOffset, List<Filter> filters, List<Sort> sortOptions, SpanContext tracingContext) {
        Objects.requireNonNull(tenantId);
        Objects.requireNonNull(filters);
        Objects.requireNonNull(sortOptions);
        if (pageSize <= 0) {
            throw new IllegalArgumentException("page size must be a positive integer");
        }
        if (pageOffset < 0) {
            throw new IllegalArgumentException("page offset must not be negative");
        }
        Span span = this.tracer.buildSpan("find Devices").addReference("child_of", tracingContext).start();
        JsonObject filterDocument = MongoDbDocumentBuilder.builder().withTenantId(tenantId).withDeviceFilters(filters).document();
        JsonObject sortDocument = MongoDbDocumentBuilder.builder().withDeviceSortOptions(sortOptions).document();
        return this.processSearchResource(pageSize, pageOffset, filterDocument, sortDocument, MongoDbBasedDeviceDao::getDevicesWithId).onFailure(t -> TracingHelper.logError((Span)span, (String)"error finding devices", (Throwable)t)).onComplete(r -> span.finish());
    }

    private static List<DeviceWithId> getDevicesWithId(JsonObject searchResult) {
        return Optional.ofNullable(searchResult.getJsonArray("result")).map(devices -> devices.stream().filter(JsonObject.class::isInstance).map(JsonObject.class::cast).peek(json -> {
            if (LOG.isDebugEnabled()) {
                LOG.debug("document from collection:{}{}", (Object)System.lineSeparator(), (Object)json.encodePrettily());
            }
        }).map(json -> (DeviceDto)json.mapTo(DeviceDto.class)).map(deviceDto -> DeviceWithId.from((String)deviceDto.getDeviceId(), (Device)deviceDto.getData())).collect(Collectors.toList())).orElseGet(List::of);
    }

    @Override
    public Future<String> update(DeviceDto deviceConfig, Optional<String> resourceVersion, SpanContext tracingContext) {
        Objects.requireNonNull(deviceConfig);
        Objects.requireNonNull(resourceVersion);
        Span span = this.tracer.buildSpan("update Device").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_TENANT_ID, (Object)deviceConfig.getTenantId()).withTag((Tag)TracingHelper.TAG_DEVICE_ID, (Object)deviceConfig.getDeviceId()).start();
        resourceVersion.ifPresent(v -> TracingHelper.TAG_RESOURCE_VERSION.set(span, v));
        JsonObject updateDeviceQuery = MongoDbDocumentBuilder.builder().withVersion(resourceVersion).withTenantId(deviceConfig.getTenantId()).withDeviceId(deviceConfig.getDeviceId()).document();
        JsonObject document = JsonObject.mapFrom((Object)deviceConfig);
        if (LOG.isTraceEnabled()) {
            LOG.trace("replacing existing device document with:{}{}", (Object)System.lineSeparator(), (Object)document.encodePrettily());
        }
        return this.mongoClient.findOneAndReplaceWithOptions(this.collectionName, updateDeviceQuery, document, new FindOptions(), new UpdateOptions().setReturningNewDocument(true)).compose(result -> {
            if (result == null) {
                return MongoDbBasedDao.checkForVersionMismatchAndFail(deviceConfig.getDeviceId(), resourceVersion, this.getById(deviceConfig.getTenantId(), deviceConfig.getDeviceId(), span));
            }
            span.log("successfully updated device");
            return Future.succeededFuture((Object)result.getString("version"));
        }).onFailure(t -> TracingHelper.logError((Span)span, (String)"error updating device", (Throwable)t)).recover(this::mapError).onComplete(r -> span.finish());
    }

    @Override
    public Future<Void> delete(String tenantId, String deviceId, Optional<String> resourceVersion, SpanContext tracingContext) {
        Objects.requireNonNull(tenantId);
        Objects.requireNonNull(deviceId);
        Objects.requireNonNull(resourceVersion);
        Span span = this.tracer.buildSpan("delete Device").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_TENANT_ID, (Object)tenantId).withTag((Tag)TracingHelper.TAG_DEVICE_ID, (Object)deviceId).start();
        resourceVersion.ifPresent(v -> TracingHelper.TAG_RESOURCE_VERSION.set(span, v));
        LOG.trace("deleting device [tenant-id: {}, device-id: {}, version: {}]", new Object[]{tenantId, deviceId, resourceVersion.orElse(null)});
        JsonObject deleteDeviceQuery = MongoDbDocumentBuilder.builder().withVersion(resourceVersion).withTenantId(tenantId).withDeviceId(deviceId).document();
        return this.mongoClient.findOneAndDelete(this.collectionName, deleteDeviceQuery).compose(result -> {
            if (result == null) {
                return MongoDbBasedDao.checkForVersionMismatchAndFail(deviceId, resourceVersion, this.getById(tenantId, deviceId, span));
            }
            span.log("successfully deleted device");
            return Future.succeededFuture((Object)null);
        }).onFailure(t -> TracingHelper.logError((Span)span, (String)"error deleting device", (Throwable)t)).recover(this::mapError).onComplete(r -> span.finish());
    }

    @Override
    public Future<Void> delete(String tenantId, SpanContext tracingContext) {
        Objects.requireNonNull(tenantId);
        Span span = this.tracer.buildSpan("delete all of Tenant's Devices").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_TENANT_ID, (Object)tenantId).start();
        JsonObject removeDevicesQuery = MongoDbDocumentBuilder.builder().withTenantId(tenantId).document();
        return this.mongoClient.removeDocuments(this.collectionName, removeDevicesQuery).compose(result -> {
            span.log("successfully deleted devices");
            LOG.debug("successfully deleted devices of tenant [tenant-id: {}]", (Object)tenantId);
            return Future.succeededFuture((Object)null);
        }).recover(error -> {
            LOG.debug("error deleting devices", error);
            TracingHelper.logError((Span)span, (String)"error deleting devices", (Throwable)error);
            return this.mapError((Throwable)error);
        }).onComplete(r -> span.finish());
    }

    @Override
    public Future<Long> count(String tenantId, SpanContext tracingContext) {
        Objects.requireNonNull(tenantId);
        Span span = this.tracer.buildSpan("count Devices").addReference("child_of", tracingContext).withTag((Tag)TracingHelper.TAG_TENANT_ID, (Object)tenantId).start();
        return this.mongoClient.count(this.collectionName, MongoDbDocumentBuilder.builder().withTenantId(tenantId).document()).onFailure(t -> TracingHelper.logError((Span)span, (String)"error counting devices", (Throwable)t)).recover(this::mapError).onComplete(r -> span.finish());
    }
}

