/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.service.auth;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.json.JsonObject;
import io.vertx.core.net.NetSocket;
import io.vertx.proton.ProtonConnection;
import io.vertx.proton.sasl.ProtonSaslAuthenticator;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.Objects;
import javax.net.ssl.SSLPeerUnverifiedException;
import org.apache.qpid.proton.engine.Sasl;
import org.apache.qpid.proton.engine.Transport;
import org.eclipse.hono.auth.HonoUser;
import org.eclipse.hono.client.ServiceInvocationException;
import org.eclipse.hono.service.auth.AuthenticationService;
import org.eclipse.hono.util.AuthenticationConstants;
import org.eclipse.hono.util.Constants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HonoSaslAuthenticator
implements ProtonSaslAuthenticator {
    private static final Logger LOG = LoggerFactory.getLogger(HonoSaslAuthenticator.class);
    private final AuthenticationService authenticationService;
    private Sasl sasl;
    private boolean succeeded;
    private ProtonConnection protonConnection;
    private X509Certificate clientCertificate;

    public HonoSaslAuthenticator(AuthenticationService authService) {
        this.authenticationService = Objects.requireNonNull(authService);
    }

    public void init(NetSocket socket, ProtonConnection protonConnection, Transport transport) {
        LOG.debug("initializing SASL authenticator");
        this.protonConnection = protonConnection;
        this.sasl = transport.sasl();
        this.sasl.server();
        this.sasl.allowSkip(false);
        this.sasl.setMechanisms(new String[]{"EXTERNAL", "PLAIN"});
        if (socket.isSsl()) {
            LOG.debug("client connected using TLS, extracting client certificate chain");
            try {
                Certificate cert = socket.sslSession().getPeerCertificates()[0];
                if (cert instanceof X509Certificate) {
                    this.clientCertificate = (X509Certificate)cert;
                }
            }
            catch (SSLPeerUnverifiedException e) {
                LOG.debug("could not extract client certificate chain, maybe client uses other mechanism than SASL EXTERNAL");
            }
        }
    }

    public void process(Handler<Boolean> completionHandler) {
        String[] remoteMechanisms = this.sasl.getRemoteMechanisms();
        if (remoteMechanisms.length == 0) {
            LOG.debug("client provided an empty list of SASL mechanisms [hostname: {}, state: {}]", (Object)this.sasl.getHostname(), (Object)this.sasl.getState().name());
            completionHandler.handle((Object)false);
        } else {
            String chosenMechanism = remoteMechanisms[0];
            LOG.debug("client wants to authenticate using SASL [mechanism: {}, host: {}, state: {}]", new Object[]{chosenMechanism, this.sasl.getHostname(), this.sasl.getState().name()});
            Future authTracker = Future.future();
            authTracker.setHandler(s -> {
                Sasl.SaslOutcome saslOutcome;
                if (s.succeeded()) {
                    HonoUser user = (HonoUser)s.result();
                    LOG.debug("authentication of client [authorization ID: {}] succeeded", (Object)user.getName());
                    Constants.setClientPrincipal((ProtonConnection)this.protonConnection, (HonoUser)user);
                    this.succeeded = true;
                    saslOutcome = Sasl.SaslOutcome.PN_SASL_OK;
                } else if (s.cause() instanceof ServiceInvocationException) {
                    int status = ((ServiceInvocationException)s.cause()).getErrorCode();
                    LOG.debug("authentication check failed: {} (status {})", (Object)s.cause().getMessage(), (Object)status);
                    saslOutcome = this.getSaslOutcomeForErrorStatus(status);
                } else {
                    LOG.debug("authentication check failed (no status code given in exception)", s.cause());
                    saslOutcome = Sasl.SaslOutcome.PN_SASL_TEMP;
                }
                this.sasl.done(saslOutcome);
                completionHandler.handle((Object)Boolean.TRUE);
            });
            byte[] saslResponse = new byte[this.sasl.pending()];
            this.sasl.recv(saslResponse, 0, saslResponse.length);
            this.verify(chosenMechanism, saslResponse, (Handler<AsyncResult<HonoUser>>)authTracker);
        }
    }

    private Sasl.SaslOutcome getSaslOutcomeForErrorStatus(int status) {
        Sasl.SaslOutcome saslOutcome;
        switch (status) {
            case 400: 
            case 401: {
                saslOutcome = Sasl.SaslOutcome.PN_SASL_AUTH;
                break;
            }
            case 500: {
                saslOutcome = Sasl.SaslOutcome.PN_SASL_SYS;
                break;
            }
            case 503: {
                saslOutcome = Sasl.SaslOutcome.PN_SASL_TEMP;
                break;
            }
            default: {
                saslOutcome = status >= 400 && status < 500 ? Sasl.SaslOutcome.PN_SASL_PERM : Sasl.SaslOutcome.PN_SASL_TEMP;
            }
        }
        return saslOutcome;
    }

    public boolean succeeded() {
        return this.succeeded;
    }

    private void verify(String mechanism, byte[] saslResponse, Handler<AsyncResult<HonoUser>> authResultHandler) {
        JsonObject authRequest = AuthenticationConstants.getAuthenticationRequest((String)mechanism, (byte[])saslResponse);
        if (this.clientCertificate != null) {
            String subjectDn = this.clientCertificate.getSubjectX500Principal().getName("RFC2253");
            LOG.debug("client has provided X.509 certificate [subject DN: {}]", (Object)subjectDn);
            authRequest.put("subject-dn", subjectDn);
        }
        this.authenticationService.authenticate(authRequest, authResultHandler);
    }
}

