/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.service.auth.device;

import io.opentracing.Span;
import io.opentracing.SpanContext;
import io.opentracing.Tracer;
import io.opentracing.noop.NoopTracerFactory;
import io.opentracing.tag.Tags;
import io.vertx.core.Future;
import io.vertx.core.json.JsonObject;
import java.security.GeneralSecurityException;
import java.security.cert.Certificate;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import org.eclipse.hono.client.ClientErrorException;
import org.eclipse.hono.client.HonoClient;
import org.eclipse.hono.service.auth.device.DeviceCertificateValidator;
import org.eclipse.hono.service.auth.device.X509Authentication;
import org.eclipse.hono.tracing.TracingHelper;
import org.eclipse.hono.util.TenantObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TenantServiceBasedX509Authentication
implements X509Authentication {
    private static final ClientErrorException UNAUTHORIZED = new ClientErrorException(401);
    private static final Logger log = LoggerFactory.getLogger(TenantServiceBasedX509Authentication.class);
    private final Tracer tracer;
    private final HonoClient tenantServiceClient;
    private final DeviceCertificateValidator certPathValidator;

    public TenantServiceBasedX509Authentication(HonoClient tenantServiceClient) {
        this(tenantServiceClient, (Tracer)NoopTracerFactory.create());
    }

    public TenantServiceBasedX509Authentication(HonoClient tenantServiceClient, Tracer tracer) {
        this(tenantServiceClient, tracer, new DeviceCertificateValidator());
    }

    public TenantServiceBasedX509Authentication(HonoClient tenantServiceClient, Tracer tracer, DeviceCertificateValidator certPathValidator) {
        this.tenantServiceClient = Objects.requireNonNull(tenantServiceClient);
        this.tracer = Objects.requireNonNull(tracer);
        this.certPathValidator = Objects.requireNonNull(certPathValidator);
    }

    @Override
    public Future<JsonObject> validateClientCertificate(Certificate[] path, SpanContext currentSpan) {
        Objects.requireNonNull(path);
        Span span = this.tracer.buildSpan("verify device certificate").asChildOf(currentSpan).ignoreActiveSpan().withTag(Tags.SPAN_KIND.getKey(), "client").withTag(Tags.COMPONENT.getKey(), this.getClass().getSimpleName()).start();
        return this.getX509CertificatePath(path).compose(x509chain -> {
            X509Certificate deviceCert = (X509Certificate)x509chain.get(0);
            HashMap<String, String> detail = new HashMap<String, String>(3);
            detail.put("subject DN", deviceCert.getSubjectX500Principal().getName());
            detail.put("not before", deviceCert.getNotBefore().toString());
            detail.put("not after", deviceCert.getNotAfter().toString());
            span.log(detail);
            Future<TenantObject> tenantTracker = this.getTenant(deviceCert, span);
            return tenantTracker.compose(tenant -> {
                try {
                    TrustAnchor trustAnchor = tenant.getTrustAnchor();
                    List<X509Certificate> chainToValidate = Collections.singletonList(deviceCert);
                    return this.certPathValidator.validate(chainToValidate, trustAnchor).recover(t -> Future.failedFuture((Throwable)UNAUTHORIZED));
                }
                catch (GeneralSecurityException e) {
                    log.debug("cannot de-serialize trust anchor from tenant: {}", (Object)e.getMessage());
                    return Future.failedFuture((Throwable)UNAUTHORIZED);
                }
            }).compose(ok -> this.getCredentials((List<X509Certificate>)x509chain, (TenantObject)tenantTracker.result()));
        }).map(authInfo -> {
            span.log("certificate verified successfully");
            span.finish();
            return authInfo;
        }).recover(t -> {
            log.debug("verification of client certificate failed: {}", (Object)t.getMessage());
            TracingHelper.logError((Span)span, (Throwable)t);
            span.finish();
            return Future.failedFuture((Throwable)t);
        });
    }

    private Future<TenantObject> getTenant(X509Certificate clientCert, Span span) {
        return this.tenantServiceClient.getOrCreateTenantClient().compose(tenantClient -> tenantClient.get(clientCert.getIssuerX500Principal(), span.context()));
    }

    private Future<List<X509Certificate>> getX509CertificatePath(Certificate[] clientPath) {
        LinkedList<X509Certificate> path = new LinkedList<X509Certificate>();
        for (Certificate cert : clientPath) {
            if (!(cert instanceof X509Certificate)) {
                log.info("cannot authenticate device using unsupported certificate type [{}]", (Object)cert.getClass().getName());
                return Future.failedFuture((Throwable)UNAUTHORIZED);
            }
            path.add((X509Certificate)cert);
        }
        return Future.succeededFuture(path);
    }

    protected Future<JsonObject> getCredentials(List<X509Certificate> clientCertPath, TenantObject tenant) {
        String subjectDn = clientCertPath.get(0).getSubjectX500Principal().getName("RFC2253");
        log.debug("authenticating device of tenant [{}] using X509 certificate [subject DN: {}]", (Object)tenant.getTenantId(), (Object)subjectDn);
        return Future.succeededFuture((Object)new JsonObject().put("subject-dn", subjectDn).put("tenant-id", tenant.getTenantId()));
    }
}

