/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.adapter.resourcelimits;

import com.github.benmanes.caffeine.cache.AsyncCache;
import io.opentracing.Span;
import io.opentracing.SpanContext;
import io.opentracing.Tracer;
import io.opentracing.noop.NoopTracerFactory;
import io.opentracing.tag.Tags;
import io.vertx.core.Context;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.ext.web.client.HttpRequest;
import io.vertx.ext.web.client.HttpResponse;
import io.vertx.ext.web.client.WebClient;
import io.vertx.ext.web.client.predicate.ResponsePredicate;
import io.vertx.ext.web.codec.BodyCodec;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.YearMonth;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAdjusters;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import org.eclipse.hono.adapter.resourcelimits.LimitedResource;
import org.eclipse.hono.adapter.resourcelimits.PrometheusBasedResourceLimitChecksConfig;
import org.eclipse.hono.adapter.resourcelimits.ResourceLimitChecks;
import org.eclipse.hono.service.metric.MetricsTags;
import org.eclipse.hono.tracing.TracingHelper;
import org.eclipse.hono.util.ConnectionDuration;
import org.eclipse.hono.util.DataVolume;
import org.eclipse.hono.util.ResourceLimitsPeriod;
import org.eclipse.hono.util.Strings;
import org.eclipse.hono.util.TenantObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PrometheusBasedResourceLimitChecks
implements ResourceLimitChecks {
    private static final Logger LOG = LoggerFactory.getLogger(PrometheusBasedResourceLimitChecks.class);
    private static final String METRIC_NAME_COMMANDS_PAYLOAD_SIZE = String.format("%s_bytes_sum", "hono.commands.payload".replace(".", "_"));
    private static final String METRIC_NAME_CONNECTIONS = "hono.connections.authenticated".replace(".", "_");
    private static final String METRIC_NAME_CONNECTIONS_DURATION = String.format("%s_seconds_sum", "hono.connections.authenticated.duration".replace(".", "_"));
    private static final String METRIC_NAME_MESSAGES_PAYLOAD_SIZE = String.format("%s_bytes_sum", "hono.messages.payload".replace(".", "_"));
    private static final String QUERY_TEMPLATE_MESSAGE_LIMIT = String.format("floor(sum(increase(%1$s{status=~\"%3$s|%4$s\", tenant=\"%%1$s\"} [%%2$dm:%%3$ds]) or %2$s*0) + sum(increase(%2$s{status=~\"%3$s|%4$s\", tenant=\"%%1$s\"} [%%2$dm:%%3$ds]) or %1$s*0))", METRIC_NAME_MESSAGES_PAYLOAD_SIZE, METRIC_NAME_COMMANDS_PAYLOAD_SIZE, MetricsTags.ProcessingOutcome.FORWARDED.asTag().getValue(), MetricsTags.ProcessingOutcome.UNPROCESSABLE.asTag().getValue());
    private static final String QUERY_URI = "/api/v1/query";
    private final Tracer tracer;
    private final WebClient client;
    private final PrometheusBasedResourceLimitChecksConfig config;
    private final AsyncCache<String, LimitedResource<Long>> connectionCountCache;
    private final AsyncCache<String, LimitedResource<Duration>> connectionDurationCache;
    private final AsyncCache<String, LimitedResource<Long>> dataVolumeCache;
    private final String url;
    private Clock clock = Clock.systemUTC();

    public PrometheusBasedResourceLimitChecks(WebClient webClient, PrometheusBasedResourceLimitChecksConfig config, AsyncCache<String, LimitedResource<Long>> connectionCountCache, AsyncCache<String, LimitedResource<Duration>> connectionDurationCache, AsyncCache<String, LimitedResource<Long>> dataVolumeCache) {
        this(webClient, config, connectionCountCache, connectionDurationCache, dataVolumeCache, (Tracer)NoopTracerFactory.create());
    }

    public PrometheusBasedResourceLimitChecks(WebClient webClient, PrometheusBasedResourceLimitChecksConfig config, AsyncCache<String, LimitedResource<Long>> connectionCountCache, AsyncCache<String, LimitedResource<Duration>> connectionDurationCache, AsyncCache<String, LimitedResource<Long>> dataVolumeCache, Tracer tracer) {
        this.client = Objects.requireNonNull(webClient);
        this.config = Objects.requireNonNull(config);
        this.connectionCountCache = Objects.requireNonNull(connectionCountCache);
        this.connectionDurationCache = Objects.requireNonNull(connectionDurationCache);
        this.dataVolumeCache = Objects.requireNonNull(dataVolumeCache);
        this.tracer = Objects.requireNonNull(tracer);
        this.url = String.format("%s://%s:%d%s", config.isTlsEnabled() ? "https" : "http", config.getHost(), config.getPort(), QUERY_URI);
    }

    private Span createSpan(String name, SpanContext parent, TenantObject tenant) {
        return TracingHelper.buildChildSpan((Tracer)this.tracer, (SpanContext)parent, (String)name, (String)this.getClass().getSimpleName()).withTag(Tags.SPAN_KIND.getKey(), "client").withTag(Tags.PEER_HOSTNAME.getKey(), this.config.getHost()).withTag(Tags.PEER_PORT.getKey(), (Number)this.config.getPort()).withTag(Tags.HTTP_URL.getKey(), QUERY_URI).withTag(TracingHelper.TAG_TENANT_ID.getKey(), tenant.getTenantId()).start();
    }

    void setClock(Clock clock) {
        this.clock = Objects.requireNonNull(clock);
    }

    @Override
    public Future<Boolean> isConnectionLimitReached(TenantObject tenant, SpanContext spanContext) {
        Objects.requireNonNull(tenant);
        Span span = this.createSpan("verify connection limit", spanContext, tenant);
        HashMap<String, String> traceItems = new HashMap<String, String>();
        Promise result = Promise.promise();
        if (tenant.getResourceLimits() == null) {
            traceItems.put("event", "no resource limits configured");
            LOG.trace("no resource limits configured for tenant [{}]", (Object)tenant.getTenantId());
            result.complete((Object)Boolean.FALSE);
        } else {
            long maxConnections = tenant.getResourceLimits().getMaxConnections();
            LOG.trace("connection limit for tenant [{}] is [{}]", (Object)tenant.getTenantId(), (Object)maxConnections);
            if (maxConnections == -1L) {
                traceItems.put("event", "no connection limit configured");
                result.complete((Object)Boolean.FALSE);
            } else {
                Context originalContext = Vertx.currentContext();
                AtomicBoolean cacheHit = new AtomicBoolean(true);
                this.connectionCountCache.get((Object)tenant.getTenantId(), (tenantId, executor) -> {
                    CompletableFuture r = new CompletableFuture();
                    cacheHit.set(false);
                    String queryParams = String.format("sum(%s{tenant=\"%s\"})", METRIC_NAME_CONNECTIONS, tenantId);
                    this.executeQuery(queryParams, span).onSuccess(currentConnections -> r.complete(new LimitedResource<Long>(maxConnections, (Long)currentConnections))).onFailure(r::completeExceptionally);
                    return r;
                }).whenComplete((value, error) -> this.runOnContext(originalContext, (Handler<Void>)((Handler)v -> {
                    TracingHelper.TAG_CACHE_HIT.set(span, Boolean.valueOf(cacheHit.get()));
                    if (error != null) {
                        TracingHelper.logError((Span)span, (Throwable)error);
                        result.complete((Object)Boolean.FALSE);
                    } else {
                        traceItems.put("max-connections", (String)maxConnections);
                        traceItems.put("current-connections", (String)value.getCurrentValue());
                        boolean isExceeded = (Long)value.getCurrentValue() >= (Long)value.getCurrentLimit();
                        LOG.trace("connection limit {}exceeded [tenant: {}, current connections: {}, max-connections: {}]", new Object[]{isExceeded ? "" : "not ", tenant.getTenantId(), value.getCurrentValue(), value.getCurrentLimit()});
                        result.complete((Object)isExceeded);
                    }
                })));
            }
        }
        return result.future().map(b -> {
            traceItems.put("limit exceeded", (String)b);
            span.log(traceItems);
            span.finish();
            return b;
        });
    }

    @Override
    public Future<Boolean> isMessageLimitReached(TenantObject tenant, long payloadSize, SpanContext spanContext) {
        Objects.requireNonNull(tenant);
        Span span = this.createSpan("verify message limit", spanContext, tenant);
        HashMap<String, Object> items = new HashMap<String, Object>();
        items.put("payload-size", payloadSize);
        Promise result = Promise.promise();
        if (tenant.getResourceLimits() == null) {
            items.put("event", "no resource limits configured");
            LOG.trace("no resource limits configured for tenant [{}]", (Object)tenant.getTenantId());
            result.complete((Object)Boolean.FALSE);
        } else if (tenant.getResourceLimits().getDataVolume() == null) {
            items.put("event", "no message limits configured");
            LOG.trace("no message limits configured for tenant [{}]", (Object)tenant.getTenantId());
            result.complete((Object)Boolean.FALSE);
        } else {
            this.checkMessageLimit(tenant, payloadSize, items, span, (Promise<Boolean>)result);
        }
        return result.future().map(b -> {
            items.put("limit exceeded", b);
            span.log(items);
            span.finish();
            return b;
        });
    }

    private void checkMessageLimit(TenantObject tenant, long payloadSize, Map<String, Object> items, Span span, Promise<Boolean> result) {
        DataVolume dataVolumeConfig = tenant.getResourceLimits().getDataVolume();
        long maxBytes = dataVolumeConfig.getMaxBytes();
        Instant effectiveSince = dataVolumeConfig.getEffectiveSince();
        String periodMode = dataVolumeConfig.getPeriod().getMode();
        long periodInDays = dataVolumeConfig.getPeriod().getNoOfDays();
        LOG.trace("message limit config for tenant [{}] are [{}:{}, {}:{}, {}:{}, {}:{}]", new Object[]{tenant.getTenantId(), "max-bytes", maxBytes, "effective-since", effectiveSince, "mode", periodMode, "no-of-days", periodInDays});
        if (maxBytes == -1L || effectiveSince == null || !ResourceLimitsPeriod.isSupportedMode((String)periodMode) || payloadSize <= 0L) {
            result.complete((Object)Boolean.FALSE);
        } else {
            Context originalContext = Vertx.currentContext();
            AtomicBoolean cacheHit = new AtomicBoolean(true);
            this.dataVolumeCache.get((Object)tenant.getTenantId(), (tenantId, executor) -> {
                CompletableFuture<LimitedResource<Long>> r = new CompletableFuture<LimitedResource<Long>>();
                cacheHit.set(false);
                Instant nowUtc = Instant.now(this.clock);
                Long allowedMaxBytes = this.calculateEffectiveLimit(effectiveSince, nowUtc, periodMode, maxBytes);
                Duration dataUsagePeriod = this.calculateResourceUsageDuration(effectiveSince, nowUtc, periodMode, periodInDays);
                if (dataUsagePeriod.toMinutes() <= 0L) {
                    r.complete(new LimitedResource<Long>(allowedMaxBytes, 0L));
                } else {
                    String queryParams = String.format(QUERY_TEMPLATE_MESSAGE_LIMIT, tenant.getTenantId(), dataUsagePeriod.toMinutes(), this.config.getCacheTimeout());
                    this.executeQuery(queryParams, span).onSuccess(bytesConsumed -> r.complete(new LimitedResource<Long>(allowedMaxBytes, (Long)bytesConsumed))).onFailure(r::completeExceptionally);
                }
                return r;
            }).whenComplete((value, error) -> this.runOnContext(originalContext, (Handler<Void>)((Handler)v -> {
                TracingHelper.TAG_CACHE_HIT.set(span, Boolean.valueOf(cacheHit.get()));
                if (error != null) {
                    TracingHelper.logError((Span)span, (Throwable)error);
                    result.complete((Object)Boolean.FALSE);
                } else {
                    items.put("current period bytes limit", value.getCurrentLimit());
                    items.put("current period bytes consumed", value.getCurrentValue());
                    boolean isExceeded = (Long)value.getCurrentValue() + payloadSize > (Long)value.getCurrentLimit();
                    LOG.trace("data limit {}exceeded [tenant: {}, bytes consumed: {}, allowed max-bytes: {}, {}: {}, {}: {}, {}: {}]", new Object[]{isExceeded ? "" : "not ", tenant.getTenantId(), value.getCurrentValue(), value.getCurrentLimit(), "effective-since", effectiveSince, "mode", periodMode, "no-of-days", periodInDays});
                    result.complete((Object)isExceeded);
                }
            })));
        }
    }

    @Override
    public Future<Boolean> isConnectionDurationLimitReached(TenantObject tenant, SpanContext spanContext) {
        Objects.requireNonNull(tenant);
        Span span = this.createSpan("verify connection duration limit", spanContext, tenant);
        HashMap<String, Object> items = new HashMap<String, Object>();
        Promise result = Promise.promise();
        if (tenant.getResourceLimits() == null) {
            items.put("event", "no resource limits configured");
            LOG.trace("no resource limits configured for tenant [{}]", (Object)tenant.getTenantId());
            result.complete((Object)Boolean.FALSE);
        } else if (tenant.getResourceLimits().getConnectionDuration() == null) {
            items.put("event", "no connection duration limit configured");
            LOG.trace("no connection duration limit configured for tenant [{}]", (Object)tenant.getTenantId());
            result.complete((Object)Boolean.FALSE);
        } else {
            this.checkConnectionDurationLimit(tenant, items, span, (Promise<Boolean>)result);
        }
        return result.future().map(b -> {
            items.put("limit exceeded", b);
            span.log(items);
            span.finish();
            return b;
        });
    }

    private void checkConnectionDurationLimit(TenantObject tenant, Map<String, Object> items, Span span, Promise<Boolean> result) {
        ConnectionDuration connectionDurationConfig = tenant.getResourceLimits().getConnectionDuration();
        long maxConnectionDurationInMinutes = connectionDurationConfig.getMaxMinutes();
        Instant effectiveSince = connectionDurationConfig.getEffectiveSince();
        String periodMode = connectionDurationConfig.getPeriod().getMode();
        long periodInDays = connectionDurationConfig.getPeriod().getNoOfDays();
        LOG.trace("connection duration config for the tenant [{}] is [{}:{}, {}:{}, {}:{}, {}:{}]", new Object[]{tenant.getTenantId(), "max-minutes", maxConnectionDurationInMinutes, "effective-since", effectiveSince, "mode", periodMode, "no-of-days", periodInDays});
        if (maxConnectionDurationInMinutes == -1L || effectiveSince == null || !ResourceLimitsPeriod.isSupportedMode((String)periodMode)) {
            result.complete((Object)Boolean.FALSE);
        } else {
            Context originalContext = Vertx.currentContext();
            AtomicBoolean cacheHit = new AtomicBoolean(true);
            this.connectionDurationCache.get((Object)tenant.getTenantId(), (tenantId, executor) -> {
                CompletableFuture<LimitedResource<Duration>> r = new CompletableFuture<LimitedResource<Duration>>();
                cacheHit.set(false);
                Instant nowUtc = Instant.now(this.clock);
                Duration allowedMaxMinutes = Duration.ofMinutes(this.calculateEffectiveLimit(effectiveSince, nowUtc, periodMode, maxConnectionDurationInMinutes));
                Duration connectionDurationUsagePeriod = this.calculateResourceUsageDuration(effectiveSince, nowUtc, periodMode, periodInDays);
                if (connectionDurationUsagePeriod.toMinutes() <= 0L) {
                    r.complete(new LimitedResource<Duration>(allowedMaxMinutes, Duration.ofMinutes(0L)));
                } else {
                    String queryParams = String.format("minute( sum( increase( %s {tenant=\"%s\"} [%dm:%ds])))", METRIC_NAME_CONNECTIONS_DURATION, tenant.getTenantId(), connectionDurationUsagePeriod.toMinutes(), this.config.getCacheTimeout());
                    this.executeQuery(queryParams, span).onSuccess(minutesConnected -> r.complete(new LimitedResource<Duration>(allowedMaxMinutes, Duration.ofMinutes(minutesConnected)))).onFailure(r::completeExceptionally);
                }
                return r;
            }).whenComplete((value, error) -> this.runOnContext(originalContext, (Handler<Void>)((Handler)v -> {
                TracingHelper.TAG_CACHE_HIT.set(span, Boolean.valueOf(cacheHit.get()));
                if (error != null) {
                    TracingHelper.logError((Span)span, (Throwable)error);
                    result.complete((Object)Boolean.FALSE);
                } else {
                    items.put("current period's connection duration limit", value.getCurrentLimit());
                    items.put("current period's connection duration consumed", value.getCurrentValue());
                    boolean isExceeded = ((Duration)value.getCurrentValue()).compareTo((Duration)value.getCurrentLimit()) >= 0;
                    LOG.trace("connection duration limit {} exceeded [tenant: {}, connection duration consumed: {}, allowed max-duration: {}, {}: {}, {}: {}, {}: {}]", new Object[]{isExceeded ? "" : "not ", tenant.getTenantId(), value.getCurrentValue(), value.getCurrentLimit(), "effective-since", effectiveSince, "mode", periodMode, "no-of-days", periodInDays});
                    result.complete((Object)isExceeded);
                }
            })));
        }
    }

    private Future<Long> executeQuery(String query, Span span) {
        Promise result = Promise.promise();
        LOG.trace("running Prometheus query [URL: {}, query: {}]", (Object)this.url, (Object)query);
        this.newQueryRequest(query).send(sendAttempt -> {
            if (sendAttempt.succeeded()) {
                HttpResponse response = (HttpResponse)sendAttempt.result();
                result.complete((Object)this.extractLongValue((JsonObject)response.body(), span));
            } else {
                Map<String, Throwable> traceItems = Map.of("event", Tags.ERROR.getKey(), "message", "failed to run Prometheus query", "URL", this.url, "query", query, "error.kind", "Exception", "error.object", sendAttempt.cause());
                TracingHelper.logError((Span)span, traceItems);
                LOG.warn("failed to run Prometheus query [URL: {}, query: {}]: {}", new Object[]{this.url, query, sendAttempt.cause().getMessage()});
                result.fail(sendAttempt.cause());
            }
        });
        return result.future();
    }

    private HttpRequest<JsonObject> newQueryRequest(String query) {
        HttpRequest request = this.client.post(QUERY_URI).addQueryParam("query", query).expect(ResponsePredicate.SC_OK);
        if (this.config.getQueryTimeout() > 0L) {
            request.addQueryParam("timeout", String.format("%dms", this.config.getQueryTimeout()));
            request.timeout(this.config.getQueryTimeout() + 100L);
        }
        if (!Strings.isNullOrEmpty((Object)this.config.getUsername()) && !Strings.isNullOrEmpty((Object)this.config.getPassword())) {
            request.basicAuthentication(this.config.getUsername(), this.config.getPassword());
        }
        return request.as(BodyCodec.jsonObject());
    }

    private Long extractLongValue(JsonObject response, Span span) {
        Objects.requireNonNull(response);
        try {
            String status = response.getString("status");
            if ("error".equals(status)) {
                TracingHelper.logError((Span)span, Map.of("message", "error executing query", "status", status, "error-type", response.getString("errorType"), "error", response.getString("error")));
                LOG.debug("error executing query [status: {}, error type: {}, error: {}]", new Object[]{status, response.getString("errorType"), response.getString("error")});
                return 0L;
            }
            JsonObject data = response.getJsonObject("data", new JsonObject());
            JsonArray result = data.getJsonArray("result");
            if (result != null) {
                String value;
                JsonArray valueArray;
                if (result.size() == 0) {
                    span.log("no metrics available (yet)");
                    return 0L;
                }
                if (result.size() == 1 && result.getJsonObject(0) != null && (valueArray = result.getJsonObject(0).getJsonArray("value")) != null && valueArray.size() == 2 && (value = valueArray.getString(1)) != null && !value.isEmpty()) {
                    return Long.parseLong(value);
                }
            }
            String jsonResponse = response.encodePrettily();
            TracingHelper.logError((Span)span, Map.of("message", "server returned malformed response", "response", jsonResponse));
            LOG.debug("server returned malformed response: {}", (Object)jsonResponse);
        }
        catch (Exception e) {
            String jsonResponse = response.encodePrettily();
            TracingHelper.logError((Span)span, Map.of("message", "server returned malformed response", "response", jsonResponse));
            LOG.debug("server returned malformed response: {}", (Object)jsonResponse);
        }
        return 0L;
    }

    long calculateEffectiveLimit(Instant effectiveSince, Instant targetDateTime, String periodType, long configuredLimit) {
        Objects.requireNonNull(effectiveSince, "effective since");
        Objects.requireNonNull(targetDateTime, "target date-time");
        Objects.requireNonNull(periodType, "period mode");
        if (targetDateTime.isBefore(effectiveSince)) {
            return 0L;
        }
        if ("monthly".equals(periodType) && configuredLimit > 0L) {
            ZonedDateTime effectiveSinceZonedDateTime = ZonedDateTime.ofInstant(effectiveSince, ZoneOffset.UTC);
            ZonedDateTime targetZonedDateTime = ZonedDateTime.ofInstant(targetDateTime, ZoneOffset.UTC);
            if (YearMonth.from(targetZonedDateTime).equals(YearMonth.from(effectiveSinceZonedDateTime))) {
                ZonedDateTime startOfNextAccountingPeriod = effectiveSinceZonedDateTime.with(TemporalAdjusters.firstDayOfNextMonth()).withHour(0).withMinute(0).withSecond(0).withNano(0);
                long minutesTillStartOfNextAccountingPeriod = Math.max(1L, Duration.between(effectiveSinceZonedDateTime, startOfNextAccountingPeriod).toMinutes());
                long lengthOfCurrentMonthInMinutes = 1440L * effectiveSinceZonedDateTime.range(ChronoField.DAY_OF_MONTH).getMaximum();
                return (long)Math.ceil((double)minutesTillStartOfNextAccountingPeriod * (double)configuredLimit / (double)lengthOfCurrentMonthInMinutes);
            }
        }
        return configuredLimit;
    }

    Duration calculateResourceUsageDuration(Instant effectiveSince, Instant targetDateTime, String periodType, long periodLength) {
        Objects.requireNonNull(effectiveSince, "effective since");
        Objects.requireNonNull(targetDateTime, "target date-time");
        Objects.requireNonNull(periodType, "period type");
        if (targetDateTime.isBefore(effectiveSince)) {
            return Duration.ZERO;
        }
        ZonedDateTime targetZonedDateTime = ZonedDateTime.ofInstant(targetDateTime, ZoneOffset.UTC);
        ZonedDateTime beginningOfMostRecentAccountingPeriod = this.getBeginningOfMostRecentAccountingPeriod(ZonedDateTime.ofInstant(effectiveSince, ZoneOffset.UTC), targetZonedDateTime, periodType, periodLength);
        return Duration.between(beginningOfMostRecentAccountingPeriod, targetZonedDateTime);
    }

    private ZonedDateTime getBeginningOfMostRecentAccountingPeriod(ZonedDateTime effectiveSince, ZonedDateTime targetDateTime, String periodType, long periodLength) {
        switch (periodType) {
            case "monthly": {
                YearMonth targetYearMonth = YearMonth.from(targetDateTime);
                if (targetYearMonth.equals(YearMonth.from(effectiveSince))) {
                    return effectiveSince;
                }
                return ZonedDateTime.of(targetYearMonth.getYear(), targetYearMonth.getMonthValue(), 1, 0, 0, 0, 0, ZoneOffset.UTC);
            }
            case "days": {
                Duration overall = Duration.between(effectiveSince, targetDateTime);
                Duration accountingPeriodLength = Duration.ofDays(periodLength);
                if (overall.compareTo(accountingPeriodLength) < 1) {
                    return effectiveSince;
                }
                long totalPeriodsElapsed = overall.dividedBy(accountingPeriodLength);
                return effectiveSince.plus(accountingPeriodLength.multipliedBy(totalPeriodsElapsed));
            }
        }
        return targetDateTime;
    }

    private void runOnContext(Context context, Handler<Void> action) {
        if (context != null && context != Vertx.currentContext()) {
            context.runOnContext(go -> action.handle(null));
        } else {
            action.handle(null);
        }
    }
}

