/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.eclipse.californium.elements.util.DatagramReader;
import org.eclipse.californium.elements.util.DatagramWriter;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.ApplicationMessage;
import org.eclipse.californium.scandium.dtls.ChangeCipherSpecMessage;
import org.eclipse.californium.scandium.dtls.ConnectionId;
import org.eclipse.californium.scandium.dtls.ConnectionIdGenerator;
import org.eclipse.californium.scandium.dtls.ContentType;
import org.eclipse.californium.scandium.dtls.DTLSConnectionState;
import org.eclipse.californium.scandium.dtls.DTLSContext;
import org.eclipse.californium.scandium.dtls.DTLSMessage;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.eclipse.californium.scandium.dtls.ProtocolVersion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Record {
    private static final Logger LOGGER = LoggerFactory.getLogger(Record.class);
    public static final int CONTENT_TYPE_BITS = 8;
    public static final int VERSION_BITS = 8;
    public static final int EPOCH_BITS = 16;
    public static final int SEQUENCE_NUMBER_BITS = 48;
    public static final int LENGTH_BITS = 16;
    public static final int CID_LENGTH_BITS = 8;
    public static final int RECORD_HEADER_BITS = 104;
    public static final int RECORD_HEADER_BYTES = 13;
    public static final int DTLS_HANDSHAKE_HEADER_LENGTH = 25;
    public static final long MAX_SEQUENCE_NO = 0xFFFFFFFFFFFFL;
    private ContentType type;
    private final ProtocolVersion version;
    private final int epoch;
    private final long sequenceNumber;
    private final long receiveNanos;
    private final boolean followUpRecord;
    private DTLSMessage fragment;
    private byte[] fragmentBytes;
    private ConnectionId connectionId;
    private int padding;
    private InetSocketAddress peerAddress;
    private InetSocketAddress router;

    Record(ContentType type, ProtocolVersion version, int epoch, long sequenceNumber, ConnectionId connectionId, byte[] fragmentBytes, long receiveNanos, boolean followUpRecord) {
        this(version, epoch, sequenceNumber, receiveNanos, followUpRecord);
        if (type == null) {
            throw new NullPointerException("Type must not be null");
        }
        if (fragmentBytes == null) {
            throw new NullPointerException("Fragment bytes must not be null");
        }
        this.type = type;
        this.connectionId = connectionId;
        this.fragmentBytes = fragmentBytes;
    }

    public Record(ContentType type, int epoch, DTLSMessage fragment, DTLSContext context, boolean cid, int pad) throws GeneralSecurityException {
        this(ProtocolVersion.VERSION_DTLS_1_2, epoch, context == null || epoch < 0 ? 0L : context.getNextSequenceNumber(epoch), 0L, false);
        if (fragment == null) {
            throw new NullPointerException("Fragment must not be null");
        }
        if (context == null) {
            throw new NullPointerException("Context must not be null");
        }
        this.setType(type);
        if (cid) {
            this.connectionId = context.getWriteConnectionId();
            this.padding = pad;
        }
        this.setEncodedFragment(context.getWriteState(epoch), fragment);
        if (this.fragmentBytes == null) {
            throw new IllegalArgumentException("Fragment missing encoded bytes!");
        }
    }

    public Record(ContentType type, ProtocolVersion version, long sequenceNumber, DTLSMessage fragment) {
        this(version, 0, sequenceNumber, 0L, false);
        if (fragment == null) {
            throw new NullPointerException("Fragment must not be null");
        }
        this.setType(type);
        this.fragment = fragment;
        this.fragmentBytes = fragment.toByteArray();
        if (this.fragmentBytes == null) {
            throw new IllegalArgumentException("Fragment missing encoded bytes!");
        }
    }

    private Record(ProtocolVersion version, int epoch, long sequenceNumber, long receiveNanos, boolean followUpRecord) {
        if (sequenceNumber > 0xFFFFFFFFFFFFL) {
            throw new IllegalArgumentException("Sequence number must be 48 bits only! " + sequenceNumber);
        }
        if (sequenceNumber < 0L) {
            throw new IllegalArgumentException("Sequence number must not be less than 0! " + sequenceNumber);
        }
        if (epoch < 0) {
            throw new IllegalArgumentException("Epoch must not be less than 0! " + epoch);
        }
        if (version == null) {
            throw new NullPointerException("Version must not be null");
        }
        this.version = version;
        this.epoch = epoch;
        this.sequenceNumber = sequenceNumber;
        this.receiveNanos = receiveNanos;
        this.followUpRecord = followUpRecord;
    }

    public void setAddress(InetSocketAddress peerAddress, InetSocketAddress router) {
        if (this.peerAddress != null) {
            throw new IllegalStateException("Peer's address already available!");
        }
        if (peerAddress == null) {
            throw new NullPointerException("Peer's address must not be null!");
        }
        this.peerAddress = peerAddress;
        this.router = router;
    }

    public byte[] toByteArray() {
        int length = this.fragmentBytes.length + 13;
        if (this.useConnectionId()) {
            length += this.connectionId.length();
        }
        DatagramWriter writer = new DatagramWriter(length);
        if (this.useConnectionId()) {
            writer.write(ContentType.TLS12_CID.getCode(), 8);
        } else {
            writer.write(this.type.getCode(), 8);
        }
        writer.write(this.version.getMajor(), 8);
        writer.write(this.version.getMinor(), 8);
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        if (this.useConnectionId()) {
            writer.writeBytes(this.connectionId.getBytes());
        }
        writer.write(this.fragmentBytes.length, 16);
        writer.writeBytes(this.fragmentBytes);
        return writer.toByteArray();
    }

    public int size() {
        int cid = this.useConnectionId() ? this.connectionId.length() : 0;
        return 13 + cid + this.getFragmentLength();
    }

    public static List<Record> fromReader(DatagramReader reader, ConnectionIdGenerator cidGenerator, long receiveNanos) {
        if (reader == null) {
            throw new NullPointerException("Reader must not be null");
        }
        int datagramLength = reader.bitsLeft() / 8;
        ArrayList<Record> records = new ArrayList<Record>();
        while (reader.bytesAvailable()) {
            ConnectionId connectionId;
            long sequenceNumber;
            int epoch;
            ProtocolVersion version;
            int type;
            block11: {
                if (reader.bitsLeft() < 104) {
                    LOGGER.debug("Received truncated DTLS record(s). Discarding ...");
                    return records;
                }
                type = reader.read(8);
                int major = reader.read(8);
                int minor = reader.read(8);
                version = ProtocolVersion.valueOf(major, minor);
                epoch = reader.read(16);
                sequenceNumber = reader.readLong(48);
                connectionId = null;
                if (type == ContentType.TLS12_CID.getCode()) {
                    if (cidGenerator == null) {
                        LOGGER.debug("Received TLS_CID record, but cid is not supported. Discarding ...");
                        return records;
                    }
                    if (cidGenerator.useConnectionId()) {
                        try {
                            connectionId = cidGenerator.read(reader);
                            if (connectionId == null) {
                                LOGGER.debug("Received TLS_CID record, but cid is not matching. Discarding ...");
                                return records;
                            }
                            break block11;
                        }
                        catch (RuntimeException ex) {
                            LOGGER.debug("Received TLS_CID record, failed to read cid. Discarding ...", (Throwable)ex);
                            return records;
                        }
                    }
                    LOGGER.debug("Received TLS_CID record, but cid is not used. Discarding ...");
                    return records;
                }
            }
            int length = reader.read(16);
            int left = reader.bitsLeft() / 8;
            if (left < length) {
                LOGGER.debug("Received truncated DTLS record(s) ({} bytes, but only {} available). {} records, {} bytes. Discarding ...", new Object[]{length, left, records.size(), datagramLength});
                return records;
            }
            byte[] fragmentBytes = reader.readBytes(length);
            ContentType contentType = ContentType.getTypeByValue(type);
            if (contentType == null) {
                LOGGER.debug("Received DTLS record of unsupported type [{}]. Discarding ...", (Object)type);
                continue;
            }
            records.add(new Record(contentType, version, epoch, sequenceNumber, connectionId, fragmentBytes, receiveNanos, !records.isEmpty()));
        }
        return records;
    }

    public static ConnectionId readConnectionIdFromReader(DatagramReader reader, ConnectionIdGenerator cidGenerator) {
        if (reader == null) {
            throw new NullPointerException("Reader must not be null");
        }
        if (cidGenerator == null) {
            throw new NullPointerException("CID generator must not be null");
        }
        if (!cidGenerator.useConnectionId()) {
            throw new IllegalArgumentException("CID generator must use CID");
        }
        if (reader.bitsLeft() < 104) {
            throw new IllegalArgumentException("Record too small for DTLS header");
        }
        int type = reader.read(8);
        if (type != ContentType.TLS12_CID.getCode()) {
            return null;
        }
        reader.skip(80L);
        ConnectionId connectionId = cidGenerator.read(reader);
        int length = reader.read(16);
        int left = reader.bitsLeft() / 8;
        if (left < length) {
            throw new IllegalArgumentException("Record too small for DTLS length " + length);
        }
        return connectionId;
    }

    protected void writeExplicitNonce(DatagramWriter writer) {
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
    }

    protected byte[] generateAdditionalData(int length) {
        int additionDataLength = 13;
        if (this.useConnectionId()) {
            additionDataLength += this.connectionId.length() + 1;
        }
        DatagramWriter writer = new DatagramWriter(additionDataLength);
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        if (this.useConnectionId()) {
            writer.write(ContentType.TLS12_CID.getCode(), 8);
        } else {
            writer.write(this.type.getCode(), 8);
        }
        writer.write(this.version.getMajor(), 8);
        writer.write(this.version.getMinor(), 8);
        if (this.useConnectionId()) {
            writer.writeBytes(this.connectionId.getBytes());
            writer.write(this.connectionId.length(), 8);
        }
        writer.write(length, 16);
        return writer.toByteArray();
    }

    public boolean isFollowUpRecord() {
        return this.followUpRecord;
    }

    public boolean isNewClientHello() {
        if (0 < this.epoch || this.type != ContentType.HANDSHAKE || 0 == this.fragmentBytes.length) {
            return false;
        }
        HandshakeType handshakeType = HandshakeType.getTypeByCode(this.fragmentBytes[0]);
        return handshakeType == HandshakeType.CLIENT_HELLO;
    }

    public boolean isDecoded() {
        return this.fragment != null;
    }

    public ContentType getType() {
        return this.type;
    }

    public ProtocolVersion getVersion() {
        return this.version;
    }

    public int getEpoch() {
        return this.epoch;
    }

    public long getSequenceNumber() {
        return this.sequenceNumber;
    }

    public int getFragmentLength() {
        return this.fragmentBytes.length;
    }

    public InetSocketAddress getPeerAddress() {
        return this.peerAddress;
    }

    public InetSocketAddress getRouter() {
        return this.router;
    }

    public ConnectionId getConnectionId() {
        return this.connectionId;
    }

    public long getReceiveNanos() {
        return this.receiveNanos;
    }

    public byte[] getFragmentBytes() {
        return this.fragmentBytes;
    }

    public DTLSMessage getFragment() {
        if (this.fragment == null) {
            throw new IllegalStateException("fragment not decoded!");
        }
        return this.fragment;
    }

    public void decodeFragment(DTLSConnectionState readState) throws GeneralSecurityException, HandshakeException {
        if (this.fragment != null) {
            LOGGER.error("DTLS read state already applied!");
            throw new IllegalArgumentException("DTLS read state already applied!");
        }
        ContentType actualType = this.type;
        byte[] decryptedMessage = readState.decrypt(this, this.fragmentBytes);
        if (ContentType.TLS12_CID == this.type) {
            int index;
            for (index = decryptedMessage.length - 1; index >= 0 && decryptedMessage[index] == 0; --index) {
            }
            if (index < 0) {
                throw new GeneralSecurityException("no inner type!");
            }
            byte typeCode = decryptedMessage[index];
            actualType = ContentType.getTypeByValue(typeCode);
            if (actualType == null) {
                throw new GeneralSecurityException("unknown inner type! " + typeCode);
            }
            decryptedMessage = Arrays.copyOf(decryptedMessage, index);
        }
        switch (actualType) {
            case ALERT: {
                this.fragment = AlertMessage.fromByteArray(decryptedMessage);
                break;
            }
            case APPLICATION_DATA: {
                this.fragment = ApplicationMessage.fromByteArray(decryptedMessage);
                break;
            }
            case CHANGE_CIPHER_SPEC: {
                this.fragment = ChangeCipherSpecMessage.fromByteArray(decryptedMessage);
                break;
            }
            case HANDSHAKE: {
                this.fragment = HandshakeMessage.fromByteArray(decryptedMessage);
                break;
            }
            default: {
                LOGGER.debug("Cannot decrypt message of unsupported type [{}]", (Object)this.type);
            }
        }
        this.type = actualType;
    }

    private void setEncodedFragment(DTLSConnectionState outgoingWriteState, DTLSMessage fragment) throws GeneralSecurityException {
        byte[] byteArray = fragment.toByteArray();
        if (byteArray == null) {
            throw new NullPointerException("fragment must not return null");
        }
        if (this.useConnectionId()) {
            int index = byteArray.length;
            byteArray = Arrays.copyOf(byteArray, index + 1 + this.padding);
            byteArray[index] = (byte)this.type.getCode();
        }
        this.fragmentBytes = outgoingWriteState.encrypt(this, byteArray);
        this.fragment = fragment;
    }

    boolean useConnectionId() {
        return this.connectionId != null && !this.connectionId.isEmpty();
    }

    private void setType(ContentType type) {
        if (type == null) {
            throw new NullPointerException("Type must not be null");
        }
        switch (type) {
            case ALERT: 
            case APPLICATION_DATA: 
            case CHANGE_CIPHER_SPEC: 
            case HANDSHAKE: {
                this.type = type;
                break;
            }
            default: {
                throw new IllegalArgumentException("Not supported content type: " + (Object)((Object)type));
            }
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("==[ DTLS Record ]==============================================");
        sb.append(StringUtil.lineSeparator()).append("Content Type: ").append(this.type.toString());
        sb.append(StringUtil.lineSeparator()).append("Peer address: ").append(this.getPeerAddress());
        sb.append(StringUtil.lineSeparator()).append("Version: ").append(this.version.getMajor()).append(", ").append(this.version.getMinor());
        sb.append(StringUtil.lineSeparator()).append("Epoch: ").append(this.epoch);
        sb.append(StringUtil.lineSeparator()).append("Sequence Number: ").append(this.sequenceNumber);
        if (this.connectionId != null) {
            sb.append(StringUtil.lineSeparator()).append("connection id: ").append(this.connectionId.getAsString());
        }
        sb.append(StringUtil.lineSeparator()).append("Length: ").append(this.fragmentBytes.length);
        sb.append(StringUtil.lineSeparator()).append("Fragment:");
        if (this.fragment != null) {
            sb.append(StringUtil.lineSeparator()).append(this.fragment);
        } else {
            sb.append(StringUtil.lineSeparator()).append("fragment is not decrypted yet");
        }
        sb.append(StringUtil.lineSeparator()).append("===============================================================");
        return sb.toString();
    }
}

