/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.cert.CertPath;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.security.auth.x500.X500Principal;
import org.eclipse.californium.elements.util.Asn1DerDecoder;
import org.eclipse.californium.elements.util.Bytes;
import org.eclipse.californium.elements.util.CertPathUtil;
import org.eclipse.californium.elements.util.DatagramReader;
import org.eclipse.californium.elements.util.DatagramWriter;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.CertificateType;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.eclipse.californium.scandium.dtls.cipher.ThreadLocalCertificateFactory;
import org.eclipse.californium.scandium.dtls.cipher.ThreadLocalCryptoMap;
import org.eclipse.californium.scandium.dtls.cipher.ThreadLocalKeyFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CertificateMessage
extends HandshakeMessage {
    private static final String CERTIFICATE_TYPE_X509 = "X.509";
    private static final Logger LOGGER = LoggerFactory.getLogger(CertificateMessage.class);
    private static final int CERTIFICATE_LENGTH_BITS = 24;
    private static final int CERTIFICATE_LIST_LENGTH_BITS = 24;
    private static final ThreadLocalCryptoMap<ThreadLocalKeyFactory> KEY_FACTORIES = new ThreadLocalCryptoMap<ThreadLocalKeyFactory>(new ThreadLocalCryptoMap.Factory<ThreadLocalKeyFactory>(){

        @Override
        public ThreadLocalKeyFactory getInstance(String algorithm) {
            return new ThreadLocalKeyFactory(algorithm);
        }
    });
    private static final ThreadLocalCertificateFactory CERTIFICATE_FACTORY = new ThreadLocalCertificateFactory("X.509");
    private final CertPath certPath;
    private final List<byte[]> encodedChain;
    private final byte[] rawPublicKeyBytes;
    private final PublicKey publicKey;
    private final int length;

    public CertificateMessage(List<X509Certificate> certificateChain, InetSocketAddress peerAddress) {
        this(certificateChain, null, peerAddress);
    }

    public CertificateMessage(List<X509Certificate> certificateChain, List<X500Principal> certificateAuthorities, InetSocketAddress peerAddress) {
        this(CertPathUtil.generateValidatableCertPath(certificateChain, certificateAuthorities), peerAddress);
        if (LOGGER.isDebugEnabled()) {
            int size = this.certPath.getCertificates().size();
            if (size < certificateChain.size()) {
                LOGGER.debug("created CERTIFICATE message with truncated certificate chain [length: {}, full-length: {}]", (Object)size, (Object)certificateChain.size());
            } else {
                LOGGER.debug("created CERTIFICATE message with certificate chain [length: {}]", (Object)size);
            }
        }
    }

    private CertificateMessage(CertPath peerCertChain, InetSocketAddress peerAddress) {
        super(peerAddress);
        if (peerCertChain == null) {
            throw new NullPointerException("Certificate chain must not be null");
        }
        this.rawPublicKeyBytes = null;
        this.certPath = peerCertChain;
        PublicKey publicKey = null;
        List<? extends Certificate> certificates = peerCertChain.getCertificates();
        ArrayList<byte[]> encodedChain = new ArrayList<byte[]>(certificates.size());
        int length = 3;
        if (!certificates.isEmpty()) {
            try {
                for (Certificate certificate : certificates) {
                    if (publicKey == null) {
                        publicKey = certificate.getPublicKey();
                    }
                    byte[] encoded = certificate.getEncoded();
                    encodedChain.add(encoded);
                    length += 3 + encoded.length;
                }
            }
            catch (CertificateEncodingException e) {
                encodedChain = null;
                publicKey = null;
                length = 3;
                LOGGER.warn("Could not encode certificate chain", (Throwable)e);
            }
        }
        this.publicKey = publicKey;
        this.encodedChain = encodedChain;
        this.length = length;
    }

    public CertificateMessage(PublicKey publicKey, InetSocketAddress peerAddress) {
        super(peerAddress);
        this.certPath = null;
        this.encodedChain = null;
        this.rawPublicKeyBytes = publicKey == null ? Bytes.EMPTY : publicKey.getEncoded();
        this.length = 3 + this.rawPublicKeyBytes.length;
        this.publicKey = publicKey;
    }

    public CertificateMessage(byte[] rawPublicKeyBytes, InetSocketAddress peerAddress) {
        super(peerAddress);
        if (rawPublicKeyBytes == null) {
            throw new NullPointerException("Raw public key byte array must not be null");
        }
        this.certPath = null;
        this.encodedChain = null;
        this.rawPublicKeyBytes = Arrays.copyOf(rawPublicKeyBytes, rawPublicKeyBytes.length);
        this.length = 3 + rawPublicKeyBytes.length;
        PublicKey publicKey = null;
        if (rawPublicKeyBytes.length > 0) {
            try {
                String keyAlgorithm = Asn1DerDecoder.readSubjectPublicKeyAlgorithm((byte[])rawPublicKeyBytes);
                if (keyAlgorithm != null) {
                    ThreadLocalKeyFactory factory = KEY_FACTORIES.get(keyAlgorithm);
                    if (factory != null && factory.current() != null) {
                        publicKey = ((KeyFactory)factory.current()).generatePublic(new X509EncodedKeySpec(rawPublicKeyBytes));
                    }
                } else {
                    LOGGER.info("Could not reconstruct the peer's public key [{}]", (Object)StringUtil.byteArray2Hex((byte[])rawPublicKeyBytes));
                }
            }
            catch (GeneralSecurityException e) {
                LOGGER.warn("Could not reconstruct the peer's public key", (Throwable)e);
            }
            catch (IllegalArgumentException e) {
                LOGGER.warn("Could not reconstruct the peer's public key", (Throwable)e);
            }
        }
        this.publicKey = publicKey;
    }

    @Override
    public HandshakeType getMessageType() {
        return HandshakeType.CERTIFICATE;
    }

    @Override
    public int getMessageLength() {
        return this.length;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(super.toString());
        if (this.rawPublicKeyBytes == null && this.certPath != null) {
            sb.append("\t\tCertificate chain length: ").append(this.getMessageLength() - 3).append(StringUtil.lineSeparator());
            int index = 0;
            for (Certificate certificate : this.certPath.getCertificates()) {
                sb.append("\t\t\tCertificate Length: ").append(this.encodedChain.get(index).length).append(StringUtil.lineSeparator());
                sb.append("\t\t\tCertificate: ").append(certificate).append(StringUtil.lineSeparator());
                ++index;
            }
        } else if (this.rawPublicKeyBytes != null && this.certPath == null) {
            sb.append("\t\tRaw Public Key: ");
            sb.append(this.getPublicKey().toString());
            sb.append(StringUtil.lineSeparator());
        }
        return sb.toString();
    }

    public CertPath getCertificateChain() {
        return this.certPath;
    }

    @Override
    public byte[] fragmentToByteArray() {
        DatagramWriter writer = new DatagramWriter();
        if (this.rawPublicKeyBytes == null) {
            writer.write(this.getMessageLength() - 3, 24);
            for (byte[] encoded : this.encodedChain) {
                writer.write(encoded.length, 24);
                writer.writeBytes(encoded);
            }
        } else {
            writer.write(this.rawPublicKeyBytes.length, 24);
            writer.writeBytes(this.rawPublicKeyBytes);
        }
        return writer.toByteArray();
    }

    public static CertificateMessage fromReader(DatagramReader reader, CertificateType certificateType, InetSocketAddress peerAddress) throws HandshakeException {
        if (CertificateType.RAW_PUBLIC_KEY == certificateType) {
            LOGGER.debug("Parsing RawPublicKey CERTIFICATE message");
            int certificateLength = reader.read(24);
            byte[] rawPublicKey = reader.readBytes(certificateLength);
            return new CertificateMessage(rawPublicKey, peerAddress);
        }
        if (CertificateType.X_509 == certificateType) {
            return CertificateMessage.readX509CertificateMessage(reader, peerAddress);
        }
        throw new IllegalArgumentException("Certificate type " + (Object)((Object)certificateType) + " not supported!");
    }

    private static CertificateMessage readX509CertificateMessage(DatagramReader reader, InetSocketAddress peerAddress) throws HandshakeException {
        LOGGER.debug("Parsing X.509 CERTIFICATE message");
        ArrayList<Certificate> certs = new ArrayList<Certificate>();
        int certificateChainLength = reader.read(24);
        DatagramReader rangeReader = reader.createRangeReader(certificateChainLength);
        try {
            CertificateFactory factory = (CertificateFactory)CERTIFICATE_FACTORY.currentWithCause();
            while (rangeReader.bytesAvailable()) {
                int certificateLength = rangeReader.read(24);
                certs.add(factory.generateCertificate(rangeReader.createRangeInputStream(certificateLength)));
            }
            return new CertificateMessage(factory.generateCertPath(certs), peerAddress);
        }
        catch (GeneralSecurityException e) {
            throw new HandshakeException("Cannot parse X.509 certificate chain provided by peer", new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.BAD_CERTIFICATE, peerAddress), e);
        }
    }

    public PublicKey getPublicKey() {
        return this.publicKey;
    }
}

