/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.eclipse.californium.elements.util.DatagramReader;
import org.eclipse.californium.elements.util.DatagramWriter;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.ApplicationMessage;
import org.eclipse.californium.scandium.dtls.ChangeCipherSpecMessage;
import org.eclipse.californium.scandium.dtls.ConnectionId;
import org.eclipse.californium.scandium.dtls.ConnectionIdGenerator;
import org.eclipse.californium.scandium.dtls.ContentType;
import org.eclipse.californium.scandium.dtls.DTLSConnectionState;
import org.eclipse.californium.scandium.dtls.DTLSMessage;
import org.eclipse.californium.scandium.dtls.DTLSSession;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeParameter;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.eclipse.californium.scandium.dtls.ProtocolVersion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Record {
    private static final Logger LOGGER = LoggerFactory.getLogger(Record.class);
    public static final int CONTENT_TYPE_BITS = 8;
    public static final int VERSION_BITS = 8;
    public static final int EPOCH_BITS = 16;
    public static final int SEQUENCE_NUMBER_BITS = 48;
    public static final int LENGTH_BITS = 16;
    public static final int CID_LENGTH_BITS = 8;
    public static final int RECORD_HEADER_BITS = 104;
    private static final long MAX_SEQUENCE_NO = 0xFFFFFFFFFFFFL;
    private ContentType type;
    private final ProtocolVersion version;
    private final int epoch;
    private long sequenceNumber;
    private final long receiveNanos;
    private DTLSMessage fragment;
    private byte[] fragmentBytes;
    private ConnectionId connectionId;
    private int padding;
    private final DTLSConnectionState outgoingWriteState;
    private DTLSSession incomingSession;
    private DTLSConnectionState incomingReadState;
    private final InetSocketAddress peerAddress;

    Record(ContentType type, ProtocolVersion version, int epoch, long sequenceNumber, ConnectionId connectionId, byte[] fragmentBytes, InetSocketAddress peerAddress, long receiveNanos) {
        this(version, epoch, sequenceNumber, receiveNanos, null, peerAddress);
        if (type == null) {
            throw new NullPointerException("Type must not be null");
        }
        if (fragmentBytes == null) {
            throw new NullPointerException("Fragment bytes must not be null");
        }
        if (peerAddress == null) {
            throw new NullPointerException("Peer address must not be null");
        }
        this.type = type;
        this.connectionId = connectionId;
        this.fragmentBytes = fragmentBytes;
    }

    public Record(ContentType type, int epoch, long sequenceNumber, DTLSMessage fragment, DTLSSession session, boolean cid, int pad) throws GeneralSecurityException {
        this(new ProtocolVersion(), epoch, sequenceNumber, 0L, session, null);
        if (fragment == null) {
            throw new NullPointerException("Fragment must not be null");
        }
        if (session == null) {
            throw new NullPointerException("Session must not be null");
        }
        if (session.getPeer() == null) {
            throw new IllegalArgumentException("Session's peer address must not be null");
        }
        this.setType(type);
        if (cid) {
            this.connectionId = session.getWriteConnectionId();
            this.padding = pad;
        }
        this.setEncodedFragment(fragment);
        if (this.fragmentBytes == null) {
            throw new IllegalArgumentException("Fragment missing encoded bytes!");
        }
    }

    public Record(ContentType type, long sequenceNumber, DTLSMessage fragment, InetSocketAddress peerAddress) {
        this(new ProtocolVersion(), 0, sequenceNumber, 0L, null, peerAddress);
        if (fragment == null) {
            throw new NullPointerException("Fragment must not be null");
        }
        if (peerAddress == null) {
            throw new NullPointerException("Peer address must not be null");
        }
        this.setType(type);
        this.fragment = fragment;
        this.fragmentBytes = fragment.toByteArray();
        if (this.fragmentBytes == null) {
            throw new IllegalArgumentException("Fragment missing encoded bytes!");
        }
    }

    private Record(ProtocolVersion version, int epoch, long sequenceNumber, long receiveNanos, DTLSSession session, InetSocketAddress peer) {
        if (sequenceNumber > 0xFFFFFFFFFFFFL) {
            throw new IllegalArgumentException("Sequence number must be 48 bits only! " + sequenceNumber);
        }
        if (sequenceNumber < 0L) {
            throw new IllegalArgumentException("Sequence number must not be less than 0! " + sequenceNumber);
        }
        if (epoch < 0) {
            throw new IllegalArgumentException("Epoch must not be less than 0! " + epoch);
        }
        if (version == null) {
            throw new NullPointerException("Version must not be null");
        }
        this.version = version;
        this.epoch = epoch;
        this.sequenceNumber = sequenceNumber;
        this.receiveNanos = receiveNanos;
        this.outgoingWriteState = session == null ? null : session.getWriteState();
        this.peerAddress = peer == null && session != null ? session.getPeer() : peer;
    }

    public byte[] toByteArray() {
        DatagramWriter writer = new DatagramWriter();
        if (this.useConnectionId()) {
            writer.write(ContentType.TLS12_CID.getCode(), 8);
        } else {
            writer.write(this.type.getCode(), 8);
        }
        writer.write(this.version.getMajor(), 8);
        writer.write(this.version.getMinor(), 8);
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        if (this.useConnectionId()) {
            writer.writeBytes(this.connectionId.getBytes());
        }
        writer.write(this.fragmentBytes.length, 16);
        writer.writeBytes(this.fragmentBytes);
        return writer.toByteArray();
    }

    public int size() {
        return 13 + this.getFragmentLength();
    }

    public static List<Record> fromByteArray(byte[] byteArray, InetSocketAddress peerAddress, ConnectionIdGenerator cidGenerator, long receiveNanos) {
        if (byteArray == null) {
            throw new NullPointerException("Byte array must not be null");
        }
        if (peerAddress == null) {
            throw new NullPointerException("Peer address must not be null");
        }
        ArrayList<Record> records = new ArrayList<Record>();
        DatagramReader reader = new DatagramReader(byteArray, false);
        while (reader.bytesAvailable()) {
            ConnectionId connectionId;
            long sequenceNumber;
            int epoch;
            ProtocolVersion version;
            int type;
            block12: {
                if (reader.bitsLeft() < 104) {
                    LOGGER.debug("Received truncated DTLS record(s). Discarding ...");
                    return records;
                }
                type = reader.read(8);
                int major = reader.read(8);
                int minor = reader.read(8);
                version = new ProtocolVersion(major, minor);
                epoch = reader.read(16);
                sequenceNumber = reader.readLong(48);
                connectionId = null;
                if (type == ContentType.TLS12_CID.getCode()) {
                    if (cidGenerator == null) {
                        LOGGER.debug("Received TLS_CID record, but cid is not supported. Discarding ...");
                        return records;
                    }
                    if (cidGenerator.useConnectionId()) {
                        try {
                            connectionId = cidGenerator.read(reader);
                            if (connectionId == null) {
                                LOGGER.debug("Received TLS_CID record, but cid is not matching. Discarding ...");
                                return records;
                            }
                            break block12;
                        }
                        catch (RuntimeException ex) {
                            LOGGER.debug("Received TLS_CID record, failed to read cid. Discarding ...", (Object)ex.getMessage());
                            return records;
                        }
                    }
                    LOGGER.debug("Received TLS_CID record, but cid is not used. Discarding ...");
                    return records;
                }
            }
            int length = reader.read(16);
            if (reader.bitsLeft() < length * 8) {
                LOGGER.debug("Received truncated DTLS record(s). Discarding ...");
                return records;
            }
            byte[] fragmentBytes = reader.readBytes(length);
            ContentType contentType = ContentType.getTypeByValue(type);
            if (contentType == null) {
                LOGGER.debug("Received DTLS record of unsupported type [{}]. Discarding ...", (Object)type);
                continue;
            }
            records.add(new Record(contentType, version, epoch, sequenceNumber, connectionId, fragmentBytes, peerAddress, receiveNanos));
        }
        return records;
    }

    protected byte[] generateExplicitNonce() {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        return writer.toByteArray();
    }

    protected byte[] generateAdditionalData(int length) {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        if (this.useConnectionId()) {
            writer.write(ContentType.TLS12_CID.getCode(), 8);
        } else {
            writer.write(this.type.getCode(), 8);
        }
        writer.write(this.version.getMajor(), 8);
        writer.write(this.version.getMinor(), 8);
        if (this.useConnectionId()) {
            writer.writeBytes(this.connectionId.getBytes());
            writer.write(this.connectionId.length(), 8);
        }
        writer.write(length, 16);
        return writer.toByteArray();
    }

    public boolean isNewClientHello() {
        if (0 < this.epoch || this.type != ContentType.HANDSHAKE || 0 == this.fragmentBytes.length) {
            return false;
        }
        HandshakeType handshakeType = HandshakeType.getTypeByCode(this.fragmentBytes[0]);
        return handshakeType == HandshakeType.CLIENT_HELLO;
    }

    public ContentType getType() {
        return this.type;
    }

    public ProtocolVersion getVersion() {
        return this.version;
    }

    public int getEpoch() {
        return this.epoch;
    }

    public long getSequenceNumber() {
        return this.sequenceNumber;
    }

    public void updateSequenceNumber(long sequenceNumber) throws GeneralSecurityException {
        if (sequenceNumber > 0xFFFFFFFFFFFFL) {
            throw new IllegalArgumentException("Sequence number must have max 48 bits");
        }
        if (sequenceNumber < 0L) {
            throw new IllegalArgumentException("Sequence number must not be smaller than 0! " + sequenceNumber);
        }
        if (this.fragment == null) {
            throw new IllegalStateException("Fragment must not be null!");
        }
        if (this.outgoingWriteState == null) {
            throw new IllegalStateException("Write state must not be null!");
        }
        if (this.sequenceNumber != sequenceNumber) {
            this.sequenceNumber = sequenceNumber;
            if (this.epoch > 0) {
                this.setEncodedFragment(this.fragment);
            }
        }
    }

    public int getFragmentLength() {
        return this.fragmentBytes.length;
    }

    public void applySession(DTLSSession session) throws GeneralSecurityException, HandshakeException {
        DTLSConnectionState readState;
        int readEpoch;
        if (session == null) {
            readEpoch = 0;
            readState = DTLSConnectionState.NULL;
        } else {
            readEpoch = session.getReadEpoch();
            readState = session.getReadState();
        }
        if (readEpoch != this.epoch) {
            throw new IllegalArgumentException("session for different epoch! session " + readEpoch + " != record " + this.epoch);
        }
        if (this.fragment != null) {
            if (this.incomingReadState != readState) {
                LOGGER.error("{} != {}", (Object)readState, (Object)this.incomingReadState);
                throw new IllegalArgumentException("session read state changed!");
            }
        } else {
            this.incomingSession = session;
            this.incomingReadState = readState;
            this.decodeFragment();
        }
    }

    public InetSocketAddress getPeerAddress() {
        if (this.peerAddress == null) {
            throw new NullPointerException("missing peer address!");
        }
        return this.peerAddress;
    }

    public ConnectionId getConnectionId() {
        return this.connectionId;
    }

    public long getReceiveNanos() {
        return this.receiveNanos;
    }

    public byte[] getFragmentBytes() {
        return this.fragmentBytes;
    }

    public DTLSMessage getFragment() {
        if (this.fragment == null) {
            throw new IllegalStateException("fragment not decoded!");
        }
        return this.fragment;
    }

    private void decodeFragment() throws GeneralSecurityException, HandshakeException {
        ContentType actualType = this.type;
        byte[] decryptedMessage = this.incomingReadState.decrypt(this, this.fragmentBytes);
        if (ContentType.TLS12_CID == this.type) {
            int index;
            for (index = decryptedMessage.length - 1; index >= 0 && decryptedMessage[index] == 0; --index) {
            }
            if (index < 0) {
                throw new GeneralSecurityException("no inner type!");
            }
            byte typeCode = decryptedMessage[index];
            actualType = ContentType.getTypeByValue(typeCode);
            if (actualType == null) {
                throw new GeneralSecurityException("unknown inner type! " + typeCode);
            }
            decryptedMessage = Arrays.copyOf(decryptedMessage, index);
        }
        switch (actualType) {
            case ALERT: {
                this.fragment = AlertMessage.fromByteArray(decryptedMessage, this.getPeerAddress());
                break;
            }
            case APPLICATION_DATA: {
                this.fragment = ApplicationMessage.fromByteArray(decryptedMessage, this.getPeerAddress());
                break;
            }
            case CHANGE_CIPHER_SPEC: {
                this.fragment = ChangeCipherSpecMessage.fromByteArray(decryptedMessage, this.getPeerAddress());
                break;
            }
            case HANDSHAKE: {
                this.fragment = this.handshakeMessageFromByteArray(decryptedMessage);
                break;
            }
            default: {
                LOGGER.warn("Cannot decrypt message of unsupported type [{}]", (Object)this.type);
            }
        }
        this.type = actualType;
    }

    private DTLSMessage handshakeMessageFromByteArray(byte[] decryptedMessage) throws GeneralSecurityException, HandshakeException {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Parsing HANDSHAKE message plaintext{}{}", (Object)StringUtil.lineSeparator(), (Object)StringUtil.byteArray2HexString((byte[])decryptedMessage));
        }
        HandshakeParameter parameter = null;
        if (this.incomingSession != null) {
            parameter = this.incomingSession.getParameter();
            LOGGER.debug("Parsing HANDSHAKE message plaintext with parameter [{}]", (Object)parameter);
        } else {
            LOGGER.debug("Parsing HANDSHAKE message without a session");
        }
        return HandshakeMessage.fromByteArray(decryptedMessage, parameter, this.getPeerAddress());
    }

    private void setEncodedFragment(DTLSMessage fragment) throws GeneralSecurityException {
        byte[] byteArray = fragment.toByteArray();
        if (byteArray == null) {
            throw new NullPointerException("fragment must not return null");
        }
        if (this.useConnectionId()) {
            int index = byteArray.length;
            byteArray = Arrays.copyOf(byteArray, index + 1 + this.padding);
            byteArray[index] = (byte)this.type.getCode();
        }
        this.fragmentBytes = this.outgoingWriteState.encrypt(this, byteArray);
        this.fragment = fragment;
    }

    boolean useConnectionId() {
        return this.connectionId != null && !this.connectionId.isEmpty();
    }

    private void setType(ContentType type) {
        if (type == null) {
            throw new NullPointerException("Type must not be null");
        }
        switch (type) {
            case ALERT: 
            case APPLICATION_DATA: 
            case CHANGE_CIPHER_SPEC: 
            case HANDSHAKE: {
                this.type = type;
                break;
            }
            default: {
                throw new IllegalArgumentException("Not supported content type: " + (Object)((Object)type));
            }
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("==[ DTLS Record ]==============================================");
        sb.append(StringUtil.lineSeparator()).append("Content Type: ").append(this.type.toString());
        sb.append(StringUtil.lineSeparator()).append("Peer address: ").append(this.getPeerAddress());
        sb.append(StringUtil.lineSeparator()).append("Version: ").append(this.version.getMajor()).append(", ").append(this.version.getMinor());
        sb.append(StringUtil.lineSeparator()).append("Epoch: ").append(this.epoch);
        sb.append(StringUtil.lineSeparator()).append("Sequence Number: ").append(this.sequenceNumber);
        if (this.connectionId != null) {
            sb.append(StringUtil.lineSeparator()).append("connection id: ").append(this.connectionId.getAsString());
        }
        sb.append(StringUtil.lineSeparator()).append("Length: ").append(this.fragmentBytes.length);
        sb.append(StringUtil.lineSeparator()).append("Fragment:");
        if (this.fragment != null) {
            sb.append(StringUtil.lineSeparator()).append(this.fragment);
        } else {
            sb.append(StringUtil.lineSeparator()).append("fragment is not decrypted yet");
        }
        sb.append(StringUtil.lineSeparator()).append("===============================================================");
        return sb.toString();
    }
}

