/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicReference;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.auth.RawPublicKeyIdentity;
import org.eclipse.californium.elements.util.DatagramWriter;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.config.DtlsConnectorConfig;
import org.eclipse.californium.scandium.dtls.AbstractMessage;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.CertificateMessage;
import org.eclipse.californium.scandium.dtls.ChangeCipherSpecMessage;
import org.eclipse.californium.scandium.dtls.ContentType;
import org.eclipse.californium.scandium.dtls.DTLSConnectionState;
import org.eclipse.californium.scandium.dtls.DTLSFlight;
import org.eclipse.californium.scandium.dtls.DTLSMessage;
import org.eclipse.californium.scandium.dtls.DTLSSession;
import org.eclipse.californium.scandium.dtls.FragmentedHandshakeMessage;
import org.eclipse.californium.scandium.dtls.GenericHandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeParameter;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.eclipse.californium.scandium.dtls.ProtocolVersion;
import org.eclipse.californium.scandium.dtls.Random;
import org.eclipse.californium.scandium.dtls.Record;
import org.eclipse.californium.scandium.dtls.RecordLayer;
import org.eclipse.californium.scandium.dtls.SessionListener;
import org.eclipse.californium.scandium.dtls.cipher.CipherSuite;
import org.eclipse.californium.scandium.dtls.cipher.ECDHECryptography;
import org.eclipse.californium.scandium.dtls.cipher.PseudoRandomFunction;
import org.eclipse.californium.scandium.dtls.rpkstore.TrustedRpkStore;
import org.eclipse.californium.scandium.dtls.x509.CertificateVerifier;
import org.eclipse.californium.scandium.util.ByteArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class Handshaker {
    private static final String MESSAGE_DIGEST_ALGORITHM_NAME = "SHA-256";
    private static final Logger LOGGER = LoggerFactory.getLogger((String)Handshaker.class.getName());
    protected final boolean isClient;
    protected int state = -1;
    protected ProtocolVersion usedProtocol;
    protected Random clientRandom;
    protected Random serverRandom;
    protected ECDHECryptography ecdhe;
    private byte[] masterSecret;
    private SecretKey clientWriteMACKey;
    private SecretKey serverWriteMACKey;
    private IvParameterSpec clientWriteIV;
    private IvParameterSpec serverWriteIV;
    private SecretKey clientWriteKey;
    private SecretKey serverWriteKey;
    protected final DTLSSession session;
    protected final CertificateVerifier certificateVerifier;
    protected final TrustedRpkStore rpkStore;
    private int sequenceNumber = 0;
    private int nextReceiveSeq = 0;
    protected int flightNumber = 0;
    private final int maxDeferredProcessedApplicationDataMessages;
    private final List<Object> deferredApplicationData;
    private final AtomicReference<DTLSFlight> pendingFlight = new AtomicReference();
    private final RecordLayer recordLayer;
    protected InboundMessageBuffer inboundMessageBuffer;
    protected Map<Integer, SortedSet<FragmentedHandshakeMessage>> fragmentedMessages = new HashMap<Integer, SortedSet<FragmentedHandshakeMessage>>();
    protected MessageDigest md;
    protected byte[] handshakeMessages = new byte[0];
    protected PrivateKey privateKey;
    protected PublicKey publicKey;
    protected List<X509Certificate> certificateChain;
    protected boolean sniEnabled = true;
    private Set<SessionListener> sessionListeners = new LinkedHashSet<SessionListener>();
    private boolean changeCipherSuiteMessageExpected = false;
    private boolean sessionEstablished = false;

    protected Handshaker(boolean isClient, DTLSSession session, RecordLayer recordLayer, SessionListener sessionListener, DtlsConnectorConfig config, int maxTransmissionUnit) {
        this(isClient, 0, session, recordLayer, sessionListener, config, maxTransmissionUnit);
    }

    protected Handshaker(boolean isClient, int initialMessageSeq, DTLSSession session, RecordLayer recordLayer, SessionListener sessionListener, DtlsConnectorConfig config, int maxTransmissionUnit) {
        if (session == null) {
            throw new NullPointerException("DTLS Session must not be null");
        }
        if (recordLayer == null) {
            throw new NullPointerException("Record layer must not be null");
        }
        if (config == null) {
            throw new NullPointerException("Dtls Connector Config must not be null");
        }
        if (initialMessageSeq < 0) {
            throw new IllegalArgumentException("Initial message sequence number must not be negative");
        }
        this.isClient = isClient;
        this.sequenceNumber = initialMessageSeq;
        this.nextReceiveSeq = initialMessageSeq;
        this.session = session;
        this.recordLayer = recordLayer;
        this.maxDeferredProcessedApplicationDataMessages = config.getMaxDeferredProcessedApplicationDataMessages();
        this.deferredApplicationData = new ArrayList<Object>(this.maxDeferredProcessedApplicationDataMessages);
        this.addSessionListener(sessionListener);
        this.certificateVerifier = config.getCertificateVerifier();
        this.session.setMaxTransmissionUnit(maxTransmissionUnit);
        this.inboundMessageBuffer = new InboundMessageBuffer();
        try {
            this.md = MessageDigest.getInstance(MESSAGE_DIGEST_ALGORITHM_NAME);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException(String.format("Message digest algorithm %s is not available on JVM", MESSAGE_DIGEST_ALGORITHM_NAME));
        }
        this.rpkStore = config.getRpkTrustStore();
    }

    public final void processMessage(Record record) throws HandshakeException {
        boolean sameEpoch;
        boolean bl = sameEpoch = this.session.getReadEpoch() == record.getEpoch();
        if (sameEpoch && !this.session.isDuplicate(record.getSequenceNumber()) || this.session.getReadEpoch() + 1 == record.getEpoch()) {
            try {
                record.setSession(this.session);
                DTLSMessage messageToProcess = this.inboundMessageBuffer.getNextMessage(record);
                while (messageToProcess != null) {
                    if (messageToProcess instanceof FragmentedHandshakeMessage) {
                        messageToProcess = this.handleFragmentation((FragmentedHandshakeMessage)messageToProcess);
                    }
                    if (messageToProcess != null) {
                        DTLSFlight flight;
                        if (messageToProcess instanceof GenericHandshakeMessage) {
                            HandshakeParameter parameter = this.session.getParameter();
                            if (parameter == null) {
                                throw new IllegalStateException("handshake parameter are required!");
                            }
                            messageToProcess = ((GenericHandshakeMessage)messageToProcess).getSpecificHandshakeMessage(parameter);
                        }
                        if (messageToProcess.getContentType() == ContentType.HANDSHAKE && (flight = this.pendingFlight.get()) != null) {
                            LOGGER.debug("response for flight {} started", (Object)flight.getFlightNumber());
                            flight.setResponseStarted();
                        }
                        this.doProcessMessage(messageToProcess);
                    }
                    messageToProcess = this.inboundMessageBuffer.getNextMessage();
                }
                this.session.markRecordAsRead(record.getEpoch(), record.getSequenceNumber());
            }
            catch (GeneralSecurityException e) {
                LOGGER.warn("Cannot process handshake message from peer [{}] due to [{}]", new Object[]{this.getSession().getPeer(), e.getMessage(), e});
                AlertMessage alert = new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.INTERNAL_ERROR, this.session.getPeer());
                throw new HandshakeException("Cannot process handshake message", alert);
            }
        } else if (sameEpoch) {
            LOGGER.trace("Discarding duplicate HANDSHAKE message received from peer [{}]:{}{}", new Object[]{record.getPeerAddress(), StringUtil.lineSeparator(), record});
        } else {
            LOGGER.trace("Discarding HANDSHAKE message with wrong epoch received from peer [{}]:{}{}", new Object[]{record.getPeerAddress(), StringUtil.lineSeparator(), record});
        }
    }

    protected abstract void doProcessMessage(DTLSMessage var1) throws HandshakeException, GeneralSecurityException;

    public abstract void startHandshake() throws HandshakeException;

    protected final void generateKeys(byte[] premasterSecret) {
        this.masterSecret = this.generateMasterSecret(premasterSecret);
        this.session.setMasterSecret(this.masterSecret);
        this.calculateKeys(this.masterSecret);
    }

    protected void calculateKeys(byte[] masterSecret) {
        byte[] seed = ByteArrayUtils.concatenate(this.serverRandom.getRandomBytes(), this.clientRandom.getRandomBytes());
        byte[] data = PseudoRandomFunction.doPRF(masterSecret, PseudoRandomFunction.Label.KEY_EXPANSION_LABEL, seed);
        int macKeyLength = this.session.getCipherSuite().getMacKeyLength();
        int encKeyLength = this.session.getCipherSuite().getEncKeyLength();
        int fixedIvLength = this.session.getCipherSuite().getFixedIvLength();
        this.clientWriteMACKey = new SecretKeySpec(data, 0, macKeyLength, "Mac");
        this.serverWriteMACKey = new SecretKeySpec(data, macKeyLength, macKeyLength, "Mac");
        this.clientWriteKey = new SecretKeySpec(data, 2 * macKeyLength, encKeyLength, "AES");
        this.serverWriteKey = new SecretKeySpec(data, 2 * macKeyLength + encKeyLength, encKeyLength, "AES");
        this.clientWriteIV = new IvParameterSpec(data, 2 * macKeyLength + 2 * encKeyLength, fixedIvLength);
        this.serverWriteIV = new IvParameterSpec(data, 2 * macKeyLength + 2 * encKeyLength + fixedIvLength, fixedIvLength);
    }

    private byte[] generateMasterSecret(byte[] premasterSecret) {
        byte[] randomSeed = ByteArrayUtils.concatenate(this.clientRandom.getRandomBytes(), this.serverRandom.getRandomBytes());
        return PseudoRandomFunction.doPRF(premasterSecret, PseudoRandomFunction.Label.MASTER_SECRET_LABEL, randomSeed);
    }

    protected final byte[] generatePremasterSecretFromPSK(byte[] psk, byte[] otherSecret) {
        int pskLength = psk.length;
        byte[] other = otherSecret == null ? new byte[pskLength] : otherSecret;
        DatagramWriter writer = new DatagramWriter();
        writer.write(other.length, 16);
        writer.writeBytes(other);
        writer.write(pskLength, 16);
        writer.writeBytes(psk);
        return writer.toByteArray();
    }

    protected final void setCurrentReadState() {
        DTLSConnectionState connectionState = this.isClient ? new DTLSConnectionState(this.session.getCipherSuite(), this.session.getCompressionMethod(), this.serverWriteKey, this.serverWriteIV, this.serverWriteMACKey) : new DTLSConnectionState(this.session.getCipherSuite(), this.session.getCompressionMethod(), this.clientWriteKey, this.clientWriteIV, this.clientWriteMACKey);
        this.session.setReadState(connectionState);
    }

    protected final void setCurrentWriteState() {
        DTLSConnectionState connectionState = this.isClient ? new DTLSConnectionState(this.session.getCipherSuite(), this.session.getCompressionMethod(), this.clientWriteKey, this.clientWriteIV, this.clientWriteMACKey) : new DTLSConnectionState(this.session.getCipherSuite(), this.session.getCompressionMethod(), this.serverWriteKey, this.serverWriteIV, this.serverWriteMACKey);
        this.session.setWriteState(connectionState);
    }

    protected final List<Record> wrapMessage(DTLSMessage fragment) throws HandshakeException {
        try {
            switch (fragment.getContentType()) {
                case HANDSHAKE: {
                    return this.wrapHandshakeMessage((HandshakeMessage)fragment);
                }
            }
            ArrayList<Record> records = new ArrayList<Record>();
            records.add(new Record(fragment.getContentType(), this.session.getWriteEpoch(), this.session.getSequenceNumber(), fragment, this.session));
            return records;
        }
        catch (GeneralSecurityException e) {
            throw new HandshakeException("Cannot create record", new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.INTERNAL_ERROR, this.session.getPeer()));
        }
    }

    private List<Record> wrapHandshakeMessage(HandshakeMessage handshakeMessage) throws GeneralSecurityException {
        this.setSequenceNumber(handshakeMessage);
        ArrayList<Record> result = new ArrayList<Record>();
        byte[] messageBytes = handshakeMessage.fragmentToByteArray();
        if (messageBytes.length <= this.session.getMaxFragmentLength()) {
            result.add(new Record(ContentType.HANDSHAKE, this.session.getWriteEpoch(), this.session.getSequenceNumber(), (DTLSMessage)handshakeMessage, this.session));
        } else {
            LOGGER.debug("Splitting up {} message for [{}] into multiple fragments of max {} bytes", new Object[]{handshakeMessage.getMessageType(), handshakeMessage.getPeer(), this.session.getMaxFragmentLength()});
            int messageSeq = handshakeMessage.getMessageSeq();
            int numFragments = messageBytes.length / this.session.getMaxFragmentLength() + 1;
            int offset = 0;
            for (int i = 0; i < numFragments; ++i) {
                int fragmentLength = this.session.getMaxFragmentLength();
                if (offset + fragmentLength > messageBytes.length) {
                    fragmentLength = messageBytes.length - offset;
                }
                byte[] fragmentBytes = new byte[fragmentLength];
                System.arraycopy(messageBytes, offset, fragmentBytes, 0, fragmentLength);
                FragmentedHandshakeMessage fragmentedMessage = new FragmentedHandshakeMessage(fragmentBytes, handshakeMessage.getMessageType(), offset, messageBytes.length, this.session.getPeer());
                fragmentedMessage.setMessageSeq(messageSeq);
                offset += fragmentBytes.length;
                result.add(new Record(ContentType.HANDSHAKE, this.session.getWriteEpoch(), this.session.getSequenceNumber(), (DTLSMessage)fragmentedMessage, this.session));
            }
        }
        return result;
    }

    protected final HandshakeMessage handleFragmentation(FragmentedHandshakeMessage fragment) throws HandshakeException {
        LOGGER.debug("Processing {} message fragment ...", (Object)fragment.getMessageType());
        HandshakeMessage reassembledMessage = null;
        int messageSeq = fragment.getMessageSeq();
        SortedSet<FragmentedHandshakeMessage> existingFragments = this.fragmentedMessages.get(messageSeq);
        if (existingFragments == null) {
            existingFragments = new TreeSet<FragmentedHandshakeMessage>(new Comparator<FragmentedHandshakeMessage>(){

                @Override
                public int compare(FragmentedHandshakeMessage o1, FragmentedHandshakeMessage o2) {
                    if (o1.getFragmentOffset() == o2.getFragmentOffset()) {
                        return 0;
                    }
                    if (o1.getFragmentOffset() < o2.getFragmentOffset()) {
                        return -1;
                    }
                    return 1;
                }
            });
            this.fragmentedMessages.put(messageSeq, existingFragments);
        }
        existingFragments.add(fragment);
        reassembledMessage = this.reassembleFragments(messageSeq, existingFragments, fragment.getMessageLength(), fragment.getMessageType(), this.session);
        if (reassembledMessage != null) {
            LOGGER.debug("Successfully re-assembled {} message", (Object)reassembledMessage.getMessageType());
            this.fragmentedMessages.remove(messageSeq);
        }
        return reassembledMessage;
    }

    private final HandshakeMessage reassembleFragments(int messageSeq, SortedSet<FragmentedHandshakeMessage> fragments, int totalLength, HandshakeType type, DTLSSession session) throws HandshakeException {
        HandshakeMessage message = null;
        byte[] reassembly = new byte[]{};
        int offset = 0;
        for (FragmentedHandshakeMessage fragmentedHandshakeMessage : fragments) {
            int fragmentOffset = fragmentedHandshakeMessage.getFragmentOffset();
            int fragmentLength = fragmentedHandshakeMessage.getFragmentLength();
            if (fragmentOffset == offset) {
                reassembly = ByteArrayUtils.concatenate(reassembly, fragmentedHandshakeMessage.fragmentToByteArray());
                offset = reassembly.length;
                continue;
            }
            if (fragmentOffset >= offset || fragmentOffset + fragmentLength <= offset) continue;
            int newOffset = offset - fragmentOffset;
            int newLength = fragmentLength - newOffset;
            byte[] newBytes = new byte[newLength];
            System.arraycopy(fragmentedHandshakeMessage.fragmentToByteArray(), newOffset, newBytes, 0, newLength);
            reassembly = ByteArrayUtils.concatenate(reassembly, newBytes);
            offset = reassembly.length;
        }
        if (reassembly.length == totalLength) {
            FragmentedHandshakeMessage wholeMessage = new FragmentedHandshakeMessage(type, totalLength, messageSeq, 0, reassembly, this.getPeerAddress());
            reassembly = wholeMessage.toByteArray();
            HandshakeParameter parameter = null;
            if (session != null) {
                parameter = session.getParameter();
            }
            message = HandshakeMessage.fromByteArray(reassembly, parameter, this.getPeerAddress());
        }
        return message;
    }

    protected final CipherSuite.KeyExchangeAlgorithm getKeyExchangeAlgorithm() {
        return this.session.getKeyExchange();
    }

    final byte[] getMasterSecret() {
        return this.masterSecret;
    }

    final SecretKey getClientWriteMACKey() {
        return this.clientWriteMACKey;
    }

    final SecretKey getServerWriteMACKey() {
        return this.serverWriteMACKey;
    }

    final IvParameterSpec getClientWriteIV() {
        return this.clientWriteIV;
    }

    final IvParameterSpec getServerWriteIV() {
        return this.serverWriteIV;
    }

    final SecretKey getClientWriteKey() {
        return this.clientWriteKey;
    }

    final SecretKey getServerWriteKey() {
        return this.serverWriteKey;
    }

    public final DTLSSession getSession() {
        return this.session;
    }

    public final InetSocketAddress getPeerAddress() {
        return this.session.getPeer();
    }

    private void setSequenceNumber(HandshakeMessage message) {
        message.setMessageSeq(this.sequenceNumber);
        ++this.sequenceNumber;
    }

    final int getNextReceiveSeq() {
        return this.nextReceiveSeq;
    }

    final void incrementNextReceiveSeq() {
        ++this.nextReceiveSeq;
    }

    public void addApplicationDataForDeferredProcessing(RawData outgoingMessage) {
        int max;
        int n = max = this.maxDeferredProcessedApplicationDataMessages == 0 ? 1 : this.maxDeferredProcessedApplicationDataMessages;
        if (this.deferredApplicationData.size() < max) {
            this.deferredApplicationData.add(outgoingMessage);
        }
    }

    public void addApplicationDataForDeferredProcessing(Record incomingMessage) {
        if (this.deferredApplicationData.size() < this.maxDeferredProcessedApplicationDataMessages) {
            this.deferredApplicationData.add(incomingMessage);
        }
    }

    public List<Object> takeDeferredApplicationData() {
        ArrayList<Object> applicationData = new ArrayList<Object>(this.deferredApplicationData);
        this.deferredApplicationData.clear();
        return applicationData;
    }

    public void takeDeferredApplicationData(Handshaker replacedHandshaker) {
        this.deferredApplicationData.addAll(replacedHandshaker.takeDeferredApplicationData());
    }

    public void setPendingFlight(DTLSFlight pendingFlight) {
        DTLSFlight flight = this.pendingFlight.getAndSet(pendingFlight);
        if (flight != null && flight != pendingFlight) {
            flight.setResponseCompleted();
        }
    }

    public void cancelPendingFlight() {
        this.setPendingFlight(null);
    }

    public void sendFlight(DTLSFlight flight) {
        this.setPendingFlight(null);
        try {
            this.recordLayer.sendFlight(flight);
            this.setPendingFlight(flight);
        }
        catch (IOException e) {
            this.handshakeFailed(new Exception("handshake flight " + flight.getFlightNumber() + " failed!", e));
        }
    }

    public final void addSessionListener(SessionListener listener) {
        if (listener != null) {
            this.sessionListeners.add(listener);
        }
    }

    public final void removeSessionListener(SessionListener listener) {
        if (listener != null) {
            this.sessionListeners.remove(listener);
        }
    }

    protected final void handshakeStarted() throws HandshakeException {
        for (SessionListener sessionListener : this.sessionListeners) {
            sessionListener.handshakeStarted(this);
        }
    }

    protected final void sessionEstablished() throws HandshakeException {
        this.sessionEstablished = true;
        for (SessionListener sessionListener : this.sessionListeners) {
            sessionListener.sessionEstablished(this, this.getSession());
        }
    }

    public final void handshakeCompleted() {
        this.cancelPendingFlight();
        for (SessionListener sessionListener : this.sessionListeners) {
            sessionListener.handshakeCompleted(this);
        }
    }

    public final void handshakeFailed(Throwable cause) {
        this.cancelPendingFlight();
        if (!this.sessionEstablished) {
            for (SessionListener sessionListener : this.sessionListeners) {
                sessionListener.handshakeFailed(this, cause);
            }
            for (Object message : this.takeDeferredApplicationData()) {
                if (!(message instanceof RawData)) continue;
                ((RawData)message).onError(cause);
            }
        }
    }

    public final void handshakeFlightRetransmitted(int flight) {
        for (SessionListener sessionListener : this.sessionListeners) {
            sessionListener.handshakeFlightRetransmitted(this, flight);
        }
        for (Object message : this.deferredApplicationData) {
            if (!(message instanceof RawData)) continue;
            ((RawData)message).onDtlsRetransmission(flight);
        }
    }

    public final boolean hasBeenStartedByMessage(HandshakeMessage handshakeMessage) {
        return this.isFirstMessageReceived(handshakeMessage);
    }

    protected boolean isFirstMessageReceived(HandshakeMessage handshakeMessage) {
        return false;
    }

    public final boolean isChangeCipherSpecMessageExpected() {
        return this.changeCipherSuiteMessageExpected;
    }

    protected final void expectChangeCipherSpecMessage() {
        this.changeCipherSuiteMessageExpected = true;
    }

    /*
     * Enabled aggressive block sorting
     */
    public void verifyCertificate(CertificateMessage message) throws HandshakeException {
        if (message.getCertificateChain() == null) {
            RawPublicKeyIdentity rpk = new RawPublicKeyIdentity(message.getPublicKey());
            if (this.rpkStore.isTrusted(rpk)) return;
            LOGGER.debug("Certificate validation failed: Raw public key is not trusted");
            AlertMessage alert = new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.BAD_CERTIFICATE, this.session.getPeer());
            throw new HandshakeException("Raw public key is not trusted", alert);
        }
        if (this.certificateVerifier != null) {
            this.certificateVerifier.verifyCertificate(message, this.session);
            return;
        }
        LOGGER.debug("Certificate validation failed: x509 could not be trusted!");
        AlertMessage alert = new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.UNEXPECTED_MESSAGE, this.session.getPeer());
        throw new HandshakeException("Trust is not possible!", alert);
    }

    class InboundMessageBuffer {
        private ChangeCipherSpecMessage changeCipherSpec = null;
        private SortedSet<Record> queue = new TreeSet<Record>(new Comparator<Record>(){

            @Override
            public int compare(Record r1, Record r2) {
                if (r1.getEpoch() < r2.getEpoch()) {
                    return -1;
                }
                if (r1.getEpoch() > r2.getEpoch()) {
                    return 1;
                }
                if (r1.getSequenceNumber() < r2.getSequenceNumber()) {
                    return -1;
                }
                if (r1.getSequenceNumber() > r2.getSequenceNumber()) {
                    return 1;
                }
                return 0;
            }
        });

        InboundMessageBuffer() {
        }

        boolean isEmpty() {
            return this.queue.isEmpty();
        }

        DTLSMessage getNextMessage() throws GeneralSecurityException, HandshakeException {
            AbstractMessage result = null;
            if (Handshaker.this.isChangeCipherSpecMessageExpected() && this.changeCipherSpec != null) {
                result = this.changeCipherSpec;
                this.changeCipherSpec = null;
            } else {
                for (Record record : this.queue) {
                    HandshakeMessage msg;
                    if (record.getEpoch() != Handshaker.this.session.getReadEpoch() || (msg = (HandshakeMessage)record.getFragment(Handshaker.this.session.getReadState())).getMessageSeq() != Handshaker.this.nextReceiveSeq) continue;
                    result = msg;
                    this.queue.remove(record);
                    break;
                }
            }
            return result;
        }

        DTLSMessage getNextMessage(Record candidate) throws GeneralSecurityException, HandshakeException {
            int epoch = candidate.getEpoch();
            if (epoch < Handshaker.this.session.getReadEpoch()) {
                LOGGER.debug("Discarding message from peer [{}] from past epoch [{}] < current epoch [{}]", new Object[]{Handshaker.this.getPeerAddress(), epoch, Handshaker.this.session.getReadEpoch()});
                return null;
            }
            if (epoch == Handshaker.this.session.getReadEpoch()) {
                DTLSMessage fragment = candidate.getFragment();
                switch (fragment.getContentType()) {
                    case ALERT: {
                        return fragment;
                    }
                    case CHANGE_CIPHER_SPEC: {
                        if (Handshaker.this.isChangeCipherSpecMessageExpected()) {
                            return fragment;
                        }
                        LOGGER.debug("Change Cipher Spec is not expected and therefore kept for later processing!");
                        this.changeCipherSpec = (ChangeCipherSpecMessage)fragment;
                        return null;
                    }
                    case HANDSHAKE: {
                        HandshakeMessage handshakeMessage = (HandshakeMessage)fragment;
                        int messageSeq = handshakeMessage.getMessageSeq();
                        if (messageSeq == Handshaker.this.nextReceiveSeq) {
                            return handshakeMessage;
                        }
                        if (messageSeq > Handshaker.this.nextReceiveSeq) {
                            LOGGER.debug("Queued newer message from current epoch, message_seq [{}] > next_receive_seq [{}]", (Object)messageSeq, (Object)Handshaker.this.nextReceiveSeq);
                            this.queue.add(candidate);
                            return null;
                        }
                        LOGGER.debug("Discarding old message, message_seq [{}] < next_receive_seq [{}]", (Object)messageSeq, (Object)Handshaker.this.nextReceiveSeq);
                        return null;
                    }
                }
                LOGGER.debug("Cannot process message of type [{}], discarding...", (Object)fragment.getContentType());
                return null;
            }
            this.queue.add(candidate);
            LOGGER.debug("Queueing HANDSHAKE message from future epoch [{}] > current epoch [{}]", (Object)epoch, (Object)Handshaker.this.getSession().getReadEpoch());
            return null;
        }
    }
}

