/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.net.InetSocketAddress;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.security.auth.x500.X500Principal;
import org.eclipse.californium.elements.util.DatagramReader;
import org.eclipse.californium.elements.util.DatagramWriter;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.eclipse.californium.scandium.dtls.SignatureAndHashAlgorithm;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CertificateRequest
extends HandshakeMessage {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)CertificateRequest.class.getName());
    private static final String THREE_TABS = "\t\t\t";
    private static final int CERTIFICATE_TYPES_LENGTH_BITS = 8;
    private static final int CERTIFICATE_TYPE_BITS = 8;
    private static final int SUPPORTED_SIGNATURE_LENGTH_BITS = 16;
    private static final int CERTIFICATE_AUTHORITIES_LENGTH_BITS = 16;
    private static final int CERTIFICATE_AUTHORITY_LENGTH_BITS = 16;
    private static final int SUPPORTED_SIGNATURE_BITS = 8;
    private static final int MAX_LENGTH_CERTIFICATE_AUTHORITIES = 65535;
    private final List<ClientCertificateType> certificateTypes = new ArrayList<ClientCertificateType>();
    private final List<SignatureAndHashAlgorithm> supportedSignatureAlgorithms = new ArrayList<SignatureAndHashAlgorithm>();
    private final List<X500Principal> certificateAuthorities = new ArrayList<X500Principal>();
    private int certificateAuthoritiesEncodedLength = 0;

    public CertificateRequest(InetSocketAddress peerAddress) {
        super(peerAddress);
    }

    public CertificateRequest(List<ClientCertificateType> certificateTypes, List<SignatureAndHashAlgorithm> supportedSignatureAlgorithms, List<X500Principal> certificateAuthorities, InetSocketAddress peerAddress) {
        super(peerAddress);
        if (certificateTypes != null) {
            this.certificateTypes.addAll(certificateTypes);
        }
        if (supportedSignatureAlgorithms != null) {
            this.supportedSignatureAlgorithms.addAll(supportedSignatureAlgorithms);
        }
        if (certificateAuthorities != null) {
            this.addCerticiateAuthorities(certificateAuthorities);
        }
    }

    @Override
    public HandshakeType getMessageType() {
        return HandshakeType.CERTIFICATE_REQUEST;
    }

    @Override
    public int getMessageLength() {
        return 1 + this.certificateTypes.size() + 2 + this.supportedSignatureAlgorithms.size() * 2 + 2 + this.certificateAuthoritiesEncodedLength;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        if (!this.certificateTypes.isEmpty()) {
            sb.append("\t\tClient certificate type:").append(StringUtil.lineSeparator());
            for (ClientCertificateType type : this.certificateTypes) {
                sb.append(THREE_TABS).append((Object)type).append(StringUtil.lineSeparator());
            }
        }
        if (!this.supportedSignatureAlgorithms.isEmpty()) {
            sb.append("\t\tSignature and hash algorithm:").append(StringUtil.lineSeparator());
            for (SignatureAndHashAlgorithm algo : this.supportedSignatureAlgorithms) {
                sb.append(THREE_TABS).append(algo.jcaName()).append(StringUtil.lineSeparator());
            }
        }
        if (!this.certificateAuthorities.isEmpty()) {
            sb.append("\t\tCertificate authorities:").append(StringUtil.lineSeparator());
            for (X500Principal subject : this.certificateAuthorities) {
                sb.append(THREE_TABS).append(subject.getName()).append(StringUtil.lineSeparator());
            }
        }
        return sb.toString();
    }

    @Override
    public byte[] fragmentToByteArray() {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.certificateTypes.size(), 8);
        for (ClientCertificateType certificateType : this.certificateTypes) {
            writer.write(certificateType.getCode(), 8);
        }
        writer.write(this.supportedSignatureAlgorithms.size() * 2, 16);
        for (SignatureAndHashAlgorithm signatureAndHashAlgorithm : this.supportedSignatureAlgorithms) {
            writer.write(signatureAndHashAlgorithm.getHash().getCode(), 8);
            writer.write(signatureAndHashAlgorithm.getSignature().getCode(), 8);
        }
        writer.write(this.certificateAuthoritiesEncodedLength, 16);
        for (X500Principal distinguishedName : this.certificateAuthorities) {
            byte[] encoded = distinguishedName.getEncoded();
            writer.write(encoded.length, 16);
            writer.writeBytes(encoded);
        }
        return writer.toByteArray();
    }

    public static HandshakeMessage fromByteArray(byte[] byteArray, InetSocketAddress peerAddress) {
        byte[] name;
        DatagramReader reader = new DatagramReader(byteArray);
        int length = reader.read(8);
        ArrayList<ClientCertificateType> certificateTypes = new ArrayList<ClientCertificateType>();
        for (int i = 0; i < length; ++i) {
            int code = reader.read(8);
            certificateTypes.add(ClientCertificateType.getTypeByCode(code));
        }
        length = reader.read(16);
        ArrayList<SignatureAndHashAlgorithm> supportedSignatureAlgorithms = new ArrayList<SignatureAndHashAlgorithm>();
        for (int i = 0; i < length; i += 2) {
            int codeHash = reader.read(8);
            int codeSignature = reader.read(8);
            supportedSignatureAlgorithms.add(new SignatureAndHashAlgorithm(SignatureAndHashAlgorithm.HashAlgorithm.getAlgorithmByCode(codeHash), SignatureAndHashAlgorithm.SignatureAlgorithm.getAlgorithmByCode(codeSignature)));
        }
        ArrayList<X500Principal> certificateAuthorities = new ArrayList<X500Principal>();
        for (length = reader.read(16); length > 0; length -= 2 + name.length) {
            int nameLength = reader.read(16);
            name = reader.readBytes(nameLength);
            certificateAuthorities.add(new X500Principal(name));
        }
        return new CertificateRequest(certificateTypes, supportedSignatureAlgorithms, certificateAuthorities, peerAddress);
    }

    public void addCertificateType(ClientCertificateType certificateType) {
        this.certificateTypes.add(certificateType);
    }

    public void addSignatureAlgorithm(SignatureAndHashAlgorithm signatureAndHashAlgorithm) {
        this.supportedSignatureAlgorithms.add(signatureAndHashAlgorithm);
    }

    public boolean addCertificateAuthority(X500Principal authority) {
        if (authority == null) {
            throw new NullPointerException("authority must not be null");
        }
        int encodedAuthorityLength = 2 + authority.getEncoded().length;
        if (this.certificateAuthoritiesEncodedLength + encodedAuthorityLength <= 65535) {
            this.certificateAuthorities.add(authority);
            this.certificateAuthoritiesEncodedLength += encodedAuthorityLength;
            return true;
        }
        return false;
    }

    private boolean addCerticiateAuthorities(List<X500Principal> authorities) {
        int authoritiesAdded = 0;
        for (X500Principal authority : authorities) {
            if (!this.addCertificateAuthority(authority)) {
                LOGGER.debug("could add only {} of {} certificate authorities, max length exceeded", new Object[]{authoritiesAdded, authorities.size()});
                return false;
            }
            ++authoritiesAdded;
        }
        return true;
    }

    public boolean addCertificateAuthorities(X509Certificate[] trustedCas) {
        if (trustedCas != null) {
            int authoritiesAdded = 0;
            for (X509Certificate certificate : trustedCas) {
                if (!this.addCertificateAuthority(certificate.getSubjectX500Principal())) {
                    LOGGER.debug("could add only {} of {} certificate authorities, max length exceeded", new Object[]{authoritiesAdded, trustedCas.length});
                    return false;
                }
                ++authoritiesAdded;
            }
        }
        return true;
    }

    public List<ClientCertificateType> getCertificateTypes() {
        return Collections.unmodifiableList(this.certificateTypes);
    }

    boolean isSupportedKeyType(PublicKey key) {
        for (ClientCertificateType type : this.certificateTypes) {
            if (!type.isCompatibleWithKeyAlgorithm(key.getAlgorithm())) continue;
            return true;
        }
        return false;
    }

    boolean isSupportedKeyType(X509Certificate cert) {
        for (ClientCertificateType type : this.certificateTypes) {
            boolean isCompatibleType = type.isCompatibleWithKeyAlgorithm(cert.getPublicKey().getAlgorithm());
            boolean meetsSigningRequirements = !type.requiresSigningCapability() || cert.getKeyUsage() == null || cert.getKeyUsage()[0];
            LOGGER.debug("type: {}, isCompatibleWithKeyAlgorithm[{}]: {}, meetsSigningRequirements: {}", new Object[]{type, cert.getPublicKey().getAlgorithm(), isCompatibleType, meetsSigningRequirements});
            if (!isCompatibleType || !meetsSigningRequirements) continue;
            return true;
        }
        LOGGER.debug("certificate [{}] is not of any supported type", (Object)cert);
        return false;
    }

    public SignatureAndHashAlgorithm getSignatureAndHashAlgorithm(PublicKey key) {
        if (this.isSupportedKeyType(key)) {
            return this.getSupportedSignatureAlgorithm(key);
        }
        return null;
    }

    public SignatureAndHashAlgorithm getSignatureAndHashAlgorithm(List<X509Certificate> chain) {
        X509Certificate x509Certificate;
        if (this.isSignedWithSupportedAlgorithm(chain) && this.isSupportedKeyType(x509Certificate = chain.get(0))) {
            return this.getSupportedSignatureAlgorithm(x509Certificate.getPublicKey());
        }
        return null;
    }

    boolean isSignedWithSupportedAlgorithm(List<X509Certificate> chain) {
        for (X509Certificate certificate : chain) {
            if (this.isSignedWithSupportedAlgorithm(certificate)) continue;
            LOGGER.debug("certificate chain is NOT signed with supported algorithm(s)");
            return false;
        }
        LOGGER.debug("certificate chain is signed with supported algorithm(s)");
        return true;
    }

    boolean isSignedWithSupportedAlgorithm(X509Certificate certificate) {
        String sigAlgName = certificate.getSigAlgName();
        for (SignatureAndHashAlgorithm supportedAlgorithm : this.supportedSignatureAlgorithms) {
            if (!supportedAlgorithm.jcaName().equalsIgnoreCase(sigAlgName)) continue;
            return true;
        }
        LOGGER.debug("certificate is NOT signed with supported algorithm(s)");
        return false;
    }

    SignatureAndHashAlgorithm getSupportedSignatureAlgorithm(PublicKey key) {
        for (SignatureAndHashAlgorithm supportedAlgorithm : this.supportedSignatureAlgorithms) {
            try {
                Signature sign = Signature.getInstance(supportedAlgorithm.jcaName());
                sign.initVerify(key);
                return supportedAlgorithm;
            }
            catch (InvalidKeyException | NoSuchAlgorithmException generalSecurityException) {
            }
        }
        return null;
    }

    public List<X509Certificate> removeTrustedCertificates(List<X509Certificate> chain) {
        if (chain == null) {
            throw new NullPointerException("certificate chain must not be null");
        }
        if (chain.size() > 1) {
            ArrayList<X509Certificate> result = new ArrayList<X509Certificate>();
            result.add(chain.get(0));
            for (int i = 1; i < chain.size(); ++i) {
                X509Certificate x509Certificate = chain.get(i);
                result.add(x509Certificate);
                if (this.certificateAuthorities.contains(x509Certificate.getSubjectX500Principal())) break;
            }
            return Collections.unmodifiableList(result);
        }
        return chain;
    }

    public List<SignatureAndHashAlgorithm> getSupportedSignatureAlgorithms() {
        return Collections.unmodifiableList(this.supportedSignatureAlgorithms);
    }

    public List<X500Principal> getCertificateAuthorities() {
        return Collections.unmodifiableList(this.certificateAuthorities);
    }

    public static enum ClientCertificateType {
        RSA_SIGN(1, "RSA", true),
        DSS_SIGN(2, "DSA", true),
        RSA_FIXED_DH(3, "DH", false),
        DSS_FIXED_DH(4, "DH", false),
        RSA_EPHEMERAL_DH_RESERVED(5, "DH", false),
        DSS_EPHEMERAL_DH_RESERVED(6, "DH", false),
        FORTEZZA_DMS_RESERVED(20, "UNKNOWN", false),
        ECDSA_SIGN(64, "EC", true),
        RSA_FIXED_ECDH(65, "DH", false),
        ECDSA_FIXED_ECDH(66, "DH", false);

        private final int code;
        private final String jcaAlgorithm;
        private final boolean requiresSigningCapability;

        private ClientCertificateType(int code, String algorithm, boolean requiresSigningCapability) {
            this.code = code;
            this.jcaAlgorithm = algorithm;
            this.requiresSigningCapability = requiresSigningCapability;
        }

        public int getCode() {
            return this.code;
        }

        public String getJcaAlgorithm() {
            return this.jcaAlgorithm;
        }

        public boolean requiresSigningCapability() {
            return this.requiresSigningCapability;
        }

        public boolean isCompatibleWithKeyAlgorithm(String algorithm) {
            return this.jcaAlgorithm.equals(algorithm);
        }

        public static ClientCertificateType getTypeByCode(int code) {
            for (ClientCertificateType type : ClientCertificateType.values()) {
                if (type.code != code) continue;
                return type;
            }
            return null;
        }
    }
}

