/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.nio.channels.ClosedByInterruptException;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.DtlsEndpointContext;
import org.eclipse.californium.elements.EndpointContext;
import org.eclipse.californium.elements.EndpointContextMatcher;
import org.eclipse.californium.elements.EndpointMismatchException;
import org.eclipse.californium.elements.EndpointUnconnectedException;
import org.eclipse.californium.elements.MulticastNotSupportedException;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.RawDataChannel;
import org.eclipse.californium.elements.util.DaemonThreadFactory;
import org.eclipse.californium.elements.util.ExecutorsUtil;
import org.eclipse.californium.elements.util.LeastRecentlyUsedCache;
import org.eclipse.californium.elements.util.NamedThreadFactory;
import org.eclipse.californium.elements.util.SerialExecutor;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.AlertHandler;
import org.eclipse.californium.scandium.CookieGenerator;
import org.eclipse.californium.scandium.config.DtlsConnectorConfig;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.ApplicationMessage;
import org.eclipse.californium.scandium.dtls.ClientHandshaker;
import org.eclipse.californium.scandium.dtls.ClientHello;
import org.eclipse.californium.scandium.dtls.Connection;
import org.eclipse.californium.scandium.dtls.ContentType;
import org.eclipse.californium.scandium.dtls.DTLSFlight;
import org.eclipse.californium.scandium.dtls.DTLSMessage;
import org.eclipse.californium.scandium.dtls.DTLSSession;
import org.eclipse.californium.scandium.dtls.DtlsHandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.Handshaker;
import org.eclipse.californium.scandium.dtls.HelloVerifyRequest;
import org.eclipse.californium.scandium.dtls.InMemoryConnectionStore;
import org.eclipse.californium.scandium.dtls.MaxFragmentLengthExtension;
import org.eclipse.californium.scandium.dtls.ProtocolVersion;
import org.eclipse.californium.scandium.dtls.Record;
import org.eclipse.californium.scandium.dtls.RecordLayer;
import org.eclipse.californium.scandium.dtls.ResumingClientHandshaker;
import org.eclipse.californium.scandium.dtls.ResumingServerHandshaker;
import org.eclipse.californium.scandium.dtls.ResumptionSupportingConnectionStore;
import org.eclipse.californium.scandium.dtls.ServerHandshaker;
import org.eclipse.californium.scandium.dtls.ServerNameExtension;
import org.eclipse.californium.scandium.dtls.SessionAdapter;
import org.eclipse.californium.scandium.dtls.SessionCache;
import org.eclipse.californium.scandium.dtls.SessionId;
import org.eclipse.californium.scandium.dtls.SessionListener;
import org.eclipse.californium.scandium.dtls.SessionTicket;
import org.eclipse.californium.scandium.dtls.cipher.CipherSuite;
import org.eclipse.californium.scandium.util.ByteArrayUtils;
import org.eclipse.californium.scandium.util.ServerNames;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DTLSConnector
implements Connector,
RecordLayer {
    public static final String KEY_TLS_SERVER_HOST_NAME = "TLS_SERVER_HOST_NAME";
    public static final int MAX_MTU = 65535;
    public static final int DEFAULT_IPV6_MTU = 1280;
    public static final int DEFAULT_IPV4_MTU = 576;
    private static final Logger LOGGER = LoggerFactory.getLogger((String)DTLSConnector.class.getCanonicalName());
    private static final int MAX_PLAINTEXT_FRAGMENT_LENGTH = 16384;
    private static final int MAX_CIPHERTEXT_EXPANSION = CipherSuite.TLS_PSK_WITH_AES_128_CBC_SHA256.getMaxCiphertextExpansion();
    private static final int MAX_DATAGRAM_BUFFER_SIZE = 16409 + MAX_CIPHERTEXT_EXPANSION;
    private final DtlsConnectorConfig config;
    private final LeastRecentlyUsedCache<InetSocketAddress, SerialExecutor> connectionExecutors;
    private final ResumptionSupportingConnectionStore connectionStore;
    private final Long autoResumptionTimeoutMillis;
    private final int thresholdHandshakesWithoutVerifiedPeer;
    private final AtomicInteger pendingHandshakesWithoutVerifiedPeer = new AtomicInteger();
    private final boolean serverOnly;
    private final AtomicInteger pendingOutboundMessagesCountdown = new AtomicInteger();
    private final List<Thread> receiverThreads = new LinkedList<Thread>();
    private InetSocketAddress lastBindAddress;
    private int maximumTransmissionUnit = 576;
    private int inboundDatagramBufferSize = MAX_DATAGRAM_BUFFER_SIZE;
    private CookieGenerator cookieGenerator = new CookieGenerator();
    private Object alertHandlerLock = new Object();
    private volatile DatagramSocket socket;
    private ScheduledExecutorService timer;
    private AtomicBoolean running = new AtomicBoolean(false);
    private EndpointContextMatcher endpointContextMatcher;
    private RawDataChannel messageHandler;
    private AlertHandler alertHandler;
    private SessionListener sessionListener;
    private ExecutorService executorService;
    private boolean hasInternalExecutor;

    public DTLSConnector(DtlsConnectorConfig configuration) {
        this(configuration, (SessionCache)null);
    }

    public DTLSConnector(DtlsConnectorConfig configuration, SessionCache sessionCache) {
        this(configuration, new InMemoryConnectionStore(configuration.getMaxConnections(), configuration.getStaleConnectionThreshold(), sessionCache));
    }

    protected DTLSConnector(DtlsConnectorConfig configuration, final ResumptionSupportingConnectionStore connectionStore) {
        if (configuration == null) {
            throw new NullPointerException("Configuration must not be null");
        }
        if (connectionStore == null) {
            throw new NullPointerException("Connection store must not be null");
        }
        this.config = configuration;
        this.pendingOutboundMessagesCountdown.set(this.config.getOutboundMessageBufferSize());
        this.autoResumptionTimeoutMillis = this.config.getAutoResumptionTimeoutMillis();
        this.serverOnly = this.config.isServerOnly();
        this.connectionStore = connectionStore;
        this.sessionListener = new SessionAdapter(){

            @Override
            public void sessionEstablished(Handshaker handshaker, DTLSSession establishedSession) throws HandshakeException {
                DTLSConnector.this.sessionEstablished(handshaker, establishedSession);
            }

            @Override
            public void handshakeFailed(Handshaker handshaker, Throwable error) {
                connectionStore.remove(handshaker.getPeerAddress(), false);
            }
        };
        int maxConnections = configuration.getMaxConnections();
        long thresholdInPercent = this.config.getVerifyPeersOnResumptionThreshold().intValue();
        long threshold = ((long)maxConnections * thresholdInPercent + 50L) / 100L;
        if (threshold == 0L && thresholdInPercent > 0L) {
            threshold = 1L;
        }
        this.thresholdHandshakesWithoutVerifiedPeer = (int)threshold;
        maxConnections = maxConnections < 80 ? 100 : (maxConnections += maxConnections / 4);
        this.connectionExecutors = new LeastRecentlyUsedCache(maxConnections, 20L);
    }

    private final void sessionEstablished(Handshaker handshaker, final DTLSSession establishedSession) throws HandshakeException {
        SerialExecutor serialExecutor;
        final Connection connection = this.connectionStore.get(handshaker.getPeerAddress());
        this.connectionStore.putEstablishedSession(establishedSession, connection);
        List<Object> list = handshaker.takeDeferredApplicationData();
        if (!list.isEmpty() && (serialExecutor = this.getSerialExecutor(handshaker.getPeerAddress())) != null) {
            LOGGER.debug("Session with [{}] established, now process deferred {} messages", (Object)establishedSession.getPeer(), (Object)list.size());
            for (Object message : list) {
                if (message instanceof RawData) {
                    final RawData rawData = (RawData)message;
                    serialExecutor.execute(new Runnable(){

                        @Override
                        public void run() {
                            DTLSConnector.this.sendMessage(rawData, connection, establishedSession);
                        }
                    });
                    continue;
                }
                if (!(message instanceof Record)) continue;
                final Record record = (Record)message;
                serialExecutor.execute(new Runnable(){

                    @Override
                    public void run() {
                        DTLSConnector.this.processApplicationDataRecord(record, connection);
                    }
                });
            }
        }
    }

    protected void onInitializeHandshaker(Handshaker handshaker) {
    }

    private final void initializeHandshaker(Handshaker handshaker) {
        if (this.sessionListener != null) {
            handshaker.addSessionListener(this.sessionListener);
        }
        this.onInitializeHandshaker(handshaker);
    }

    public final synchronized void setExecutor(ExecutorService executor) {
        if (this.executorService != executor) {
            if (this.running.get()) {
                throw new IllegalStateException("cannot set new executor while connector is running");
            }
            this.executorService = executor;
        }
    }

    public final void close(InetSocketAddress peerAddress) {
        final Connection connection = this.connectionStore.get(peerAddress);
        if (connection != null && connection.hasEstablishedSession()) {
            SerialExecutor serialExecutor = this.getSerialExecutor(peerAddress);
            if (serialExecutor != null) {
                serialExecutor.execute(new Runnable(){

                    @Override
                    public void run() {
                        DTLSConnector.this.terminateConnection(connection, new AlertMessage(AlertMessage.AlertLevel.WARNING, AlertMessage.AlertDescription.CLOSE_NOTIFY, connection.getPeerAddress()), connection.getEstablishedSession());
                    }
                });
            } else {
                throw new IllegalStateException("executor cache exceeded!");
            }
        }
    }

    public final synchronized void start() throws IOException {
        this.start(this.config.getAddress());
    }

    final synchronized void restart() throws IOException {
        if (this.lastBindAddress == null) {
            throw new IllegalStateException("Connector has never been started before");
        }
        this.start(this.lastBindAddress);
    }

    private synchronized Executor getExecutorService() {
        return this.executorService;
    }

    private void start(InetSocketAddress bindAddress) throws IOException {
        if (this.running.get()) {
            return;
        }
        this.pendingOutboundMessagesCountdown.set(this.config.getOutboundMessageBufferSize());
        this.timer = this.executorService instanceof ScheduledExecutorService ? (ScheduledExecutorService)this.executorService : ExecutorsUtil.newSingleThreadScheduledExecutor((ThreadFactory)new DaemonThreadFactory("DTLS-Retransmit-Task-", NamedThreadFactory.SCANDIUM_THREAD_GROUP));
        if (this.executorService == null) {
            int threadCount = this.config.getConnectionThreadCount();
            this.executorService = threadCount > 1 ? ExecutorsUtil.newFixedThreadPool((int)(threadCount - 1), (ThreadFactory)new DaemonThreadFactory("DTLS-Connection-Handler-", NamedThreadFactory.SCANDIUM_THREAD_GROUP)) : this.timer;
            this.hasInternalExecutor = true;
        }
        this.socket = new DatagramSocket(null);
        if (bindAddress.getPort() != 0 && this.config.isAddressReuseEnabled().booleanValue()) {
            LOGGER.info("Enable address reuse for socket!");
            this.socket.setReuseAddress(true);
            if (!this.socket.getReuseAddress()) {
                LOGGER.warn("Enable address reuse for socket failed!");
            }
        }
        this.socket.bind(bindAddress);
        if (!(this.lastBindAddress == null || this.socket.getLocalAddress().equals(this.lastBindAddress.getAddress()) && this.socket.getLocalPort() == this.lastBindAddress.getPort())) {
            if (this.connectionStore instanceof ResumptionSupportingConnectionStore) {
                this.connectionStore.markAllAsResumptionRequired();
            } else {
                this.connectionStore.clear();
            }
        }
        if (this.config.getMaxTransmissionUnit() == null) {
            InetAddress localInterfaceAddress = bindAddress.getAddress();
            if (localInterfaceAddress.isAnyLocalAddress()) {
                int mtu = 65535;
                Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
                while (interfaces.hasMoreElements()) {
                    NetworkInterface iface = interfaces.nextElement();
                    int ifaceMtu = iface.getMTU();
                    if (ifaceMtu <= 0 || ifaceMtu >= mtu) continue;
                    mtu = ifaceMtu;
                }
                LOGGER.info("multiple network interfaces, using smallest MTU [{}]", (Object)mtu);
                this.maximumTransmissionUnit = mtu;
            } else {
                NetworkInterface ni = NetworkInterface.getByInetAddress(localInterfaceAddress);
                if (ni != null && ni.getMTU() > 0) {
                    this.maximumTransmissionUnit = ni.getMTU();
                } else if (localInterfaceAddress instanceof Inet4Address) {
                    LOGGER.info("Cannot determine MTU of network interface, using minimum MTU [{}] of IPv4 instead", (Object)576);
                    this.maximumTransmissionUnit = 576;
                } else {
                    LOGGER.info("Cannot determine MTU of network interface, using minimum MTU [{}] of IPv6 instead", (Object)1280);
                    this.maximumTransmissionUnit = 1280;
                }
            }
        } else {
            this.maximumTransmissionUnit = this.config.getMaxTransmissionUnit();
        }
        if (this.config.getMaxFragmentLengthCode() != null) {
            MaxFragmentLengthExtension.Length lengthCode = MaxFragmentLengthExtension.Length.fromCode(this.config.getMaxFragmentLengthCode());
            this.inboundDatagramBufferSize = lengthCode.length() + MAX_CIPHERTEXT_EXPANSION + 25;
        }
        this.lastBindAddress = new InetSocketAddress(this.socket.getLocalAddress(), this.socket.getLocalPort());
        this.running.set(true);
        int receiverThreadCount = this.config.getReceiverThreadCount();
        for (int i = 0; i < receiverThreadCount; ++i) {
            Worker receiver = new Worker("DTLS-Receiver-" + i + "-" + this.lastBindAddress){
                private final byte[] receiverBuffer;
                private final DatagramPacket packet;
                {
                    this.receiverBuffer = new byte[DTLSConnector.this.inboundDatagramBufferSize];
                    this.packet = new DatagramPacket(this.receiverBuffer, DTLSConnector.this.inboundDatagramBufferSize);
                }

                @Override
                public void doWork() throws Exception {
                    this.packet.setData(this.receiverBuffer);
                    DTLSConnector.this.receiveNextDatagramFromNetwork(this.packet);
                }
            };
            receiver.setDaemon(true);
            receiver.start();
            this.receiverThreads.add(receiver);
        }
        LOGGER.info("DTLS connector listening on [{}] with MTU [{}] using (inbound) datagram buffer size [{} bytes]", new Object[]{this.lastBindAddress, this.maximumTransmissionUnit, this.inboundDatagramBufferSize});
    }

    public final synchronized void forceResumeSessionFor(InetSocketAddress peer) {
        Connection peerConnection = this.connectionStore.get(peer);
        if (peerConnection != null && peerConnection.hasEstablishedSession()) {
            peerConnection.setResumptionRequired(true);
        }
    }

    public final synchronized void forceResumeAllSessions() {
        this.connectionStore.markAllAsResumptionRequired();
    }

    public final synchronized void clearConnectionState() {
        this.connectionStore.clear();
    }

    private final synchronized DatagramSocket getSocket() {
        return this.socket;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void stop() {
        ExecutorService shutdownTimer = null;
        ExecutorService shutdown = null;
        ArrayList<Runnable> pending = new ArrayList<Runnable>();
        DTLSConnector dTLSConnector = this;
        synchronized (dTLSConnector) {
            if (this.running.compareAndSet(true, false)) {
                LOGGER.info("Stopping DTLS connector on [{}]", (Object)this.lastBindAddress);
                for (Thread t : this.receiverThreads) {
                    t.interrupt();
                }
                this.receiverThreads.clear();
                if (this.socket != null) {
                    this.socket.close();
                    this.socket = null;
                }
                this.maximumTransmissionUnit = 0;
                Iterator<Thread> i$ = this.connectionExecutors;
                synchronized (i$) {
                    for (SerialExecutor executors : this.connectionExecutors.values()) {
                        executors.shutdownNow(pending);
                    }
                    this.connectionExecutors.clear();
                }
                if (this.executorService != this.timer) {
                    pending.addAll(this.timer.shutdownNow());
                    shutdownTimer = this.timer;
                    this.timer = null;
                }
                if (this.hasInternalExecutor) {
                    pending.addAll(this.executorService.shutdownNow());
                    shutdown = this.executorService;
                    this.executorService = null;
                    this.hasInternalExecutor = false;
                }
            }
        }
        if (shutdownTimer != null) {
            try {
                if (!shutdownTimer.awaitTermination(500L, TimeUnit.MILLISECONDS)) {
                    LOGGER.warn("Shutdown DTLS connector on [{}] timer not terminated in time!", (Object)this.lastBindAddress);
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        if (shutdown != null) {
            try {
                if (!shutdown.awaitTermination(500L, TimeUnit.MILLISECONDS)) {
                    LOGGER.warn("Shutdown DTLS connector on [{}] executor not terminated in time!", (Object)this.lastBindAddress);
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        for (Runnable job : pending) {
            job.run();
        }
    }

    public final synchronized void destroy() {
        this.stop();
        this.connectionStore.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private final SerialExecutor getSerialExecutor(InetSocketAddress peerAddress) {
        Executor executor = this.getExecutorService();
        LeastRecentlyUsedCache<InetSocketAddress, SerialExecutor> leastRecentlyUsedCache = this.connectionExecutors;
        synchronized (leastRecentlyUsedCache) {
            SerialExecutor serialExecutor = (SerialExecutor)this.connectionExecutors.get((Object)peerAddress);
            if (serialExecutor == null) {
                serialExecutor = new SerialExecutor(executor);
                if (this.running.get() && !this.connectionExecutors.put((Object)peerAddress, (Object)serialExecutor)) {
                    serialExecutor = null;
                }
            }
            return serialExecutor;
        }
    }

    private void receiveNextDatagramFromNetwork(DatagramPacket packet) throws IOException {
        DatagramSocket currentSocket = this.getSocket();
        if (currentSocket == null) {
            return;
        }
        currentSocket.receive(packet);
        if (packet.getLength() == 0) {
            return;
        }
        InetSocketAddress peerAddress = new InetSocketAddress(packet.getAddress(), packet.getPort());
        byte[] data = Arrays.copyOfRange(packet.getData(), packet.getOffset(), packet.getLength());
        List<Record> records = Record.fromByteArray(data, peerAddress);
        LOGGER.debug("Received {} DTLS records using a {} byte datagram buffer", (Object)records.size(), (Object)this.inboundDatagramBufferSize);
        if (records.isEmpty()) {
            return;
        }
        if (!this.running.get()) {
            LOGGER.debug("Execution shutdown while processing incoming records from peer: {}", (Object)peerAddress);
            return;
        }
        final Connection connection = this.connectionStore.get(peerAddress);
        if (!(connection != null || records.size() == 1 && records.get(0).isNewClientHello())) {
            LOGGER.debug("Discarding {} records from [{}] received without existing connection", (Object)records.size(), (Object)peerAddress);
            return;
        }
        SerialExecutor serialExecutor = this.getSerialExecutor(peerAddress);
        if (serialExecutor == null) {
            LOGGER.debug("Execution cache is full while processing incoming records from peer: {}", (Object)peerAddress);
            return;
        }
        for (final Record record : records) {
            try {
                serialExecutor.execute(new Runnable(){

                    @Override
                    public void run() {
                        if (DTLSConnector.this.running.get()) {
                            DTLSConnector.this.processRecord(record, connection);
                        }
                    }
                });
            }
            catch (RejectedExecutionException e) {
                LOGGER.debug("Execution rejected while processing record [type: {}, peer: {}]", new Object[]{record.getType(), peerAddress, e});
                break;
            }
            catch (RuntimeException e) {
                if (connection == null) break;
                LOGGER.info("Unexpected error occurred while processing record [type: {}, peer: {}]", new Object[]{record.getType(), peerAddress, e});
                this.terminateConnection(connection, e, AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.INTERNAL_ERROR);
                break;
            }
        }
    }

    private void processRecord(Record record, Connection connection) {
        block8: {
            try {
                LOGGER.trace("Received DTLS record of type [{}], length: {}, [epoche:{},reqn:{}]", new Object[]{record.getType(), record.getLength(), record.getEpoch(), record.getSequenceNumber()});
                switch (record.getType()) {
                    case APPLICATION_DATA: {
                        this.processApplicationDataRecord(record, connection);
                        break;
                    }
                    case ALERT: {
                        this.processAlertRecord(record, connection);
                        break;
                    }
                    case CHANGE_CIPHER_SPEC: {
                        this.processChangeCipherSpecRecord(record, connection);
                        break;
                    }
                    case HANDSHAKE: {
                        this.processHandshakeRecord(record, connection);
                        break;
                    }
                    default: {
                        LOGGER.debug("Discarding record of unsupported type [{}] from peer [{}]", (Object)record.getType(), (Object)record.getPeerAddress());
                        break;
                    }
                }
            }
            catch (RuntimeException e) {
                if (connection == null) break block8;
                LOGGER.info("Unexpected error occurred while processing record from peer [{}]", (Object)record.getPeerAddress(), (Object)e);
                this.terminateConnection(connection, e, AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.INTERNAL_ERROR);
            }
        }
    }

    private void terminateOngoingHandshake(Connection connection, Throwable cause, AlertMessage.AlertDescription description) {
        if (connection.hasOngoingHandshake()) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Aborting handshake with peer [{}]:", (Object)connection.getPeerAddress(), (Object)cause);
            } else if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Aborting handshake with peer [{}]: {}", (Object)connection.getPeerAddress(), (Object)cause.getMessage());
            }
            Handshaker handshaker = connection.getOngoingHandshake();
            DTLSSession session = handshaker.getSession();
            AlertMessage alert = new AlertMessage(AlertMessage.AlertLevel.FATAL, description, connection.getPeerAddress());
            if (!connection.hasEstablishedSession()) {
                this.terminateConnection(connection, alert, session);
            } else {
                this.send(alert, session);
            }
            handshaker.handshakeFailed(cause);
        }
    }

    private void terminateConnection(Connection connection) {
        if (connection != null) {
            connection.cancelPendingFlight();
            this.connectionStore.remove(connection);
        }
    }

    private void terminateConnection(Connection connection, Throwable cause, AlertMessage.AlertLevel level, AlertMessage.AlertDescription description) {
        if (connection.hasEstablishedSession()) {
            this.terminateConnection(connection, new AlertMessage(level, description, connection.getPeerAddress()), connection.getEstablishedSession());
        } else if (connection.hasOngoingHandshake()) {
            this.terminateConnection(connection, new AlertMessage(level, description, connection.getPeerAddress()), connection.getOngoingHandshake().getSession());
        }
    }

    private void terminateConnection(Connection connection, AlertMessage alert, DTLSSession session) {
        if (alert != null && session == null) {
            throw new IllegalArgumentException("Session must not be NULL if alert message is to be sent");
        }
        connection.cancelPendingFlight();
        if (alert == null) {
            LOGGER.debug("Terminating connection with peer [{}]", (Object)connection.getPeerAddress());
        } else {
            LOGGER.debug("Terminating connection with peer [{}], reason [{}]", (Object)connection.getPeerAddress(), (Object)alert.getDescription());
            this.send(alert, session);
        }
        this.connectionStore.remove(connection);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processApplicationDataRecord(Record record, Connection connection) {
        DTLSSession session = connection.getEstablishedSession();
        if (session != null) {
            RawData receivedApplicationMessage = null;
            Handshaker ongoingHandshake = connection.getOngoingHandshake();
            DTLSSession dTLSSession = session;
            synchronized (dTLSSession) {
                if (session.isRecordProcessable(record.getEpoch(), record.getSequenceNumber())) {
                    try {
                        record.setSession(session);
                        ApplicationMessage message = (ApplicationMessage)record.getFragment();
                        session.markRecordAsRead(record.getEpoch(), record.getSequenceNumber());
                        receivedApplicationMessage = RawData.inbound((byte[])message.getData(), (EndpointContext)session.getConnectionReadContext(), (boolean)false);
                    }
                    catch (GeneralSecurityException | HandshakeException e) {
                        DTLSConnector.discardRecord(record, e);
                    }
                } else {
                    LOGGER.debug("Discarding duplicate APPLICATION_DATA record received from peer [{}]", (Object)record.getPeerAddress());
                }
            }
            if (receivedApplicationMessage != null) {
                if (ongoingHandshake != null) {
                    ongoingHandshake.handshakeCompleted();
                }
                connection.refreshAutoResumptionTime();
                this.connectionStore.update(connection);
                RawDataChannel channel = this.messageHandler;
                if (channel != null) {
                    channel.receiveData(receivedApplicationMessage);
                }
            }
        } else {
            Handshaker ongoingHandshake = connection.getOngoingHandshake();
            if (ongoingHandshake != null && ongoingHandshake.isChangeCipherSpecMessageExpected()) {
                ongoingHandshake.addApplicationDataForDeferredProcessing(record);
            } else {
                LOGGER.debug("Discarding APPLICATION_DATA record received from peer [{}] without an active session", (Object)record.getPeerAddress());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processAlertRecord(Record record, Connection connection) {
        DTLSSession session = connection.getSession(record.getEpoch());
        if (session == null) {
            LOGGER.debug("Epoch of ALERT record [epoch={}] from [{}] does not match expected epoch(s), discarding ...", (Object)record.getEpoch(), (Object)record.getPeerAddress());
            return;
        }
        record.setSession(session);
        try {
            AlertMessage alert = (AlertMessage)record.getFragment();
            Handshaker handshaker = connection.getOngoingHandshake();
            HandshakeException error = null;
            LOGGER.trace("Processing {} ALERT from [{}]: {}", new Object[]{alert.getLevel(), alert.getPeer(), alert.getDescription()});
            if (AlertMessage.AlertDescription.CLOSE_NOTIFY.equals((Object)alert.getDescription())) {
                error = new HandshakeException("Received 'close notify'", alert);
                this.terminateConnection(connection, new AlertMessage(AlertMessage.AlertLevel.WARNING, AlertMessage.AlertDescription.CLOSE_NOTIFY, alert.getPeer()), session);
            } else if (AlertMessage.AlertLevel.FATAL.equals((Object)alert.getLevel())) {
                error = new HandshakeException("Received 'fatal alert'", alert);
                this.terminateConnection(connection);
            }
            Object object = this.alertHandlerLock;
            synchronized (object) {
                if (this.alertHandler != null) {
                    this.alertHandler.onAlert(alert.getPeer(), alert);
                }
            }
            if (null != error && null != handshaker) {
                handshaker.handshakeFailed(error);
            }
        }
        catch (GeneralSecurityException | HandshakeException e) {
            DTLSConnector.discardRecord(record, e);
        }
    }

    private void processChangeCipherSpecRecord(Record record, Connection connection) {
        Handshaker ongoingHandshaker = connection.getOngoingHandshake();
        if (ongoingHandshaker != null) {
            try {
                ongoingHandshaker.processMessage(record);
            }
            catch (HandshakeException e) {
                this.handleExceptionDuringHandshake(e, e.getAlert().getLevel(), e.getAlert().getDescription(), connection, record);
            }
        } else {
            LOGGER.debug("Received CHANGE_CIPHER_SPEC record from peer [{}] with no handshake going on", (Object)record.getPeerAddress());
        }
    }

    private void processHandshakeRecord(Record record, Connection connection) {
        block12: {
            LOGGER.debug("Received {} record from peer [{}]", (Object)record.getType(), (Object)record.getPeerAddress());
            try {
                if (connection != null && connection.hasOngoingHandshake()) {
                    DTLSSession handshakeSession = connection.getOngoingHandshake().getSession();
                    if (handshakeSession.getReadEpoch() == record.getEpoch()) {
                        record.setSession(handshakeSession);
                    } else if (!record.isNewClientHello()) {
                        connection.getOngoingHandshake().processMessage(record);
                        return;
                    }
                } else if (connection != null && connection.hasEstablishedSession() && connection.getEstablishedSession().getReadEpoch() == record.getEpoch()) {
                    record.setSession(connection.getEstablishedSession());
                } else if (!record.isNewClientHello()) {
                    LOGGER.debug("Discarding HANDSHAKE message [epoch={}] from peer [{}] which does not match expected epoch(s)", (Object)record.getEpoch(), (Object)record.getPeerAddress());
                    return;
                }
                try {
                    HandshakeMessage handshakeMessage = (HandshakeMessage)record.getFragment();
                    this.processDecryptedHandshakeMessage(handshakeMessage, record, connection);
                }
                catch (GeneralSecurityException e) {
                    DTLSConnector.discardRecord(record, e);
                }
            }
            catch (HandshakeException e) {
                if (connection == null) break block12;
                this.handleExceptionDuringHandshake(e, e.getAlert().getLevel(), e.getAlert().getDescription(), connection, record);
            }
        }
    }

    private void processDecryptedHandshakeMessage(HandshakeMessage handshakeMessage, Record record, Connection connection) throws HandshakeException {
        switch (handshakeMessage.getMessageType()) {
            case CLIENT_HELLO: {
                this.processClientHello((ClientHello)handshakeMessage, record, connection);
                break;
            }
            case HELLO_REQUEST: {
                this.processHelloRequest(connection);
                break;
            }
            default: {
                this.processOngoingHandshakeMessage(handshakeMessage, record, connection);
            }
        }
    }

    private void processOngoingHandshakeMessage(HandshakeMessage message, Record record, Connection connection) throws HandshakeException {
        if (connection.hasOngoingHandshake()) {
            connection.getOngoingHandshake().processMessage(record);
        } else {
            LOGGER.debug("Discarding {} message received from peer [{}] with no handshake going on", (Object)message.getMessageType(), (Object)message.getPeer());
        }
    }

    private void processHelloRequest(Connection connection) throws HandshakeException {
        if (connection.hasOngoingHandshake()) {
            LOGGER.debug("Ignoring HELLO_REQUEST received from [{}] while already in an ongoing handshake with peer", (Object)connection.getPeerAddress());
        } else {
            DTLSSession session = connection.getEstablishedSession();
            this.send(new AlertMessage(AlertMessage.AlertLevel.WARNING, AlertMessage.AlertDescription.NO_RENEGOTIATION, connection.getPeerAddress()), session);
        }
    }

    private void processClientHello(ClientHello clientHello, Record record, Connection connection) throws HandshakeException {
        if (LOGGER.isDebugEnabled()) {
            StringBuilder msg = new StringBuilder("Processing CLIENT_HELLO from peer [").append(record.getPeerAddress()).append("]");
            if (LOGGER.isTraceEnabled()) {
                msg.append(":").append(StringUtil.lineSeparator()).append(record);
            }
            LOGGER.debug(msg.toString());
        }
        if (record.getEpoch() > 0) {
            DTLSSession session = connection.getEstablishedSession();
            this.send(new AlertMessage(AlertMessage.AlertLevel.WARNING, AlertMessage.AlertDescription.NO_RENEGOTIATION, clientHello.getPeer()), session);
            return;
        }
        if (this.isClientInControlOfSourceIpAddress(clientHello, record, connection)) {
            if (connection != null && connection.hasOngoingHandshakeStartedByMessage(clientHello)) {
                this.processOngoingHandshakeMessage(clientHello, record, connection);
            } else if (clientHello.hasSessionId()) {
                this.resumeExistingSession(clientHello, record, connection);
            } else {
                this.terminateConnection(connection);
                this.startNewHandshake(clientHello, record);
            }
        }
    }

    private boolean isClientInControlOfSourceIpAddress(ClientHello clientHello, Record record, Connection connection) {
        try {
            Connection sessionConnection;
            byte[] expectedCookie = null;
            byte[] providedCookie = clientHello.getCookie();
            if (providedCookie != null && providedCookie.length > 0) {
                expectedCookie = this.cookieGenerator.generateCookie(clientHello);
                if (Arrays.equals(expectedCookie, providedCookie)) {
                    return true;
                }
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("provided cookie must {} match {}. Send verify request to {}", new Object[]{StringUtil.byteArray2HexString((byte[])providedCookie, (char)'\u0000', (int)6), StringUtil.byteArray2HexString((byte[])expectedCookie, (char)'\u0000', (int)6), record.getPeerAddress()});
                }
            } else if (0 < this.thresholdHandshakesWithoutVerifiedPeer && (sessionConnection = this.connectionStore.find(clientHello.getSessionId())) != null) {
                if (record.getPeerAddress().equals(sessionConnection.getPeerAddress())) {
                    LOGGER.trace("resuming peer's [{}] session", (Object)record.getPeerAddress());
                    return true;
                }
                if (connection == null || !connection.hasEstablishedSession()) {
                    int pending = this.pendingHandshakesWithoutVerifiedPeer.get();
                    LOGGER.trace("pending fast resumptions [{}], threshold [{}]", (Object)pending, (Object)this.thresholdHandshakesWithoutVerifiedPeer);
                    if (pending < this.thresholdHandshakesWithoutVerifiedPeer) {
                        LOGGER.trace("fast resume for peer [{}] [{}]", (Object)record.getPeerAddress(), (Object)pending);
                        return true;
                    }
                }
            }
            if (expectedCookie == null) {
                expectedCookie = this.cookieGenerator.generateCookie(clientHello);
            }
            this.sendHelloVerify(clientHello, record, expectedCookie);
            return false;
        }
        catch (GeneralSecurityException e) {
            throw new DtlsHandshakeException("Cannot compute cookie for peer", AlertMessage.AlertDescription.INTERNAL_ERROR, AlertMessage.AlertLevel.FATAL, clientHello.getPeer(), e);
        }
    }

    private void startNewHandshake(ClientHello clientHello, Record record) throws HandshakeException {
        Connection peerConnection = new Connection(record.getPeerAddress());
        if (!this.connectionStore.put(peerConnection)) {
            this.terminateOngoingHandshake(peerConnection, new IllegalStateException("connection store exhausted!"), AlertMessage.AlertDescription.INTERNAL_ERROR);
            return;
        }
        DTLSSession newSession = new DTLSSession(record.getPeerAddress(), record.getSequenceNumber());
        ServerHandshaker handshaker = new ServerHandshaker(clientHello.getMessageSeq(), newSession, (RecordLayer)this, peerConnection.getSessionListener(), this.config, this.maximumTransmissionUnit);
        this.initializeHandshaker(handshaker);
        handshaker.processMessage(record);
    }

    private void resumeExistingSession(ClientHello clientHello, Record record, Connection connection) throws HandshakeException {
        LOGGER.debug("Client [{}] wants to resume session with ID [{}]", (Object)clientHello.getPeer(), (Object)clientHello.getSessionId());
        SessionTicket ticket = null;
        final Connection previousConnection = this.connectionStore.find(clientHello.getSessionId());
        if (previousConnection != null && previousConnection.isActive()) {
            if (previousConnection.hasEstablishedSession()) {
                ticket = previousConnection.getEstablishedSession().getSessionTicket();
            } else if (previousConnection.hasSessionTicket()) {
                ticket = previousConnection.getSessionTicket();
            }
            if (ticket != null && this.config.isSniEnabled().booleanValue()) {
                ServerNames serverNames1 = ticket.getServerNames();
                ServerNames serverNames2 = null;
                ServerNameExtension extension = clientHello.getServerNameExtension();
                if (extension != null) {
                    serverNames2 = extension.getServerNames();
                }
                if (serverNames1 != null) {
                    if (!serverNames1.equals(serverNames2)) {
                        ticket = null;
                    }
                } else if (serverNames2 != null) {
                    ticket = null;
                }
            }
        }
        if (ticket != null) {
            Handshaker pendingHandshaker;
            Connection peerConnection = new Connection(record.getPeerAddress());
            DTLSSession sessionToResume = new DTLSSession(clientHello.getSessionId(), record.getPeerAddress(), ticket, record.getSequenceNumber());
            ResumingServerHandshaker handshaker = new ResumingServerHandshaker(clientHello.getMessageSeq(), sessionToResume, (RecordLayer)this, peerConnection.getSessionListener(), this.config, this.maximumTransmissionUnit);
            this.initializeHandshaker(handshaker);
            if (previousConnection.hasEstablishedSession()) {
                if (!previousConnection.getPeerAddress().equals(peerConnection.getPeerAddress())) {
                    handshaker.addSessionListener(new SessionAdapter(){

                        @Override
                        public void sessionEstablished(Handshaker currentHandshaker, DTLSSession establishedSession) throws HandshakeException {
                            LOGGER.debug("Discarding existing connection to [{}] after successful resumption of session [ID={}] by peer [{}]", new Object[]{previousConnection.getPeerAddress(), establishedSession.getSessionIdentifier(), establishedSession.getPeer()});
                            previousConnection.cancelPendingFlight();
                            DTLSConnector.this.connectionStore.remove(previousConnection, false);
                        }
                    });
                    if (clientHello.getCookie().length == 0) {
                        this.pendingHandshakesWithoutVerifiedPeer.incrementAndGet();
                        handshaker.addSessionListener(new SessionAdapter(){

                            @Override
                            public void sessionEstablished(Handshaker currentHandshaker, DTLSSession establishedSession) throws HandshakeException {
                                DTLSConnector.this.pendingHandshakesWithoutVerifiedPeer.decrementAndGet();
                            }

                            @Override
                            public void handshakeFailed(Handshaker handshaker, Throwable error) {
                                DTLSConnector.this.pendingHandshakesWithoutVerifiedPeer.decrementAndGet();
                            }
                        });
                    }
                } else {
                    previousConnection.cancelPendingFlight();
                    this.connectionStore.remove(previousConnection, false);
                }
            }
            if (connection != null && !connection.hasEstablishedSession() && (pendingHandshaker = connection.getOngoingHandshake()) != null) {
                pendingHandshaker.handshakeFailed(new IOException("ongoing handshake resumed!"));
            }
            this.connectionStore.put(peerConnection);
            handshaker.processMessage(record);
        } else {
            LOGGER.debug("Client [{}] tries to resume non-existing session [ID={}], performing full handshake instead ...", (Object)clientHello.getPeer(), (Object)clientHello.getSessionId());
            this.terminateConnection(connection);
            this.startNewHandshake(clientHello, record);
        }
    }

    private void sendHelloVerify(ClientHello clientHello, Record record, byte[] expectedCookie) {
        LOGGER.debug("Verifying client IP address [{}] using HELLO_VERIFY_REQUEST", (Object)record.getPeerAddress());
        HelloVerifyRequest msg = new HelloVerifyRequest(new ProtocolVersion(), expectedCookie, record.getPeerAddress());
        msg.setMessageSeq(clientHello.getMessageSeq());
        Record helloVerify = new Record(ContentType.HANDSHAKE, 0, record.getSequenceNumber(), (DTLSMessage)msg, record.getPeerAddress());
        try {
            this.sendRecord(helloVerify);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    void send(AlertMessage alert, DTLSSession session) {
        if (alert == null) {
            throw new IllegalArgumentException("Alert must not be NULL");
        }
        if (session == null) {
            throw new IllegalArgumentException("Session must not be NULL");
        }
        try {
            this.sendRecord(new Record(ContentType.ALERT, session.getWriteEpoch(), session.getSequenceNumber(), (DTLSMessage)alert, session));
        }
        catch (IOException iOException) {
        }
        catch (GeneralSecurityException e) {
            LOGGER.debug("Cannot create ALERT message for peer [{}]", (Object)session.getPeer(), (Object)e);
        }
    }

    public final void send(final RawData msg) {
        if (msg == null) {
            throw new NullPointerException("Message must not be null");
        }
        if (msg.isMulticast()) {
            LOGGER.warn("DTLSConnector drops {} bytes to multicast {}:{}", new Object[]{msg.getSize(), msg.getAddress(), msg.getPort()});
            msg.onError((Throwable)new MulticastNotSupportedException("DTLS doesn't support multicast!"));
            return;
        }
        SerialExecutor serialExecutor = null;
        RuntimeException error = null;
        if (!this.running.get()) {
            error = new IllegalStateException("connector must be started before sending messages is possible");
        } else if (msg.getSize() > 16384) {
            error = new IllegalArgumentException("Message data must not exceed 16384 bytes");
        } else {
            serialExecutor = this.getSerialExecutor(msg.getInetSocketAddress());
            if (serialExecutor == null) {
                error = new IllegalStateException("connector's serial-executors-cache is exhausted!");
            }
        }
        if (error != null) {
            msg.onError((Throwable)error);
            throw error;
        }
        if (this.pendingOutboundMessagesCountdown.decrementAndGet() >= 0) {
            try {
                serialExecutor.execute(new Runnable(){

                    /*
                     * WARNING - Removed try catching itself - possible behaviour change.
                     */
                    @Override
                    public void run() {
                        try {
                            if (DTLSConnector.this.running.get()) {
                                DTLSConnector.this.sendMessage(msg);
                            } else {
                                msg.onError((Throwable)new InterruptedIOException("Connector is not running."));
                            }
                        }
                        catch (Exception e) {
                            if (DTLSConnector.this.running.get()) {
                                LOGGER.debug("Exception thrown by executor thread [{}]", (Object)Thread.currentThread().getName(), (Object)e);
                            }
                            msg.onError((Throwable)e);
                        }
                        finally {
                            DTLSConnector.this.pendingOutboundMessagesCountdown.incrementAndGet();
                        }
                    }
                });
            }
            catch (RejectedExecutionException e) {
                LOGGER.debug("Execution rejected while sending application record [peer: {}]", (Object)msg.getInetSocketAddress(), (Object)e);
                msg.onError((Throwable)new InterruptedIOException("Connector is not running."));
            }
        } else {
            this.pendingOutboundMessagesCountdown.incrementAndGet();
            LOGGER.warn("Outbound message overflow! Dropping outbound message to peer [{}]", (Object)msg.getInetSocketAddress());
            msg.onError((Throwable)new IllegalStateException("Outbound message overflow!"));
        }
    }

    private void sendMessage(RawData message) throws HandshakeException {
        InetSocketAddress peerAddress = message.getInetSocketAddress();
        LOGGER.debug("Sending application layer message to peer [{}]", (Object)peerAddress);
        Connection connection = this.connectionStore.get(peerAddress);
        if (connection == null) {
            if (this.serverOnly) {
                message.onError((Throwable)new EndpointUnconnectedException());
                return;
            }
            connection = new Connection(peerAddress);
            this.connectionStore.put(connection);
        }
        DTLSSession session = connection.getEstablishedSession();
        SessionTicket ticket = connection.getSessionTicket();
        if (session == null && ticket == null) {
            if (this.serverOnly) {
                message.onError((Throwable)new EndpointUnconnectedException());
                return;
            }
            if (!this.checkOutboundEndpointContext(message, null)) {
                return;
            }
            message.onConnecting();
            Handshaker handshaker = connection.getOngoingHandshake();
            if (handshaker == null) {
                session = new DTLSSession(peerAddress);
                session.setVirtualHost(message.getEndpointContext().getVirtualHost());
                handshaker = new ClientHandshaker(session, this, connection.getSessionListener(), this.config, this.maximumTransmissionUnit);
                this.initializeHandshaker(handshaker);
                handshaker.startHandshake();
            }
            handshaker.addApplicationDataForDeferredProcessing(message);
        } else {
            Long timeout = this.autoResumptionTimeoutMillis;
            String contextTimeout = message.getEndpointContext().get("*DTLS_RESUMPTION_TIMEOUT");
            if (contextTimeout != null) {
                if (contextTimeout.isEmpty()) {
                    timeout = null;
                } else {
                    try {
                        timeout = Long.valueOf(contextTimeout);
                    }
                    catch (NumberFormatException numberFormatException) {
                        // empty catch block
                    }
                }
            }
            if (connection.isAutoResumptionRequired(timeout)) {
                ClientHandshaker handshaker;
                SessionId sessionId;
                if (this.serverOnly) {
                    message.onError((Throwable)new EndpointMismatchException());
                    return;
                }
                message.onConnecting();
                if (ticket == null) {
                    ticket = session.getSessionTicket();
                    sessionId = session.getSessionIdentifier();
                } else {
                    sessionId = connection.getSessionIdentity();
                }
                Connection newConnection = new Connection(peerAddress);
                connection.cancelPendingFlight();
                this.connectionStore.remove(connection, false);
                this.connectionStore.put(newConnection);
                if (sessionId.isEmpty()) {
                    DTLSSession newSession = new DTLSSession(peerAddress);
                    newSession.setVirtualHost(message.getEndpointContext().getVirtualHost());
                    handshaker = new ClientHandshaker(newSession, this, newConnection.getSessionListener(), this.config, this.maximumTransmissionUnit);
                } else {
                    DTLSSession resumableSession = new DTLSSession(sessionId, peerAddress, ticket, 0L);
                    resumableSession.setVirtualHost(message.getEndpointContext().getVirtualHost());
                    handshaker = new ResumingClientHandshaker(resumableSession, this, newConnection.getSessionListener(), this.config, this.maximumTransmissionUnit);
                }
                this.initializeHandshaker(handshaker);
                Handshaker previous = connection.getOngoingHandshake();
                if (previous != null) {
                    handshaker.takeDeferredApplicationData(previous);
                }
                handshaker.addApplicationDataForDeferredProcessing(message);
                ((Handshaker)handshaker).startHandshake();
            } else {
                this.sendMessage(message, connection, session);
            }
        }
    }

    private void sendMessage(RawData message, Connection connection, DTLSSession session) {
        try {
            DtlsEndpointContext ctx = session.getConnectionWriteContext();
            if (!this.checkOutboundEndpointContext(message, (EndpointContext)ctx)) {
                return;
            }
            message.onContextEstablished((EndpointContext)ctx);
            Record record = new Record(ContentType.APPLICATION_DATA, session.getWriteEpoch(), session.getSequenceNumber(), (DTLSMessage)new ApplicationMessage(message.getBytes(), message.getInetSocketAddress()), session);
            this.sendRecord(record);
            message.onSent();
            connection.refreshAutoResumptionTime();
        }
        catch (IOException e) {
            message.onError((Throwable)e);
        }
        catch (GeneralSecurityException e) {
            LOGGER.debug("Cannot send APPLICATION record to peer [{}]", (Object)message.getInetSocketAddress(), (Object)e);
            message.onError((Throwable)e);
        }
    }

    private boolean checkOutboundEndpointContext(RawData message, EndpointContext connectionContext) {
        EndpointContextMatcher endpointMatcher = this.getEndpointContextMatcher();
        if (null != endpointMatcher && !endpointMatcher.isToBeSent(message.getEndpointContext(), connectionContext)) {
            LOGGER.warn("DTLSConnector ({}) drops {} bytes to {}:{}", new Object[]{this, message.getSize(), message.getAddress(), message.getPort()});
            message.onError((Throwable)new EndpointMismatchException());
            return false;
        }
        return true;
    }

    public final DTLSSession getSessionByAddress(InetSocketAddress address) {
        if (address == null) {
            return null;
        }
        Connection connection = this.connectionStore.get(address);
        if (connection != null) {
            return connection.getEstablishedSession();
        }
        return null;
    }

    @Override
    public void sendFlight(DTLSFlight flight) throws IOException {
        if (flight != null) {
            if (flight.isRetransmissionNeeded()) {
                this.scheduleRetransmission(flight);
            }
            this.sendFlightOverNetwork(flight);
        }
    }

    private void sendFlightOverNetwork(DTLSFlight flight) throws IOException {
        byte[] payload = new byte[]{};
        int maxDatagramSize = this.maximumTransmissionUnit;
        if (flight.getSession() != null) {
            maxDatagramSize = flight.getSession().getMaxDatagramSize();
        }
        ArrayList<DatagramPacket> datagrams = new ArrayList<DatagramPacket>();
        for (Record record : flight.getMessages()) {
            byte[] recordBytes = record.toByteArray();
            if (recordBytes.length > maxDatagramSize) {
                LOGGER.info("{} record of {} bytes for peer [{}] exceeds max. datagram size [{}], discarding...", new Object[]{record.getType(), recordBytes.length, record.getPeerAddress(), maxDatagramSize});
                continue;
            }
            LOGGER.trace("Sending record of {} bytes to peer [{}]:\n{}", new Object[]{recordBytes.length, flight.getPeerAddress(), record});
            if (payload.length + recordBytes.length > maxDatagramSize) {
                DatagramPacket datagram = new DatagramPacket(payload, payload.length, flight.getPeerAddress().getAddress(), flight.getPeerAddress().getPort());
                datagrams.add(datagram);
                payload = new byte[]{};
            }
            payload = ByteArrayUtils.concatenate(payload, recordBytes);
        }
        DatagramPacket datagram = new DatagramPacket(payload, payload.length, flight.getPeerAddress().getAddress(), flight.getPeerAddress().getPort());
        datagrams.add(datagram);
        LOGGER.debug("Sending flight of {} message(s) to peer [{}] using {} datagram(s) of max. {} bytes", new Object[]{flight.getMessages().size(), flight.getPeerAddress(), datagrams.size(), maxDatagramSize});
        for (DatagramPacket datagramPacket : datagrams) {
            this.sendNextDatagramOverNetwork(datagramPacket);
        }
    }

    protected void sendRecord(Record record) throws IOException {
        byte[] recordBytes = record.toByteArray();
        DatagramPacket datagram = new DatagramPacket(recordBytes, recordBytes.length, record.getPeerAddress());
        this.sendNextDatagramOverNetwork(datagram);
    }

    protected void sendNextDatagramOverNetwork(DatagramPacket datagramPacket) throws IOException {
        DatagramSocket socket = this.getSocket();
        if (socket != null && !socket.isClosed()) {
            try {
                socket.send(datagramPacket);
            }
            catch (IOException e) {
                LOGGER.warn("Could not send record", (Throwable)e);
                throw e;
            }
        } else {
            LOGGER.debug("Socket [{}] is closed, discarding packet ...", (Object)this.config.getAddress());
            throw new IOException("Socket closed.");
        }
    }

    private void handleTimeout(DTLSFlight flight) {
        Handshaker handshaker;
        Connection connection;
        if (!flight.isResponseCompleted() && null != (connection = this.connectionStore.get(flight.getPeerAddress())) && !connection.hasEstablishedSession() && null != (handshaker = connection.getOngoingHandshake())) {
            Exception cause = null;
            if (this.running.get()) {
                int max = this.config.getMaxRetransmissions();
                int tries = flight.getTries();
                if (tries < max) {
                    if (this.config.isEarlyStopRetransmission().booleanValue() && flight.isResponseStarted()) {
                        while (tries < max) {
                            ++tries;
                            flight.incrementTries();
                            flight.incrementTimeout();
                        }
                        flight.incrementTries();
                        LOGGER.debug("schedule handshake timeout {}ms after flight {}", (Object)flight.getTimeout(), (Object)flight.getFlightNumber());
                        ScheduledFuture<?> f = this.timer.schedule(new TimeoutPeerTask(flight), (long)flight.getTimeout(), TimeUnit.MILLISECONDS);
                        flight.setTimeoutTask(f);
                        return;
                    }
                    LOGGER.debug("Re-transmitting flight for [{}], [{}] retransmissions left", (Object)flight.getPeerAddress(), (Object)(max - tries - 1));
                    try {
                        flight.incrementTries();
                        flight.setNewSequenceNumbers();
                        this.sendFlightOverNetwork(flight);
                        this.scheduleRetransmission(flight);
                        handshaker.handshakeFlightRetransmitted(flight.getFlightNumber());
                        return;
                    }
                    catch (IOException e) {
                        cause = e;
                        LOGGER.info("Cannot retransmit flight to peer [{}]", (Object)flight.getPeerAddress(), (Object)e);
                    }
                    catch (GeneralSecurityException e) {
                        LOGGER.info("Cannot retransmit flight to peer [{}]", (Object)flight.getPeerAddress(), (Object)e);
                        cause = e;
                    }
                } else if (tries > max) {
                    LOGGER.debug("Flight for [{}] has reached timeout, discarding ...", (Object)flight.getPeerAddress());
                    cause = new IOException("handshake timeout with flight " + flight.getFlightNumber() + "!");
                } else {
                    LOGGER.debug("Flight for [{}] has reached maximum no. [{}] of retransmissions, discarding ...", (Object)flight.getPeerAddress(), (Object)max);
                    cause = new IOException("handshake flight " + flight.getFlightNumber() + " timeout after " + max + " retransmissions!");
                }
            }
            cause = cause == null ? new Exception("handshake flight " + flight.getFlightNumber() + " timeout!") : new Exception("handshake flight " + flight.getFlightNumber() + " failed!", cause);
            handshaker.handshakeFailed(cause);
        }
    }

    private void scheduleRetransmission(DTLSFlight flight) {
        if (flight.isRetransmissionNeeded()) {
            if (flight.getTimeout() == 0) {
                flight.setTimeout(this.config.getRetransmissionTimeout());
            } else {
                flight.incrementTimeout();
            }
            ScheduledFuture<?> f = this.timer.schedule(new TimeoutPeerTask(flight), (long)flight.getTimeout(), TimeUnit.MILLISECONDS);
            flight.setTimeoutTask(f);
        }
    }

    public final int getMaximumTransmissionUnit() {
        return this.maximumTransmissionUnit;
    }

    public final int getMaximumFragmentLength(InetSocketAddress peer) {
        Connection con = this.connectionStore.get(peer);
        if (con != null && con.getEstablishedSession() != null) {
            return con.getEstablishedSession().getMaxFragmentLength();
        }
        return this.maximumTransmissionUnit - 89;
    }

    public final InetSocketAddress getAddress() {
        DatagramSocket socket = this.getSocket();
        if (socket == null) {
            return this.config.getAddress();
        }
        return new InetSocketAddress(socket.getLocalAddress(), socket.getLocalPort());
    }

    public final boolean isRunning() {
        return this.running.get();
    }

    public void setRawDataReceiver(RawDataChannel messageHandler) {
        if (this.isRunning()) {
            throw new IllegalStateException("message handler cannot be set on running connector");
        }
        this.messageHandler = messageHandler;
    }

    public synchronized void setEndpointContextMatcher(EndpointContextMatcher endpointContextMatcher) {
        this.endpointContextMatcher = endpointContextMatcher;
    }

    private synchronized EndpointContextMatcher getEndpointContextMatcher() {
        return this.endpointContextMatcher;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void setAlertHandler(AlertHandler handler) {
        Object object = this.alertHandlerLock;
        synchronized (object) {
            this.alertHandler = handler;
        }
    }

    private void handleExceptionDuringHandshake(HandshakeException cause, AlertMessage.AlertLevel level, AlertMessage.AlertDescription description, Connection connection, Record record) {
        if (!AlertMessage.AlertLevel.FATAL.equals((Object)level)) {
            DTLSConnector.discardRecord(record, cause);
            return;
        }
        if (AlertMessage.AlertDescription.UNKNOWN_PSK_IDENTITY == description) {
            DTLSConnector.discardRecord(record, cause);
            return;
        }
        this.terminateOngoingHandshake(connection, cause, description);
    }

    private static void discardRecord(Record record, Throwable cause) {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Discarding {} record from peer [{}]: ", new Object[]{record.getType(), record.getPeerAddress(), cause});
        } else if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Discarding {} record from peer [{}]: {}", new Object[]{record.getType(), record.getPeerAddress(), cause.getMessage()});
        }
    }

    public String getProtocol() {
        return "DTLS";
    }

    public String toString() {
        return this.getProtocol() + "-" + this.getAddress();
    }

    private abstract class Worker
    extends Thread {
        protected Worker(String name) {
            super(NamedThreadFactory.SCANDIUM_THREAD_GROUP, name);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                LOGGER.info("Starting worker thread [{}]", (Object)this.getName());
                while (DTLSConnector.this.running.get()) {
                    try {
                        this.doWork();
                    }
                    catch (ClosedByInterruptException e) {
                        LOGGER.info("Worker thread [{}] has been interrupted", (Object)this.getName());
                    }
                    catch (Exception e) {
                        if (!DTLSConnector.this.running.get()) continue;
                        LOGGER.debug("Exception thrown by worker thread [{}]", (Object)this.getName(), (Object)e);
                    }
                }
            }
            finally {
                LOGGER.info("Worker thread [{}] has terminated", (Object)this.getName());
            }
        }

        protected abstract void doWork() throws Exception;
    }

    private class TimeoutPeerTask
    extends PeerTask {
        private TimeoutPeerTask(final DTLSFlight flight) {
            super(flight.getPeerAddress(), new Runnable(){

                @Override
                public void run() {
                    DTLSConnector.this.handleTimeout(flight);
                }
            }, true);
        }
    }

    private class PeerTask
    implements Runnable {
        private final InetSocketAddress peer;
        private final Runnable task;
        private final boolean force;

        private PeerTask(InetSocketAddress peer, Runnable task, boolean force) {
            this.peer = peer;
            this.task = task;
            this.force = force;
        }

        @Override
        public void run() {
            SerialExecutor serialExecutor = DTLSConnector.this.getSerialExecutor(this.peer);
            if (serialExecutor == null) {
                LOGGER.debug("Execution cache is full while execute task of peer: {}", (Object)this.peer);
            } else {
                try {
                    serialExecutor.execute(this.task);
                    return;
                }
                catch (RejectedExecutionException e) {
                    LOGGER.debug("Execution rejected while execute task of peer: {}", (Object)this.peer, (Object)e);
                }
            }
            if (this.force) {
                this.task.run();
            }
        }
    }
}

