/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.nio.channels.ClosedByInterruptException;
import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.security.cert.Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.RawDataChannel;
import org.eclipse.californium.scandium.DTLSConnectorConfig;
import org.eclipse.californium.scandium.ErrorHandler;
import org.eclipse.californium.scandium.config.DtlsConnectorConfig;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.ApplicationMessage;
import org.eclipse.californium.scandium.dtls.ClientHandshaker;
import org.eclipse.californium.scandium.dtls.ClientHello;
import org.eclipse.californium.scandium.dtls.CompressionMethod;
import org.eclipse.californium.scandium.dtls.Connection;
import org.eclipse.californium.scandium.dtls.ConnectionStore;
import org.eclipse.californium.scandium.dtls.ContentType;
import org.eclipse.californium.scandium.dtls.DTLSFlight;
import org.eclipse.californium.scandium.dtls.DTLSMessage;
import org.eclipse.californium.scandium.dtls.DTLSSession;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.Handshaker;
import org.eclipse.californium.scandium.dtls.HelloVerifyRequest;
import org.eclipse.californium.scandium.dtls.InMemoryConnectionStore;
import org.eclipse.californium.scandium.dtls.MaxFragmentLengthExtension;
import org.eclipse.californium.scandium.dtls.ProtocolVersion;
import org.eclipse.californium.scandium.dtls.Record;
import org.eclipse.californium.scandium.dtls.ResumingClientHandshaker;
import org.eclipse.californium.scandium.dtls.ResumingServerHandshaker;
import org.eclipse.californium.scandium.dtls.ServerHandshaker;
import org.eclipse.californium.scandium.dtls.SessionAdapter;
import org.eclipse.californium.scandium.dtls.SessionId;
import org.eclipse.californium.scandium.dtls.SessionListener;
import org.eclipse.californium.scandium.dtls.cipher.CipherSuite;
import org.eclipse.californium.scandium.dtls.cipher.InvalidMacException;
import org.eclipse.californium.scandium.util.ByteArrayUtils;

public class DTLSConnector
implements Connector {
    private static final Logger LOGGER = Logger.getLogger(DTLSConnector.class.getCanonicalName());
    private static final int MAX_PLAINTEXT_FRAGMENT_LENGTH = 16384;
    private static final int MAX_CIPHERTEXT_EXPANSION = CipherSuite.TLS_PSK_WITH_AES_128_CBC_SHA256.getMaxCiphertextExpansion();
    private static final int MAX_DATAGRAM_BUFFER_SIZE = 16409 + MAX_CIPHERTEXT_EXPANSION;
    private InetSocketAddress lastBindAddress;
    private int maximumTransmissionUnit = 1280;
    private int inboundDatagramBufferSize = MAX_DATAGRAM_BUFFER_SIZE;
    private Object cookieMacKeyLock = new Object();
    private long lastGenerationDate = System.currentTimeMillis();
    private SecretKey cookieMacKey = new SecretKeySpec(this.randomBytes(), "MAC");
    private final DtlsConnectorConfig config;
    private DatagramSocket socket;
    private Timer timer;
    private Worker receiver;
    private Worker sender;
    private final ConnectionStore connectionStore;
    private final BlockingQueue<RawData> outboundMessages;
    private boolean running;
    private RawDataChannel messageHandler;
    private ErrorHandler errorHandler;

    public DTLSConnector(DtlsConnectorConfig configuration) {
        this(configuration, null);
    }

    public DTLSConnector(DtlsConnectorConfig configuration, ConnectionStore connectionStore) {
        if (configuration == null) {
            throw new NullPointerException("Configuration must not be null");
        }
        this.config = configuration;
        this.outboundMessages = new LinkedBlockingQueue<RawData>(this.config.getOutboundMessageBufferSize());
        this.connectionStore = connectionStore != null ? connectionStore : new InMemoryConnectionStore();
    }

    public DTLSConnector(InetSocketAddress address) {
        this(address, null);
    }

    public DTLSConnector(InetSocketAddress address, Certificate[] rootCertificates) {
        this(address, rootCertificates, null, null);
    }

    public DTLSConnector(InetSocketAddress address, Certificate[] rootCertificates, ConnectionStore connectionStore, DTLSConnectorConfig config) {
        DtlsConnectorConfig.Builder builder = new DtlsConnectorConfig.Builder(address);
        if (config != null) {
            builder.setMaxRetransmissions(config.getMaxRetransmit());
            builder.setRetransmissionTimeout(config.getRetransmissionTimeout());
            if (rootCertificates != null) {
                builder.setTrustStore(rootCertificates);
            }
            if (config.pskStore != null) {
                builder.setPskStore(config.pskStore);
            } else if (config.certChain != null) {
                builder.setIdentity(config.privateKey, config.certChain, config.sendRawKey);
            } else {
                builder.setIdentity(config.privateKey, config.publicKey);
            }
        }
        this.config = builder.build();
        this.outboundMessages = new LinkedBlockingQueue<RawData>(this.config.getOutboundMessageBufferSize());
        this.connectionStore = connectionStore != null ? connectionStore : new InMemoryConnectionStore();
    }

    public final void close(InetSocketAddress peerAddress) {
        Connection connection = this.connectionStore.get(peerAddress);
        if (connection != null && connection.getEstablishedSession() != null) {
            this.terminateConnection(connection, new AlertMessage(AlertMessage.AlertLevel.WARNING, AlertMessage.AlertDescription.CLOSE_NOTIFY, peerAddress), connection.getEstablishedSession());
        }
    }

    public final synchronized void start() throws IOException {
        this.start(this.config.getAddress());
    }

    final synchronized void restart() throws IOException {
        if (this.lastBindAddress == null) {
            throw new IllegalStateException("Connector has never been started before");
        }
        this.start(this.lastBindAddress);
    }

    private void start(InetSocketAddress bindAddress) throws IOException {
        if (this.running) {
            return;
        }
        this.timer = new Timer(true);
        this.socket = new DatagramSocket(null);
        this.socket.setReuseAddress(true);
        this.socket.bind(bindAddress);
        NetworkInterface ni = NetworkInterface.getByInetAddress(bindAddress.getAddress());
        if (ni != null && ni.getMTU() > 0) {
            this.maximumTransmissionUnit = ni.getMTU();
        } else {
            LOGGER.config("Cannot determine MTU of network interface, using minimum MTU [1280] of IPv6 instead");
            this.maximumTransmissionUnit = 1280;
        }
        if (this.config.getMaxFragmentLengthCode() != null) {
            MaxFragmentLengthExtension.Length lengthCode = MaxFragmentLengthExtension.Length.fromCode(this.config.getMaxFragmentLengthCode());
            this.inboundDatagramBufferSize = lengthCode.length() + MAX_CIPHERTEXT_EXPANSION + 25;
        }
        this.lastBindAddress = new InetSocketAddress(this.socket.getLocalAddress(), this.socket.getLocalPort());
        this.running = true;
        this.sender = new Worker("DTLS-Sender-" + this.lastBindAddress){

            @Override
            public void doWork() throws Exception {
                DTLSConnector.this.sendNextMessageOverNetwork();
            }
        };
        this.receiver = new Worker("DTLS-Receiver-" + this.lastBindAddress){

            @Override
            public void doWork() throws Exception {
                DTLSConnector.this.receiveNextDatagramFromNetwork();
            }
        };
        this.receiver.start();
        this.sender.start();
        LOGGER.log(Level.INFO, "DLTS connector listening on [{0}] with MTU [{1}] using (inbound) datagram buffer size [{2} bytes]", new Object[]{this.lastBindAddress, this.maximumTransmissionUnit, this.inboundDatagramBufferSize});
    }

    public synchronized void forceResumeSessionFor(InetSocketAddress peer) {
        Connection peerConnection = this.connectionStore.get(peer);
        if (peerConnection != null && peerConnection.getEstablishedSession() != null) {
            peerConnection.setResumptionRequired(true);
        }
    }

    final synchronized void releaseSocket() {
        this.running = false;
        this.sender.interrupt();
        this.outboundMessages.clear();
        if (this.socket != null) {
            this.socket.close();
            this.socket = null;
        }
        this.maximumTransmissionUnit = 0;
    }

    public final synchronized void stop() {
        if (!this.running) {
            return;
        }
        LOGGER.log(Level.INFO, "Stopping DLTS connector on [{0}]", this.lastBindAddress);
        this.timer.cancel();
        this.releaseSocket();
    }

    public final synchronized void destroy() {
        this.stop();
        this.connectionStore.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void receiveNextDatagramFromNetwork() throws IOException {
        byte[] buffer = new byte[this.inboundDatagramBufferSize];
        DatagramPacket packet = new DatagramPacket(buffer, buffer.length);
        DatagramSocket datagramSocket = this.socket;
        synchronized (datagramSocket) {
            this.socket.receive(packet);
        }
        if (packet.getLength() == 0) {
            return;
        }
        InetSocketAddress peerAddress = new InetSocketAddress(packet.getAddress(), packet.getPort());
        byte[] data = Arrays.copyOfRange(packet.getData(), packet.getOffset(), packet.getLength());
        List<Record> records = Record.fromByteArray(data, peerAddress);
        LOGGER.log(Level.FINER, "Received {0} DTLS records using a {1} byte datagram buffer", new Object[]{records.size(), this.inboundDatagramBufferSize});
        for (Record record : records) {
            try {
                LOGGER.log(Level.FINEST, "Received DTLS record of type [{0}]", (Object)record.getType());
                switch (record.getType()) {
                    case APPLICATION_DATA: {
                        this.processApplicationDataRecord(peerAddress, record);
                        break;
                    }
                    case ALERT: {
                        this.processAlertRecord(peerAddress, record);
                        break;
                    }
                    case CHANGE_CIPHER_SPEC: {
                        this.processChangeCipherSpecRecord(peerAddress, record);
                        break;
                    }
                    case HANDSHAKE: {
                        this.processHandshakeRecord(peerAddress, record);
                    }
                }
            }
            catch (InvalidMacException e) {
                LOGGER.log(Level.FINE, "Discarding [{0}] record from peer [{1}]: MAC validation failed", new Object[]{record.getType(), peerAddress});
            }
            catch (GeneralSecurityException e) {
                LOGGER.log(Level.FINE, "Discarding [{0}] record from peer [{1}]: {2}", new Object[]{record.getType(), peerAddress, e.getMessage()});
            }
            catch (HandshakeException e) {
                if (AlertMessage.AlertLevel.FATAL.equals((Object)e.getAlert().getLevel())) {
                    LOGGER.log(Level.INFO, "Aborting handshake with peer [{1}]: {2}", new Object[]{record.getType(), peerAddress, e.getMessage()});
                    this.terminateOngoingHandshake(peerAddress, e.getAlert());
                    break;
                }
                LOGGER.log(Level.FINE, "Discarding [{0}] record from peer [{1}]: {2}", new Object[]{record.getType(), peerAddress, e.getMessage()});
            }
        }
    }

    private void terminateOngoingHandshake(InetSocketAddress peerAddress, AlertMessage alert) {
        Connection connection = this.connectionStore.get(peerAddress);
        if (connection != null && connection.getOngoingHandshake() != null) {
            DTLSSession session = connection.getOngoingHandshake().getSession();
            if (connection.getEstablishedSession() == null) {
                this.terminateConnection(connection, alert, session);
            } else {
                if (alert != null) {
                    this.send(alert, session);
                }
                connection.terminateOngoingHandshake();
            }
        }
    }

    private void terminateConnection(Connection connection, AlertMessage alert, DTLSSession session) {
        if (alert != null && session == null) {
            throw new IllegalArgumentException("Session must not be NULL if alert message is to be sent");
        }
        connection.cancelPendingFlight();
        if (alert == null) {
            LOGGER.log(Level.FINE, "Terminating connection with peer [{0}]", connection.getPeerAddress());
        } else {
            LOGGER.log(Level.FINE, "Terminating connection with peer [{0}], reason [{1}]", new Object[]{connection.getPeerAddress(), alert.getDescription()});
            this.send(alert, session);
        }
        this.connectionClosed(connection.getPeerAddress());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processApplicationDataRecord(InetSocketAddress peerAddress, Record record) throws GeneralSecurityException, HandshakeException {
        Connection connection = this.connectionStore.get(peerAddress);
        DTLSSession session = null;
        if (connection != null) {
            session = connection.getEstablishedSession();
        }
        if (session != null) {
            DTLSSession dTLSSession = session;
            synchronized (dTLSSession) {
                if (session.isRecordProcessable(record.getEpoch(), record.getSequenceNumber())) {
                    record.setSession(session);
                    ApplicationMessage message = (ApplicationMessage)record.getFragment();
                    connection.handshakeCompleted(peerAddress);
                    session.markRecordAsRead(record.getEpoch(), record.getSequenceNumber());
                    if (this.messageHandler != null) {
                        this.messageHandler.receiveData(new RawData(message.getData(), peerAddress, session.getPeerIdentity()));
                    }
                } else {
                    LOGGER.log(Level.FINER, "Discarding duplicate APPLICATION_DATA record received from peer [{0}]", peerAddress);
                }
            }
        } else {
            LOGGER.log(Level.FINER, "Discarding APPLICATION_DATA record received from peer [{0}] without an active session", new Object[]{peerAddress});
        }
    }

    /*
     * Enabled aggressive block sorting
     */
    private void processAlertRecord(InetSocketAddress peerAddress, Record record) throws GeneralSecurityException, HandshakeException {
        DTLSSession session;
        Connection connection;
        block10: {
            connection = this.connectionStore.get(peerAddress);
            if (connection == null) {
                LOGGER.log(Level.FINER, "Received ALERT record from [{0}] without existing connection, discarding ...", peerAddress);
                return;
            }
            session = null;
            if (connection.getEstablishedSession() == null) {
                if (record.getEpoch() > 0) {
                    LOGGER.log(Level.FINER, "Received ALERT record with epoch > 0 from [{0}] without established session, discarding ...", peerAddress);
                    return;
                }
                if (connection.getOngoingHandshake() != null && connection.getOngoingHandshake().getSession() != null && connection.getOngoingHandshake().getSession().getReadEpoch() == 0) {
                    session = connection.getOngoingHandshake().getSession();
                    break block10;
                } else {
                    LOGGER.log(Level.FINER, "Received ALERT record with epoch == 0 from [{0}] without on going handshake, discarding ...", peerAddress);
                    return;
                }
            }
            if (record.getEpoch() == 0) {
                LOGGER.log(Level.FINER, "Received ALERT record with epoch == 0 from [{0}] with established session, discarding ...", peerAddress);
                return;
            }
            session = connection.getEstablishedSession();
        }
        if (session == null) {
            LOGGER.log(Level.FINER, "Received ALERT record from [{0}] without existing session, discarding ...", peerAddress);
            return;
        }
        record.setSession(session);
        AlertMessage alert = (AlertMessage)record.getFragment();
        LOGGER.log(Level.FINEST, "Processing ALERT message from [{0}]:\n{1}", new Object[]{peerAddress, record});
        if (AlertMessage.AlertDescription.CLOSE_NOTIFY.equals((Object)alert.getDescription())) {
            this.terminateConnection(connection, new AlertMessage(AlertMessage.AlertLevel.WARNING, AlertMessage.AlertDescription.CLOSE_NOTIFY, peerAddress), session);
        } else if (AlertMessage.AlertLevel.FATAL.equals((Object)alert.getLevel())) {
            this.terminateConnection(connection, null, null);
        }
        if (this.errorHandler == null) return;
        this.errorHandler.onError(peerAddress, alert.getLevel(), alert.getDescription());
    }

    private void processChangeCipherSpecRecord(InetSocketAddress peerAddress, Record record) throws HandshakeException {
        Connection connection = this.connectionStore.get(peerAddress);
        if (connection == null || connection.getOngoingHandshake() == null) {
            LOGGER.log(Level.FINE, "Discarding CHANGE_CIPHER_SPEC record from peer [{0}], no handshake in progress...", peerAddress);
        } else {
            connection.getOngoingHandshake().processMessage(record);
        }
    }

    private void processHandshakeRecord(InetSocketAddress peerAddress, Record record) throws HandshakeException, GeneralSecurityException {
        LOGGER.log(Level.FINER, "Received HANDSHAKE record from peer [{0}]", peerAddress);
        DTLSFlight flight = null;
        Connection connection = this.connectionStore.get(peerAddress);
        if (connection != null && connection.getOngoingHandshake() != null) {
            flight = connection.getOngoingHandshake().processMessage(record);
        } else {
            if (record.getEpoch() > 0) {
                if (connection != null && connection.getEstablishedSession() != null) {
                    record.setSession(connection.getEstablishedSession());
                } else {
                    LOGGER.log(Level.FINER, "Discarding unexpected handshake message with epoch > 0 from peer [{0}] with no session established.", new Object[]{peerAddress});
                    return;
                }
            }
            HandshakeMessage handshakeMessage = (HandshakeMessage)record.getFragment();
            switch (handshakeMessage.getMessageType()) {
                case HELLO_REQUEST: {
                    flight = this.processHelloRequest(connection, record);
                    break;
                }
                case CLIENT_HELLO: {
                    flight = this.processClientHello(connection, record);
                    break;
                }
                default: {
                    LOGGER.log(Level.FINER, "Discarding unexpected handshake message of type [{0}] from peer [{1}]", new Object[]{handshakeMessage.getMessageType(), peerAddress});
                    return;
                }
            }
        }
        if (flight != null) {
            connection = this.connectionStore.get(peerAddress);
            if (connection != null) {
                connection.cancelPendingFlight();
                if (flight.isRetransmissionNeeded()) {
                    connection.setPendingFlight(flight);
                    this.scheduleRetransmission(flight);
                }
            }
            this.sendFlight(flight);
        }
    }

    private DTLSFlight processHelloRequest(Connection connection, Record record) throws HandshakeException, GeneralSecurityException {
        if (connection == null) {
            LOGGER.log(Level.FINE, "Received HELLO_REQUEST from peer [{0}] without an existing connection, discarding ...", record.getPeerAddress());
            return null;
        }
        if (connection.getOngoingHandshake() == null) {
            DTLSSession session = connection.getEstablishedSession();
            if (session == null) {
                session = new DTLSSession(record.getPeerAddress(), true);
            }
            ClientHandshaker handshaker = new ClientHandshaker(null, session, (SessionListener)connection, this.config, this.maximumTransmissionUnit);
            return ((Handshaker)handshaker).getStartHandshakeMessage();
        }
        return null;
    }

    private DTLSFlight processClientHello(Connection connection, Record record) throws HandshakeException, GeneralSecurityException {
        DTLSFlight nextFlight = null;
        Connection peerConnection = connection;
        if (record.getEpoch() > 0) {
            if (peerConnection == null || peerConnection.getEstablishedSession() == null) {
                LOGGER.log(Level.FINE, "Ignoring request from [{0}] to re-negotiate non-existing session", record.getPeerAddress());
            } else {
                ServerHandshaker handshaker = new ServerHandshaker(connection.getEstablishedSession(), peerConnection, this.config, this.maximumTransmissionUnit);
                nextFlight = handshaker.processMessage(record);
            }
        } else {
            ClientHello clientHello = (ClientHello)record.getFragment();
            byte[] expectedCookie = this.generateCookie(record.getPeerAddress(), clientHello);
            if (!Arrays.equals(expectedCookie, clientHello.getCookie())) {
                LOGGER.log(Level.FINE, "Processing CLIENT_HELLO from peer [{0}]:\n{1}", new Object[]{record.getPeerAddress(), record});
                LOGGER.log(Level.FINER, "Verifying client IP address [{0}] using HELLO_VERIFY_REQUEST", record.getPeerAddress());
                HelloVerifyRequest msg = new HelloVerifyRequest(new ProtocolVersion(), expectedCookie, record.getPeerAddress());
                msg.setMessageSeq(clientHello.getMessageSeq());
                Record helloVerify = new Record(ContentType.HANDSHAKE, 0, record.getSequenceNumber(), (DTLSMessage)msg, record.getPeerAddress());
                nextFlight = new DTLSFlight(record.getPeerAddress());
                nextFlight.addMessage(helloVerify);
            } else {
                SessionId sessionId;
                LOGGER.log(Level.FINER, "Successfully verified client IP address [{0}] using cookie exchange", record.getPeerAddress());
                SessionId sessionId2 = sessionId = clientHello.getSessionId().length() > 0 ? clientHello.getSessionId() : null;
                if (sessionId == null) {
                    if (peerConnection != null) {
                        this.terminateConnection(connection, null, null);
                    }
                    peerConnection = new Connection(record.getPeerAddress());
                    this.connectionStore.put(peerConnection);
                    DTLSSession newSession = new DTLSSession(record.getPeerAddress(), false, record.getSequenceNumber());
                    ServerHandshaker handshaker = new ServerHandshaker(clientHello.getMessageSeq(), newSession, (SessionListener)peerConnection, this.config, this.maximumTransmissionUnit);
                    nextFlight = handshaker.processMessage(record);
                } else {
                    LOGGER.log(Level.FINER, "Client [{0}] wants to resume session with ID [{1}]", new Object[]{record.getPeerAddress(), ByteArrayUtils.toHexString(sessionId.getSessionId())});
                    final Connection previousConnection = this.connectionStore.find(sessionId);
                    if (previousConnection != null && previousConnection.getEstablishedSession() != null) {
                        DTLSSession resumableSession = new DTLSSession(record.getPeerAddress(), previousConnection.getEstablishedSession(), record.getSequenceNumber());
                        peerConnection = new Connection(record.getPeerAddress());
                        ResumingServerHandshaker handshaker = new ResumingServerHandshaker(clientHello.getMessageSeq(), resumableSession, (SessionListener)peerConnection, this.config, this.maximumTransmissionUnit);
                        if (!previousConnection.getPeerAddress().equals(peerConnection.getPeerAddress())) {
                            handshaker.addSessionListener(new SessionAdapter(){

                                @Override
                                public void sessionEstablished(Handshaker handshaker, DTLSSession establishedSession) throws HandshakeException {
                                    LOGGER.log(Level.FINER, "Remove old connection to [{0}], session with ID [{1}] was successfully resumed on peer [{2}] ", new Object[]{previousConnection.getPeerAddress(), ByteArrayUtils.toHexString(establishedSession.getSessionIdentifier().getSessionId()), establishedSession.getPeer()});
                                    DTLSConnector.this.terminateConnection(previousConnection, null, null);
                                }
                            });
                        } else {
                            this.terminateConnection(previousConnection, null, null);
                        }
                        this.connectionStore.put(peerConnection);
                        nextFlight = handshaker.processMessage(record);
                    } else {
                        LOGGER.log(Level.FINER, "Client [{0}] tries to resume non-existing session with ID [{1}], starting new handshake...", new Object[]{record.getPeerAddress(), ByteArrayUtils.toHexString(sessionId.getSessionId())});
                        if (peerConnection != null) {
                            this.terminateConnection(peerConnection, null, null);
                        }
                        peerConnection = new Connection(record.getPeerAddress());
                        this.connectionStore.put(peerConnection);
                        DTLSSession newSession = new DTLSSession(record.getPeerAddress(), false, record.getSequenceNumber());
                        ServerHandshaker handshaker = new ServerHandshaker(clientHello.getMessageSeq(), newSession, (SessionListener)peerConnection, this.config, this.maximumTransmissionUnit);
                        nextFlight = handshaker.processMessage(record);
                    }
                }
            }
        }
        return nextFlight;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private SecretKey getMacKeyForCookies() {
        Object object = this.cookieMacKeyLock;
        synchronized (object) {
            if (System.currentTimeMillis() - this.lastGenerationDate > TimeUnit.MINUTES.toMillis(5L)) {
                this.cookieMacKey = new SecretKeySpec(this.randomBytes(), "MAC");
                this.lastGenerationDate = System.currentTimeMillis();
            }
            return this.cookieMacKey;
        }
    }

    private byte[] generateCookie(InetSocketAddress peerAddress, ClientHello clientHello) throws HandshakeException {
        try {
            Mac hmac = Mac.getInstance("HmacSHA256");
            hmac.init(this.getMacKeyForCookies());
            hmac.update(peerAddress.toString().getBytes());
            hmac.update((byte)clientHello.getClientVersion().getMajor());
            hmac.update((byte)clientHello.getClientVersion().getMinor());
            hmac.update(clientHello.getRandom().getRandomBytes());
            hmac.update(clientHello.getSessionId().getSessionId());
            hmac.update(CipherSuite.listToByteArray(clientHello.getCipherSuites()));
            hmac.update(CompressionMethod.listToByteArray(clientHello.getCompressionMethods()));
            return hmac.doFinal();
        }
        catch (GeneralSecurityException e) {
            LOGGER.log(Level.SEVERE, "Could not instantiate MAC algorithm for cookie creation", e);
            throw new HandshakeException("Internal error", new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.INTERNAL_ERROR, peerAddress));
        }
    }

    void send(AlertMessage alert, DTLSSession session) {
        if (alert == null) {
            throw new IllegalArgumentException("Alert must not be NULL");
        }
        if (session == null) {
            throw new IllegalArgumentException("Session must not be NULL");
        }
        try {
            DTLSFlight flight = new DTLSFlight(session);
            flight.setRetransmissionNeeded(false);
            flight.addMessage(new Record(ContentType.ALERT, session.getWriteEpoch(), session.getSequenceNumber(), (DTLSMessage)alert, session));
            this.sendFlight(flight);
        }
        catch (GeneralSecurityException e) {
            LOGGER.log(Level.FINE, "Cannot create ALERT message for peer [{0}] due to [{1}]", new Object[]{session.getPeer(), e.getMessage()});
        }
    }

    public final void send(RawData msg) {
        if (msg == null) {
            LOGGER.finest("Ignoring NULL msg ...");
        } else {
            boolean queueFull;
            if (msg.getBytes().length > 16384) {
                throw new IllegalArgumentException("Message data must not exceed 16384 bytes");
            }
            boolean bl = queueFull = !this.outboundMessages.offer(msg);
            if (queueFull) {
                LOGGER.log(Level.WARNING, "Outbound message queue is full! Dropping outbound message to peer [{0}]", msg.getInetSocketAddress());
            }
        }
    }

    private void sendNextMessageOverNetwork() throws HandshakeException {
        RawData message;
        try {
            message = this.outboundMessages.take();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return;
        }
        InetSocketAddress peerAddress = message.getInetSocketAddress();
        LOGGER.log(Level.FINER, "Sending application layer message to peer [{0}]", peerAddress);
        Connection connection = this.connectionStore.get(peerAddress);
        ClientHandshaker handshaker = null;
        DTLSFlight flight = null;
        try {
            DTLSSession session;
            if (connection == null) {
                connection = new Connection(peerAddress);
                this.connectionStore.put(connection);
            }
            if ((session = connection.getEstablishedSession()) == null) {
                handshaker = new ClientHandshaker(message, new DTLSSession(peerAddress, true), (SessionListener)connection, this.config, this.maximumTransmissionUnit);
            } else if (connection.isResumptionRequired()) {
                DTLSSession resumableSession = new DTLSSession(peerAddress, session, 0L);
                Connection newConnection = new Connection(peerAddress);
                this.terminateConnection(connection, null, null);
                this.connectionStore.put(newConnection);
                handshaker = new ResumingClientHandshaker(message, resumableSession, (SessionListener)newConnection, this.config, this.maximumTransmissionUnit);
            } else {
                flight = new DTLSFlight(session);
                flight.addMessage(new Record(ContentType.APPLICATION_DATA, session.getWriteEpoch(), session.getSequenceNumber(), (DTLSMessage)new ApplicationMessage(message.getBytes(), peerAddress), session));
            }
            if (handshaker != null) {
                flight = ((Handshaker)handshaker).getStartHandshakeMessage();
                connection.setPendingFlight(flight);
                this.scheduleRetransmission(flight);
            }
            this.sendFlight(flight);
        }
        catch (GeneralSecurityException e) {
            LOGGER.log(Level.FINE, "Cannot send record to peer [{0}] due to [{1}]", new Object[]{peerAddress, e.getMessage()});
        }
    }

    public final DTLSSession getSessionByAddress(InetSocketAddress address) {
        if (address == null) {
            return null;
        }
        Connection connection = this.connectionStore.get(address);
        if (connection != null) {
            return connection.getEstablishedSession();
        }
        return null;
    }

    private void sendFlight(DTLSFlight flight) {
        byte[] payload = new byte[]{};
        int maxDatagramSize = this.maximumTransmissionUnit;
        if (flight.getSession() != null) {
            maxDatagramSize = flight.getSession().getMaxDatagramSize();
        }
        ArrayList<DatagramPacket> datagrams = new ArrayList<DatagramPacket>();
        try {
            for (Record record : flight.getMessages()) {
                byte[] recordBytes;
                if (flight.getTries() > 0) {
                    int epoch = record.getEpoch();
                    record.setSequenceNumber(flight.getSession().getSequenceNumber(epoch));
                }
                if ((recordBytes = record.toByteArray()).length > maxDatagramSize) {
                    LOGGER.log(Level.INFO, "{0} record of {1} bytes for peer [{2}] exceeds max. datagram size [{3}], discarding...", new Object[]{record.getType(), recordBytes.length, record.getPeerAddress(), maxDatagramSize});
                    continue;
                }
                LOGGER.log(Level.FINEST, "Sending record of {2} bytes to peer [{0}]:\n{1}", new Object[]{flight.getPeerAddress(), record, recordBytes.length});
                if (payload.length + recordBytes.length > maxDatagramSize) {
                    DatagramPacket datagram = new DatagramPacket(payload, payload.length, flight.getPeerAddress().getAddress(), flight.getPeerAddress().getPort());
                    datagrams.add(datagram);
                    payload = new byte[]{};
                }
                payload = ByteArrayUtils.concatenate(payload, recordBytes);
            }
            DatagramPacket datagram = new DatagramPacket(payload, payload.length, flight.getPeerAddress().getAddress(), flight.getPeerAddress().getPort());
            datagrams.add(datagram);
            LOGGER.log(Level.FINER, "Sending flight of {0} message(s) to peer [{1}] using {2} datagram(s) of max. {3} bytes", new Object[]{flight.getMessages().size(), flight.getPeerAddress(), datagrams.size(), maxDatagramSize});
            for (DatagramPacket datagramPacket : datagrams) {
                if (!this.socket.isClosed()) {
                    this.socket.send(datagramPacket);
                    continue;
                }
                LOGGER.log(Level.FINE, "Socket [{0}] is closed, discarding packet ...", this.config.getAddress());
            }
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Could not send datagram", e);
        }
        catch (GeneralSecurityException e) {
            LOGGER.log(Level.INFO, "Cannot send flight to peer [{0}] due to [{1}]", new Object[]{flight.getPeerAddress(), e.getMessage()});
        }
    }

    private void handleTimeout(DTLSFlight flight) {
        int max = this.config.getMaxRetransmissions();
        if (flight.getTries() < max) {
            LOGGER.log(Level.FINE, "Re-transmitting flight for [{0}], [{1}] retransmissions left", new Object[]{flight.getPeerAddress(), max - flight.getTries() - 1});
            flight.incrementTries();
            this.sendFlight(flight);
            this.scheduleRetransmission(flight);
        } else {
            LOGGER.log(Level.FINE, "Flight for [{0}] has reached maximum no. [{1}] of retransmissions", new Object[]{flight.getPeerAddress(), max});
        }
    }

    private void scheduleRetransmission(DTLSFlight flight) {
        if (flight.getRetransmitTask() != null) {
            flight.getRetransmitTask().cancel();
        }
        if (flight.isRetransmissionNeeded()) {
            flight.setRetransmitTask(new RetransmitTask(flight));
            if (flight.getTimeout() == 0) {
                flight.setTimeout(this.config.getRetransmissionTimeout());
            } else {
                flight.incrementTimeout();
            }
            this.timer.schedule(flight.getRetransmitTask(), flight.getTimeout());
        }
    }

    public int getMaximumTransmissionUnit() {
        return this.maximumTransmissionUnit;
    }

    public int getMaximumFragmentLength(InetSocketAddress peer) {
        Connection con = this.connectionStore.get(peer);
        if (con != null && con.getEstablishedSession() != null) {
            return con.getEstablishedSession().getMaxFragmentLength();
        }
        return this.maximumTransmissionUnit - 53;
    }

    public final InetSocketAddress getAddress() {
        if (this.socket == null) {
            return this.config.getAddress();
        }
        return new InetSocketAddress(this.socket.getLocalAddress(), this.socket.getLocalPort());
    }

    public final boolean isRunning() {
        return this.running;
    }

    public void setRawDataReceiver(RawDataChannel messageHandler) {
        this.messageHandler = messageHandler;
    }

    public final void setErrorHandler(ErrorHandler errorHandler) {
        this.errorHandler = errorHandler;
    }

    private void connectionClosed(InetSocketAddress peerAddress) {
        if (peerAddress != null) {
            this.connectionStore.remove(peerAddress);
        }
    }

    private byte[] randomBytes() {
        SecureRandom rng = new SecureRandom();
        byte[] result = new byte[32];
        rng.nextBytes(result);
        return result;
    }

    private abstract class Worker
    extends Thread {
        private Worker(String name) {
            super(name);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                LOGGER.log(Level.CONFIG, "Starting worker thread [{0}]", this.getName());
                while (DTLSConnector.this.running) {
                    try {
                        this.doWork();
                    }
                    catch (ClosedByInterruptException e) {
                        LOGGER.log(Level.CONFIG, "Worker thread [{0}] has been interrupted", this.getName());
                    }
                    catch (Exception e) {
                        if (!DTLSConnector.this.running) continue;
                        LOGGER.log(Level.FINE, "Exception thrown by worker thread [" + this.getName() + "]", e);
                    }
                }
            }
            finally {
                LOGGER.log(Level.CONFIG, "Worker thread [{0}] has terminated", this.getName());
            }
        }

        protected abstract void doWork() throws Exception;
    }

    private class RetransmitTask
    extends TimerTask {
        private DTLSFlight flight;

        RetransmitTask(DTLSFlight flight) {
            this.flight = flight;
        }

        @Override
        public void run() {
            DTLSConnector.this.handleTimeout(this.flight);
        }
    }
}

