/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.security.GeneralSecurityException;
import java.security.Key;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.crypto.Cipher;
import javax.crypto.Mac;
import javax.crypto.spec.IvParameterSpec;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.ApplicationMessage;
import org.eclipse.californium.scandium.dtls.ChangeCipherSpecMessage;
import org.eclipse.californium.scandium.dtls.ContentType;
import org.eclipse.californium.scandium.dtls.DTLSConnectionState;
import org.eclipse.californium.scandium.dtls.DTLSMessage;
import org.eclipse.californium.scandium.dtls.DTLSSession;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.ProtocolVersion;
import org.eclipse.californium.scandium.dtls.cipher.CCMBlockCipher;
import org.eclipse.californium.scandium.dtls.cipher.CipherSuite;
import org.eclipse.californium.scandium.dtls.cipher.InvalidMacException;
import org.eclipse.californium.scandium.util.ByteArrayUtils;
import org.eclipse.californium.scandium.util.DatagramReader;
import org.eclipse.californium.scandium.util.DatagramWriter;

public class Record {
    protected static final Logger LOGGER = Logger.getLogger(Record.class.getCanonicalName());
    private static final int CONTENT_TYPE_BITS = 8;
    private static final int VERSION_BITS = 8;
    private static final int EPOCH_BITS = 16;
    private static final int SEQUENCE_NUMBER_BITS = 48;
    private static final int LENGTH_BITS = 16;
    private static final long MAX_SEQUENCE_NO = 0xFFFFFFFFFFFFL;
    private ContentType type = null;
    private ProtocolVersion version = new ProtocolVersion();
    private int epoch = -1;
    private long sequenceNumber;
    private int length = 0;
    private DTLSMessage fragment = null;
    private byte[] fragmentBytes = null;
    private DTLSSession session;

    public Record(ContentType type, ProtocolVersion version, int epoch, long sequenceNumber, byte[] fragmentBytes) {
        this.type = type;
        this.version = version;
        this.epoch = epoch;
        this.sequenceNumber = sequenceNumber;
        this.fragmentBytes = fragmentBytes;
        this.length = fragmentBytes.length;
    }

    public Record(ContentType type, int epoch, long sequenceNumber, DTLSMessage fragment, DTLSSession session) throws IllegalArgumentException, GeneralSecurityException {
        if (sequenceNumber > 0xFFFFFFFFFFFFL) {
            throw new IllegalArgumentException("Sequence number must be 48 bits only");
        }
        this.type = type;
        this.epoch = epoch;
        this.sequenceNumber = sequenceNumber;
        this.session = session;
        this.setFragment(fragment);
    }

    public synchronized byte[] toByteArray() {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.type.getCode(), 8);
        writer.write(this.version.getMajor(), 8);
        writer.write(this.version.getMinor(), 8);
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        this.length = this.fragmentBytes.length;
        writer.write(this.length, 16);
        writer.writeBytes(this.fragmentBytes);
        return writer.toByteArray();
    }

    public static List<Record> fromByteArray(byte[] byteArray) {
        ArrayList<Record> records = new ArrayList<Record>();
        DatagramReader reader = new DatagramReader(byteArray);
        while (reader.bytesAvailable()) {
            int type = reader.read(8);
            int major = reader.read(8);
            int minor = reader.read(8);
            ProtocolVersion version = new ProtocolVersion(major, minor);
            int epoch = reader.read(16);
            long sequenceNumber = reader.readLong(48);
            int length = reader.read(16);
            byte[] fragmentBytes = reader.readBytes(length);
            ContentType contentType = ContentType.getTypeByValue(type);
            if (contentType == null) {
                LOGGER.log(Level.FINE, "Received DTLS record of unsupported type [{0}]. Discarding ...", type);
                continue;
            }
            records.add(new Record(contentType, version, epoch, sequenceNumber, fragmentBytes));
        }
        return records;
    }

    private byte[] encryptFragment(byte[] plaintextFragment) throws GeneralSecurityException {
        if (this.session == null) {
            return plaintextFragment;
        }
        byte[] encryptedFragment = plaintextFragment;
        CipherSuite cipherSuite = this.session.getWriteState().getCipherSuite();
        LOGGER.log(Level.FINER, "Encrypting record fragment using current write state\n{0}", this.session.getWriteState());
        switch (cipherSuite.getCipherType()) {
            case NULL: {
                break;
            }
            case AEAD: {
                encryptedFragment = this.encryptAEAD(plaintextFragment);
                break;
            }
            case BLOCK: {
                encryptedFragment = this.encryptBlockCipher(plaintextFragment);
                break;
            }
            case STREAM: {
                break;
            }
        }
        return encryptedFragment;
    }

    private byte[] decryptFragment(byte[] ciphertextFragment) throws GeneralSecurityException {
        if (this.session == null) {
            return ciphertextFragment;
        }
        byte[] fragment = ciphertextFragment;
        CipherSuite cipherSuite = this.session.getReadState().getCipherSuite();
        LOGGER.log(Level.FINER, "Decrypting record fragment using current read state\n{0}", this.session.getReadState());
        switch (cipherSuite.getCipherType()) {
            case NULL: {
                break;
            }
            case AEAD: {
                fragment = this.decryptAEAD(ciphertextFragment);
                break;
            }
            case BLOCK: {
                fragment = this.decryptBlockCipher(ciphertextFragment);
                break;
            }
            case STREAM: {
                break;
            }
        }
        return fragment;
    }

    protected final synchronized byte[] encryptBlockCipher(byte[] compressedFragment) throws GeneralSecurityException {
        int smallestMultipleOfBlocksize;
        if (this.session == null) {
            throw new IllegalStateException("DTLS session must be set on record");
        }
        if (compressedFragment == null) {
            throw new NullPointerException("Compressed fragment must not be null");
        }
        DatagramWriter plaintext = new DatagramWriter();
        plaintext.writeBytes(compressedFragment);
        plaintext.writeBytes(this.getBlockCipherMac(this.session.getWriteState(), compressedFragment));
        int length = compressedFragment.length + this.session.getWriteState().getCipherSuite().getMacLength() + 1;
        for (smallestMultipleOfBlocksize = this.session.getWriteState().getRecordIvLength(); smallestMultipleOfBlocksize <= length; smallestMultipleOfBlocksize += this.session.getWriteState().getRecordIvLength()) {
        }
        int paddingLength = smallestMultipleOfBlocksize % length;
        byte[] padding = new byte[paddingLength + 1];
        Arrays.fill(padding, (byte)paddingLength);
        plaintext.writeBytes(padding);
        Cipher blockCipher = Cipher.getInstance(this.session.getWriteState().getCipherSuite().getTransformation());
        blockCipher.init(1, this.session.getWriteState().getEncryptionKey());
        DatagramWriter result = new DatagramWriter();
        result.writeBytes(blockCipher.getIV());
        result.writeBytes(blockCipher.doFinal(plaintext.toByteArray()));
        return result.toByteArray();
    }

    protected final byte[] decryptBlockCipher(byte[] ciphertextFragment) throws GeneralSecurityException {
        if (this.session == null) {
            throw new IllegalStateException("DTLS session must be set on record");
        }
        if (ciphertextFragment == null) {
            throw new NullPointerException("Ciphertext must not be null");
        }
        DatagramReader reader = new DatagramReader(ciphertextFragment);
        byte[] iv = reader.readBytes(this.session.getReadState().getRecordIvLength());
        Cipher blockCipher = Cipher.getInstance(this.session.getReadState().getCipherSuite().getTransformation());
        blockCipher.init(2, (Key)this.session.getReadState().getEncryptionKey(), new IvParameterSpec(iv));
        byte[] plaintext = blockCipher.doFinal(reader.readBytesLeft());
        byte paddingLength = plaintext[plaintext.length - 1];
        int fragmentLength = plaintext.length - 1 - paddingLength - this.session.getReadState().getCipherSuite().getMacLength();
        reader = new DatagramReader(plaintext);
        byte[] content = reader.readBytes(fragmentLength);
        byte[] macFromMessage = reader.readBytes(this.session.getReadState().getCipherSuite().getMacLength());
        byte[] mac = this.getBlockCipherMac(this.session.getReadState(), content);
        if (Arrays.equals(macFromMessage, mac)) {
            return content;
        }
        throw new InvalidMacException(mac, macFromMessage);
    }

    private byte[] getBlockCipherMac(DTLSConnectionState conState, byte[] content) throws GeneralSecurityException {
        Mac hmac = Mac.getInstance(conState.getCipherSuite().getMacName());
        hmac.init(conState.getMacKey());
        DatagramWriter mac = new DatagramWriter();
        mac.writeBytes(this.generateAdditionalData(content.length));
        mac.writeBytes(content);
        return hmac.doFinal(mac.toByteArray());
    }

    protected byte[] encryptAEAD(byte[] byteArray) throws GeneralSecurityException {
        byte[] iv = this.session.getWriteState().getIv().getIV();
        byte[] nonce = this.generateNonce(iv);
        byte[] key = this.session.getWriteState().getEncryptionKey().getEncoded();
        byte[] additionalData = this.generateAdditionalData(byteArray.length);
        byte[] encryptedFragment = CCMBlockCipher.encrypt(key, nonce, additionalData, byteArray, 8);
        byte[] explicitNonce = this.generateExplicitNonce();
        encryptedFragment = ByteArrayUtils.concatenate(explicitNonce, encryptedFragment);
        return encryptedFragment;
    }

    protected byte[] decryptAEAD(byte[] byteArray) throws GeneralSecurityException {
        byte[] explicitNonceUsed;
        byte[] iv = this.session.getReadState().getIv().getIV();
        byte[] key = this.session.getReadState().getEncryptionKey().getEncoded();
        byte[] additionalData = this.generateAdditionalData(byteArray.length - 16);
        DatagramReader reader = new DatagramReader(byteArray);
        byte[] explicitNonce = this.generateExplicitNonce();
        if (!Arrays.equals(explicitNonce, explicitNonceUsed = reader.readBytes(8)) && LOGGER.isLoggable(Level.FINE)) {
            StringBuffer b = new StringBuffer("The explicit nonce used by the sender does not match the values provided in the DTLS record");
            b.append("\nUsed    : ").append(ByteArrayUtils.toHexString(explicitNonceUsed));
            b.append("\nExpected: ").append(ByteArrayUtils.toHexString(explicitNonce));
            LOGGER.log(Level.FINE, b.toString());
        }
        byte[] nonce = this.getNonce(iv, explicitNonceUsed);
        byte[] decrypted = CCMBlockCipher.decrypt(key, nonce, additionalData, reader.readBytesLeft(), 8);
        return decrypted;
    }

    private byte[] generateNonce(byte[] iv) {
        return this.getNonce(iv, this.generateExplicitNonce());
    }

    private byte[] getNonce(byte[] implicitNonce, byte[] explicitNonce) {
        DatagramWriter writer = new DatagramWriter();
        writer.writeBytes(implicitNonce);
        writer.writeBytes(explicitNonce);
        return writer.toByteArray();
    }

    private byte[] generateExplicitNonce() {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        return writer.toByteArray();
    }

    private byte[] generateAdditionalData(int length) {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.epoch, 16);
        writer.writeLong(this.sequenceNumber, 48);
        writer.write(this.type.getCode(), 8);
        writer.write(this.version.getMajor(), 8);
        writer.write(this.version.getMinor(), 8);
        writer.write(length, 16);
        return writer.toByteArray();
    }

    public ContentType getType() {
        return this.type;
    }

    public ProtocolVersion getVersion() {
        return this.version;
    }

    public int getEpoch() {
        return this.epoch;
    }

    public long getSequenceNumber() {
        return this.sequenceNumber;
    }

    public synchronized void setSequenceNumber(long sequenceNumber) throws GeneralSecurityException {
        if (sequenceNumber > 0xFFFFFFFFFFFFL) {
            throw new IllegalArgumentException("Sequence number must have max 48 bits");
        }
        this.sequenceNumber = sequenceNumber;
        if (this.session != null && this.session.getWriteState() != null && CipherSuite.CipherType.BLOCK.equals((Object)this.session.getWriteState().getCipherSuite().getCipherType())) {
            this.fragmentBytes = this.encryptBlockCipher(this.fragment.toByteArray());
        }
    }

    public int getLength() {
        return this.length;
    }

    public synchronized void setSession(DTLSSession session) {
        this.session = session;
    }

    public synchronized DTLSMessage getFragment() throws GeneralSecurityException, HandshakeException {
        if (this.fragment == null) {
            switch (this.type) {
                case ALERT: {
                    byte[] decryptedMessage = this.decryptFragment(this.fragmentBytes);
                    if (decryptedMessage == null) break;
                    this.fragment = AlertMessage.fromByteArray(decryptedMessage);
                    break;
                }
                case APPLICATION_DATA: {
                    byte[] decryptedMessage = this.decryptFragment(this.fragmentBytes);
                    if (decryptedMessage == null) break;
                    this.fragment = ApplicationMessage.fromByteArray(decryptedMessage);
                    break;
                }
                case CHANGE_CIPHER_SPEC: {
                    byte[] decryptedMessage = this.decryptFragment(this.fragmentBytes);
                    if (decryptedMessage == null) break;
                    this.fragment = ChangeCipherSpecMessage.fromByteArray(decryptedMessage);
                    break;
                }
                case HANDSHAKE: {
                    if (LOGGER.isLoggable(Level.FINEST)) {
                        LOGGER.log(Level.FINEST, "Decrypting HANDSHAKE message ciphertext\n{0}", ByteArrayUtils.toHexString(this.fragmentBytes));
                    }
                    byte[] decryptedMessage = this.decryptFragment(this.fragmentBytes);
                    CipherSuite.KeyExchangeAlgorithm keyExchangeAlgorithm = CipherSuite.KeyExchangeAlgorithm.NULL;
                    boolean receiveRawPublicKey = false;
                    if (this.session != null) {
                        keyExchangeAlgorithm = this.session.getKeyExchange();
                        receiveRawPublicKey = this.session.receiveRawPublicKey();
                        LOGGER.log(Level.FINEST, "Using KeyExchange [{0}] and receiveRawPublicKey [{1}] from session", new Object[]{keyExchangeAlgorithm, receiveRawPublicKey});
                    }
                    if (decryptedMessage == null) break;
                    if (LOGGER.isLoggable(Level.FINEST)) {
                        LOGGER.log(Level.FINEST, "Parsing HANDSHAKE message plaintext using KeyExchange [{0}] and receiveRawPublicKey [{1}]\n{2}", new Object[]{keyExchangeAlgorithm, receiveRawPublicKey, ByteArrayUtils.toHexString(decryptedMessage)});
                    }
                    this.fragment = HandshakeMessage.fromByteArray(decryptedMessage, keyExchangeAlgorithm, receiveRawPublicKey);
                }
            }
        }
        return this.fragment;
    }

    public synchronized void setFragment(DTLSMessage fragment) throws GeneralSecurityException {
        if (this.fragmentBytes == null) {
            byte[] byteArray = fragment.toByteArray();
            this.length = byteArray.length;
            switch (this.type) {
                case ALERT: 
                case APPLICATION_DATA: 
                case CHANGE_CIPHER_SPEC: 
                case HANDSHAKE: {
                    byteArray = this.encryptFragment(byteArray);
                    break;
                }
                default: {
                    LOGGER.severe("Unknown content type: " + this.type.toString());
                }
            }
            this.fragmentBytes = byteArray;
        }
        this.fragment = fragment;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("==[ DTLS Record ]==============================================");
        sb.append("\nContent Type: ").append(this.type.toString());
        sb.append("\nVersion: ").append(this.version.getMajor()).append(", ").append(this.version.getMinor());
        sb.append("\nEpoch: ").append(this.epoch);
        sb.append("\nSequence Number: ").append(this.sequenceNumber);
        sb.append("\nLength: ").append(this.length);
        sb.append("\nFragment:");
        if (this.fragment != null) {
            sb.append("\n").append(this.fragment);
        } else {
            sb.append("\nfragment is not decrypted yet\n");
        }
        sb.append("\n===============================================================");
        return sb.toString();
    }
}

