/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.elements.util;

import java.util.AbstractCollection;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import org.eclipse.californium.elements.util.ClockUtil;

public class LeastRecentlyUsedCache<K, V> {
    public static final int DEFAULT_INITIAL_CAPACITY = 16;
    public static final long DEFAULT_THRESHOLD_SECS = 1800L;
    public static final int DEFAULT_CAPACITY = 150000;
    private Collection<V> values;
    private final Map<K, CacheEntry<K, V>> cache;
    private volatile int capacity;
    private CacheEntry<K, V> header;
    private volatile long expirationThresholdNanos;
    private volatile boolean evictOnReadAccess = true;
    private volatile boolean updateOnReadAccess = true;
    private final List<EvictionListener<V>> evictionListeners = new LinkedList<EvictionListener<V>>();

    public LeastRecentlyUsedCache() {
        this(16, 150000, 1800L);
    }

    public LeastRecentlyUsedCache(int capacity, long threshold) {
        this(Math.min(capacity, 16), capacity, threshold);
    }

    public LeastRecentlyUsedCache(int initialCapacity, int maxCapacity, long threshold) {
        if (initialCapacity > maxCapacity) {
            throw new IllegalArgumentException("initial capacity must be <= max capacity");
        }
        this.capacity = maxCapacity;
        this.cache = new ConcurrentHashMap<K, CacheEntry<K, V>>(initialCapacity);
        this.setExpirationThreshold(threshold);
        this.initLinkedList();
    }

    private void initLinkedList() {
        this.header = new CacheEntry();
        ((CacheEntry)this.header).after = (((CacheEntry)this.header).before = (CacheEntry)this.header);
    }

    public void addEvictionListener(EvictionListener<V> listener) {
        if (listener != null) {
            this.evictionListeners.add(listener);
        }
    }

    public boolean isEvictingOnReadAccess() {
        return this.evictOnReadAccess;
    }

    public void setEvictingOnReadAccess(boolean evict) {
        this.evictOnReadAccess = evict;
    }

    public boolean isUpdatingOnReadAccess() {
        return this.updateOnReadAccess;
    }

    public void setUpdatingOnReadAccess(boolean update) {
        this.updateOnReadAccess = update;
    }

    public final long getExpirationThreshold() {
        return TimeUnit.NANOSECONDS.toSeconds(this.expirationThresholdNanos);
    }

    public final void setExpirationThreshold(long newThreshold) {
        this.setExpirationThreshold(newThreshold, TimeUnit.SECONDS);
    }

    public final void setExpirationThreshold(long newThreshold, TimeUnit unit) {
        this.expirationThresholdNanos = unit.toNanos(newThreshold);
    }

    public final int getCapacity() {
        return this.capacity;
    }

    public final void setCapacity(int capacity) {
        this.capacity = capacity;
    }

    public final int size() {
        return this.cache.size();
    }

    public final int remainingCapacity() {
        return Math.max(0, this.capacity - this.cache.size());
    }

    public final void clear() {
        this.cache.clear();
        this.initLinkedList();
    }

    public final boolean put(K key, V value) {
        if (value != null) {
            CacheEntry<K, V> existingEntry = this.cache.get(key);
            if (existingEntry != null) {
                ((CacheEntry)existingEntry).remove();
                this.add(key, value);
                return true;
            }
            if (this.cache.size() < this.capacity) {
                this.add(key, value);
                return true;
            }
            CacheEntry eldest = ((CacheEntry)this.header).after;
            if (eldest.isStale(this.expirationThresholdNanos)) {
                eldest.remove();
                this.cache.remove(eldest.getKey());
                this.add(key, value);
                this.notifyEvictionListeners(eldest.getValue());
                return true;
            }
        }
        return false;
    }

    private void notifyEvictionListeners(V session) {
        for (EvictionListener<V> listener : this.evictionListeners) {
            listener.onEviction(session);
        }
    }

    final V getEldest() {
        CacheEntry eldest = ((CacheEntry)this.header).after;
        return (V)eldest.getValue();
    }

    private void add(K key, V value) {
        CacheEntry entry = new CacheEntry(key, value);
        this.cache.put(key, entry);
        entry.addBefore((CacheEntry)this.header);
    }

    public final V get(K key) {
        if (key == null) {
            return null;
        }
        CacheEntry<K, V> entry = this.cache.get(key);
        if (entry == null) {
            return null;
        }
        return this.access(entry, null);
    }

    private final V access(CacheEntry<K, V> entry, Iterator<CacheEntry<K, V>> iterator) {
        if (this.evictOnReadAccess && this.expirationThresholdNanos > 0L && ((CacheEntry)entry).isStale(this.expirationThresholdNanos)) {
            if (iterator != null) {
                iterator.remove();
            } else {
                this.cache.remove(((CacheEntry)entry).getKey());
            }
            ((CacheEntry)entry).remove();
            this.notifyEvictionListeners(((CacheEntry)entry).getValue());
            return null;
        }
        if (this.updateOnReadAccess) {
            ((CacheEntry)entry).recordAccess((CacheEntry)this.header);
        }
        return (V)((CacheEntry)entry).getValue();
    }

    public final boolean update(K key) {
        if (key == null) {
            return false;
        }
        CacheEntry<K, V> entry = this.cache.get(key);
        if (entry == null) {
            return false;
        }
        ((CacheEntry)entry).recordAccess((CacheEntry)this.header);
        return true;
    }

    public final V remove(K key) {
        if (key == null) {
            return null;
        }
        CacheEntry<K, V> entry = this.cache.remove(key);
        if (entry != null) {
            ((CacheEntry)entry).remove();
            return (V)((CacheEntry)entry).getValue();
        }
        return null;
    }

    public final V remove(K key, V value) {
        if (key == null) {
            return null;
        }
        CacheEntry<K, V> entry = this.cache.get(key);
        if (entry != null && ((CacheEntry)entry).getValue() == value) {
            this.cache.remove(key);
            ((CacheEntry)entry).remove();
            return value;
        }
        return null;
    }

    public final V find(Predicate<V> predicate) {
        return this.find(predicate, true);
    }

    public final V find(Predicate<V> predicate, boolean unique) {
        if (predicate != null) {
            Iterator<CacheEntry<K, V>> iterator = this.cache.values().iterator();
            while (iterator.hasNext()) {
                CacheEntry<K, V> entry = iterator.next();
                if (!predicate.accept(((CacheEntry)entry).getValue())) continue;
                V value = this.access(entry, iterator);
                if (!unique && value == null) continue;
                return value;
            }
        }
        return null;
    }

    public final Iterator<V> valuesIterator() {
        final Iterator<CacheEntry<K, V>> iterator = this.cache.values().iterator();
        return new Iterator<V>(){
            private boolean hasNextCalled;
            private CacheEntry<K, V> nextEntry;

            @Override
            public boolean hasNext() {
                if (!this.hasNextCalled) {
                    this.nextEntry = null;
                    while (iterator.hasNext()) {
                        CacheEntry entry = (CacheEntry)iterator.next();
                        if (LeastRecentlyUsedCache.this.access(entry, iterator) == null) continue;
                        this.nextEntry = entry;
                        break;
                    }
                    this.hasNextCalled = true;
                }
                return this.nextEntry != null;
            }

            @Override
            public V next() {
                this.hasNext();
                this.hasNextCalled = false;
                if (this.nextEntry == null) {
                    throw new NoSuchElementException();
                }
                return this.nextEntry.value;
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    public final Collection<V> values() {
        AbstractCollection vs = this.values;
        if (vs == null) {
            this.values = vs = new AbstractCollection<V>(){

                @Override
                public final int size() {
                    return LeastRecentlyUsedCache.this.cache.size();
                }

                @Override
                public final boolean contains(final Object o) {
                    return null != LeastRecentlyUsedCache.this.find(new Predicate<V>(){

                        @Override
                        public boolean accept(V value) {
                            return value.equals(o);
                        }
                    }, false);
                }

                @Override
                public final Iterator<V> iterator() {
                    return LeastRecentlyUsedCache.this.valuesIterator();
                }

                @Override
                public final boolean add(Object o) {
                    throw new UnsupportedOperationException();
                }

                @Override
                public final boolean remove(Object o) {
                    throw new UnsupportedOperationException();
                }

                @Override
                public final void clear() {
                    throw new UnsupportedOperationException();
                }
            };
        }
        return vs;
    }

    private static class CacheEntry<K, V> {
        private final K key;
        private final V value;
        private long lastUpdate;
        private CacheEntry<K, V> after;
        private CacheEntry<K, V> before;

        private CacheEntry() {
            this.key = null;
            this.value = null;
            this.lastUpdate = -1L;
        }

        private CacheEntry(K key, V value) {
            this.value = value;
            this.key = key;
            this.lastUpdate = ClockUtil.nanoRealtime();
        }

        private K getKey() {
            return this.key;
        }

        private V getValue() {
            return this.value;
        }

        private boolean isStale(long thresholdNanos) {
            return ClockUtil.nanoRealtime() - this.lastUpdate >= thresholdNanos;
        }

        private void recordAccess(CacheEntry<K, V> header) {
            this.remove();
            this.lastUpdate = ClockUtil.nanoRealtime();
            this.addBefore(header);
        }

        private void addBefore(CacheEntry<K, V> existingEntry) {
            this.after = existingEntry;
            this.before = existingEntry.before;
            this.before.after = this;
            this.after.before = this;
        }

        private void remove() {
            this.before.after = this.after;
            this.after.before = this.before;
        }

        public String toString() {
            return "CacheEntry [key: " + this.key + ", last access: " + this.lastUpdate + "]";
        }
    }

    public static interface EvictionListener<V> {
        public void onEviction(V var1);
    }

    public static interface Predicate<V> {
        public boolean accept(V var1);
    }
}

