/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls.cipher;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.ECGenParameterSpec;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.EllipticCurve;
import java.security.spec.KeySpec;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.crypto.KeyAgreement;
import javax.crypto.SecretKey;
import javax.security.auth.Destroyable;
import org.eclipse.californium.elements.util.Asn1DerDecoder;
import org.eclipse.californium.elements.util.Bytes;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.dtls.cipher.RandomManager;
import org.eclipse.californium.scandium.dtls.cipher.ThreadLocalKeyAgreement;
import org.eclipse.californium.scandium.dtls.cipher.ThreadLocalKeyFactory;
import org.eclipse.californium.scandium.dtls.cipher.ThreadLocalKeyPairGenerator;
import org.eclipse.californium.scandium.util.SecretUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class XECDHECryptography
implements Destroyable {
    protected static final Logger LOGGER = LoggerFactory.getLogger(XECDHECryptography.class);
    private static final String EC_KEYPAIR_GENERATOR_ALGORITHM = "EC";
    private static final ThreadLocalKeyPairGenerator EC_KEYPAIR_GENERATOR = new ThreadLocalKeyPairGenerator("EC");
    private static final String XDH_KEYPAIR_GENERATOR_ALGORITHM = "XDH";
    private static final ThreadLocalKeyPairGenerator XDH_KEYPAIR_GENERATOR = new ThreadLocalKeyPairGenerator("XDH");
    private static final String EC_KEY_FACTORY_ALGORITHM = "EC";
    private static final ThreadLocalKeyFactory EC_KEY_FACTORY = new ThreadLocalKeyFactory("EC");
    private static final String XDH_KEY_FACTORY_ALGORITHM = "XDH";
    private static final ThreadLocalKeyFactory XDH_KEY_FACTORY = new ThreadLocalKeyFactory("XDH");
    private static final String ECDH_KEY_AGREEMENT_ALGORITHM = "ECDH";
    private static final ThreadLocalKeyAgreement ECDH_KEY_AGREEMENT = new ThreadLocalKeyAgreement("ECDH");
    private static final String XDH_KEY_AGREEMENT_ALGORITHM = "XDH";
    private static final ThreadLocalKeyAgreement XDH_KEY_AGREEMENT = new ThreadLocalKeyAgreement("XDH");
    private static final Class<?> XECPublicKeyClass;
    private static final Method XECPublicKeyGetU;
    private static final Method XECPublicKeyGetParams;
    private static final Method NamedParameterSpecGetName;
    private static final Constructor<?> XECPublicKeySpecInit;
    private static final Map<Integer, SupportedGroup> EC_CURVE_MAP_BY_ID;
    private static final Map<EllipticCurve, SupportedGroup> EC_CURVE_MAP_BY_CURVE;
    private final SupportedGroup supportedGroup;
    private PrivateKey privateKey;
    private PublicKey publicKey;
    private byte[] encodedPoint;

    public XECDHECryptography(SupportedGroup supportedGroup) throws GeneralSecurityException {
        KeyPair keyPair;
        if (supportedGroup.getAlgorithmName().equals("EC")) {
            KeyPairGenerator keyPairGenerator = (KeyPairGenerator)EC_KEYPAIR_GENERATOR.currentWithCause();
            ECGenParameterSpec params = new ECGenParameterSpec(supportedGroup.name());
            keyPairGenerator.initialize(params, RandomManager.currentSecureRandom());
            keyPair = keyPairGenerator.generateKeyPair();
        } else if (supportedGroup.getAlgorithmName().equals("XDH")) {
            KeyPairGenerator keyPairGenerator = (KeyPairGenerator)XDH_KEYPAIR_GENERATOR.currentWithCause();
            ECGenParameterSpec params = new ECGenParameterSpec(supportedGroup.name());
            keyPairGenerator.initialize(params, RandomManager.currentSecureRandom());
            keyPair = keyPairGenerator.generateKeyPair();
        } else {
            throw new GeneralSecurityException(supportedGroup.name() + " not supported by KeyPairGenerator!");
        }
        this.privateKey = keyPair.getPrivate();
        this.publicKey = keyPair.getPublic();
        this.supportedGroup = supportedGroup;
        this.encodedPoint = this.encodedPoint(keyPair.getPublic());
    }

    PublicKey getPublicKey() {
        return this.publicKey;
    }

    public SupportedGroup getSupportedGroup() {
        return this.supportedGroup;
    }

    public byte[] getEncodedPoint() {
        return this.encodedPoint;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public SecretKey generateSecret(byte[] encodedPoint) throws GeneralSecurityException {
        PublicKey peerPublicKey;
        if (encodedPoint == null) {
            throw new NullPointerException("encoded point must not be null!");
        }
        if (this.privateKey == null) {
            throw new IllegalStateException("private key must not be destroyed");
        }
        int keySize = this.supportedGroup.getKeySizeInBytes();
        if (this.supportedGroup.getAlgorithmName().equals("EC")) {
            int left = encodedPoint.length - 1;
            if (encodedPoint[0] != 4 || left % 2 != 0 || left / 2 != keySize) throw new GeneralSecurityException("DHE: failed to decoded point! " + this.supportedGroup.name());
            left /= 2;
            byte[] encoded = new byte[keySize];
            System.arraycopy(encodedPoint, 1, encoded, 0, keySize);
            BigInteger x = new BigInteger(1, encoded);
            System.arraycopy(encodedPoint, 1 + keySize, encoded, 0, keySize);
            BigInteger y = new BigInteger(1, encoded);
            ECParameterSpec ecParams = ((ECPrivateKey)this.privateKey).getParams();
            ECPublicKeySpec publicKeySpec = new ECPublicKeySpec(new ECPoint(x, y), ecParams);
            KeyFactory keyFactory = (KeyFactory)EC_KEY_FACTORY.currentWithCause();
            peerPublicKey = keyFactory.generatePublic(publicKeySpec);
        } else {
            BigInteger u = new BigInteger(1, XECDHECryptography.revert(encodedPoint, keySize));
            KeySpec spec = this.getXECPublicKeySpec(this.supportedGroup.name(), u);
            KeyFactory keyFactory = (KeyFactory)XDH_KEY_FACTORY.currentWithCause();
            peerPublicKey = keyFactory.generatePublic(spec);
        }
        this.check("IN: ", peerPublicKey, encodedPoint);
        SecretKey secretKey = null;
        KeyAgreement keyAgreement = null;
        if (this.supportedGroup.getAlgorithmName().equals("EC")) {
            keyAgreement = (KeyAgreement)ECDH_KEY_AGREEMENT.currentWithCause();
        } else if (this.supportedGroup.getAlgorithmName().equals("XDH")) {
            keyAgreement = (KeyAgreement)XDH_KEY_AGREEMENT.currentWithCause();
        }
        if (keyAgreement == null) return secretKey;
        keyAgreement.init(this.privateKey);
        keyAgreement.doPhase(peerPublicKey, true);
        byte[] secret = keyAgreement.generateSecret();
        secretKey = SecretUtil.create(secret, "TlsPremasterSecret");
        Bytes.clear(secret);
        return secretKey;
    }

    @Override
    public void destroy() {
        this.privateKey = null;
    }

    @Override
    public boolean isDestroyed() {
        return this.privateKey == null;
    }

    private byte[] encodedPoint(PublicKey publicKey) throws GeneralSecurityException {
        byte[] result = null;
        int keySizeInBytes = this.supportedGroup.getKeySizeInBytes();
        try {
            if (this.supportedGroup.getAlgorithmName().equals("EC")) {
                ECPublicKey ecPublicKey = (ECPublicKey)publicKey;
                result = XECDHECryptography.encodePoint(ecPublicKey.getW(), keySizeInBytes);
            } else if (this.supportedGroup.getAlgorithmName().equals("XDH")) {
                BigInteger u = this.getXECPublicKeyU(publicKey);
                result = XECDHECryptography.revert(u.toByteArray(), keySizeInBytes);
            }
        }
        catch (RuntimeException ex) {
            throw new GeneralSecurityException("DHE: failed to encoded point! " + this.supportedGroup.name(), ex);
        }
        if (result == null) {
            throw new GeneralSecurityException("DHE: failed to encoded point! " + this.supportedGroup.name());
        }
        this.check("OUT: ", publicKey, result);
        return result;
    }

    private void check(String tag, PublicKey publicKey, byte[] point) throws GeneralSecurityException {
        if (LOGGER.isDebugEnabled()) {
            byte[] asn1 = publicKey.getEncoded();
            String s1 = StringUtil.byteArray2Hex(asn1);
            String s2 = StringUtil.byteArray2Hex(point);
            if (s2.length() < s1.length()) {
                s2 = String.format("%" + s1.length() + "s", s2);
            }
            LOGGER.debug("{}ASN1 encoded '{}'", (Object)tag, (Object)s1);
            LOGGER.debug("{}DHE  encoded '{}'", (Object)tag, (Object)s2);
            for (int index = 0; index < point.length; ++index) {
                if (point[point.length - index - 1] == asn1[asn1.length - index - 1]) continue;
                throw new GeneralSecurityException("DHE: failed to encoded point! " + this.supportedGroup.name() + ", position: " + index);
            }
        }
    }

    private BigInteger getXECPublicKeyU(PublicKey publicKey) throws GeneralSecurityException {
        if (XECPublicKeyGetU == null) {
            throw new GeneralSecurityException(this.supportedGroup.name() + " not supported by JRE!");
        }
        try {
            return (BigInteger)XECPublicKeyGetU.invoke((Object)publicKey, new Object[0]);
        }
        catch (Exception e) {
            throw new GeneralSecurityException(this.supportedGroup.name() + " not supported by JRE!", e);
        }
    }

    private KeySpec getXECPublicKeySpec(String name, BigInteger u) throws GeneralSecurityException {
        if (XECPublicKeySpecInit == null) {
            throw new GeneralSecurityException(this.supportedGroup.name() + " not supported by JRE!");
        }
        try {
            ECGenParameterSpec parameterSpec = new ECGenParameterSpec(name);
            return (KeySpec)XECPublicKeySpecInit.newInstance(parameterSpec, u);
        }
        catch (Exception e) {
            throw new GeneralSecurityException(this.supportedGroup.name() + " not supported by JRE!", e);
        }
    }

    private static String getXECPublicKeyName(PublicKey publicKey) throws GeneralSecurityException {
        if (XECPublicKeyGetParams == null || NamedParameterSpecGetName == null) {
            throw new GeneralSecurityException("X25519/X448 not supported by JRE!");
        }
        try {
            Object params = XECPublicKeyGetParams.invoke((Object)publicKey, new Object[0]);
            return (String)NamedParameterSpecGetName.invoke(params, new Object[0]);
        }
        catch (Exception e) {
            throw new GeneralSecurityException("X25519/X448 not supported by JRE!");
        }
    }

    private static int noneZeroOffset(byte[] byteArray) {
        int offset;
        for (offset = 0; offset < byteArray.length && byteArray[offset] == 0; ++offset) {
        }
        return offset;
    }

    private static byte[] revert(byte[] byteArray, int size) {
        int offset = XECDHECryptography.noneZeroOffset(byteArray);
        int length = byteArray.length - offset;
        if (length > size) {
            throw new IllegalArgumentException("big integer array exceeds size! " + length + " > " + size);
        }
        byte[] result = new byte[size];
        for (int index = 0; index < length; ++index) {
            result[length - 1 - index] = byteArray[index + offset];
        }
        return result;
    }

    private static byte[] encodePoint(ECPoint point, int keySizeInBytes) {
        byte[] xb = point.getAffineX().toByteArray();
        byte[] yb = point.getAffineY().toByteArray();
        int xbOffset = XECDHECryptography.noneZeroOffset(xb);
        int xbLength = xb.length - xbOffset;
        int ybOffset = XECDHECryptography.noneZeroOffset(yb);
        int ybLength = yb.length - ybOffset;
        if (xbLength > keySizeInBytes || ybLength > keySizeInBytes) {
            throw new IllegalArgumentException("ec point exceeds size! " + xbLength + "," + ybLength + " > " + keySizeInBytes);
        }
        byte[] encoded = new byte[1 + keySizeInBytes * 2];
        encoded[0] = 4;
        System.arraycopy(xb, xbOffset, encoded, keySizeInBytes + 1 - xbLength, xbLength);
        System.arraycopy(yb, ybOffset, encoded, encoded.length - ybLength, ybLength);
        return encoded;
    }

    static {
        Class<?> cls = null;
        Method getParams = null;
        Method getU = null;
        Method getName = null;
        Constructor<?> init = null;
        try {
            cls = Class.forName("java.security.spec.XECPublicKeySpec");
            init = cls.getConstructor(AlgorithmParameterSpec.class, BigInteger.class);
            cls = Class.forName("java.security.spec.NamedParameterSpec");
            getName = cls.getMethod("getName", new Class[0]);
            cls = Class.forName("java.security.interfaces.XECPublicKey");
            getU = cls.getMethod("getU", new Class[0]);
            getParams = cls.getMethod("getParams", new Class[0]);
        }
        catch (Throwable t) {
            LOGGER.info("X25519/X448 not supported!");
        }
        XECPublicKeyClass = cls;
        XECPublicKeyGetU = getU;
        XECPublicKeyGetParams = getParams;
        NamedParameterSpecGetName = getName;
        XECPublicKeySpecInit = init;
        EC_CURVE_MAP_BY_ID = new HashMap<Integer, SupportedGroup>();
        EC_CURVE_MAP_BY_CURVE = new HashMap<EllipticCurve, SupportedGroup>();
    }

    private static class Initialize {
        private static final SupportedGroup[] PREFERRED = new SupportedGroup[]{SupportedGroup.secp256r1, SupportedGroup.X25519, SupportedGroup.X448, SupportedGroup.secp384r1};
        private static final List<SupportedGroup> USABLE_GROUPS;
        private static final List<SupportedGroup> PREFERRED_GROUPS;

        private Initialize() {
        }

        static {
            ArrayList<SupportedGroup> usableGroups = new ArrayList<SupportedGroup>();
            ArrayList<SupportedGroup> preferredGroups = new ArrayList<SupportedGroup>();
            for (SupportedGroup group : SupportedGroup.values()) {
                if (!group.isUsable()) continue;
                usableGroups.add(group);
            }
            for (SupportedGroup group : PREFERRED) {
                if (!group.isUsable()) continue;
                preferredGroups.add(group);
            }
            if (preferredGroups.isEmpty() && !usableGroups.isEmpty()) {
                preferredGroups.add((SupportedGroup)((Object)usableGroups.get(0)));
            }
            USABLE_GROUPS = Collections.unmodifiableList(usableGroups);
            PREFERRED_GROUPS = Collections.unmodifiableList(preferredGroups);
        }
    }

    public static enum SupportedGroup {
        sect163k1(1, false),
        sect163r1(2, false),
        sect163r2(3, false),
        sect193r1(4, false),
        sect193r2(5, false),
        sect233k1(6, false),
        sect233r1(7, false),
        sect239k1(8, false),
        sect283k1(9, false),
        sect283r1(10, false),
        sect409k1(11, false),
        sect409r1(12, false),
        sect571k1(13, false),
        sect571r1(14, false),
        secp160k1(15, false),
        secp160r1(16, false),
        secp160r2(17, false),
        secp192k1(18, false),
        secp192r1(19, false),
        secp224k1(20, false),
        secp224r1(21, false),
        secp256k1(22, false),
        secp256r1(23, true),
        secp384r1(24, true),
        secp521r1(25, false),
        brainpoolP256r1(26, false),
        brainpoolP384r1(27, false),
        brainpoolP512r1(28, false),
        ffdhe2048(256, false),
        ffdhe3072(257, false),
        ffdhe4096(258, false),
        ffdhe6144(259, false),
        ffdhe8192(260, false),
        arbitrary_explicit_prime_curves(65281, false),
        arbitrary_explicit_char2_curves(65282, false),
        X25519(29, 32, "XDH", true),
        X448(30, 56, "XDH", true);

        private final int id;
        private final String algorithmName;
        private final int keySizeInBytes;
        private final boolean usable;
        private final boolean recommended;

        private SupportedGroup(int code, boolean recommended) {
            this.id = code;
            this.algorithmName = "EC";
            this.recommended = recommended;
            EllipticCurve curve = null;
            int keySize = 0;
            try {
                KeyPairGenerator keyPairGenerator = (KeyPairGenerator)EC_KEYPAIR_GENERATOR.currentWithCause();
                ECGenParameterSpec genParams = new ECGenParameterSpec(this.name());
                keyPairGenerator.initialize(genParams);
                ECPublicKey apub = (ECPublicKey)keyPairGenerator.generateKeyPair().getPublic();
                curve = apub.getParams().getCurve();
                keySize = (curve.getField().getFieldSize() + 8 - 1) / 8;
                EC_CURVE_MAP_BY_CURVE.put(curve, this);
            }
            catch (Throwable e) {
                LOGGER.trace("Group [{}] is not supported by JRE! {}", (Object)this.name(), (Object)e.getMessage());
                curve = null;
            }
            this.keySizeInBytes = keySize;
            this.usable = curve != null;
            EC_CURVE_MAP_BY_ID.put(code, this);
        }

        private SupportedGroup(int code, int keySizeInBytes, String algorithmName, boolean recommended) {
            this.id = code;
            this.algorithmName = algorithmName;
            this.keySizeInBytes = keySizeInBytes;
            this.recommended = recommended;
            boolean usable = false;
            try {
                KeyPairGenerator keyPairGenerator = (KeyPairGenerator)XDH_KEYPAIR_GENERATOR.currentWithCause();
                ECGenParameterSpec params = new ECGenParameterSpec(this.name());
                keyPairGenerator.initialize(params);
                keyPairGenerator.generateKeyPair();
                usable = true;
            }
            catch (Throwable e) {
                LOGGER.trace("Group [{}] is not supported by JRE! {}", (Object)this.name(), (Object)e.getMessage());
            }
            this.usable = usable;
            EC_CURVE_MAP_BY_ID.put(code, this);
        }

        public int getId() {
            return this.id;
        }

        public String getAlgorithmName() {
            return this.algorithmName;
        }

        public static SupportedGroup fromId(int id) {
            return (SupportedGroup)((Object)EC_CURVE_MAP_BY_ID.get(id));
        }

        public static SupportedGroup fromPublicKey(PublicKey publicKey) {
            if (publicKey != null) {
                if (publicKey instanceof ECPublicKey) {
                    ECParameterSpec params = ((ECPublicKey)publicKey).getParams();
                    return (SupportedGroup)((Object)EC_CURVE_MAP_BY_CURVE.get(params.getCurve()));
                }
                if (XECPublicKeyClass != null && XECPublicKeyClass.isInstance(publicKey)) {
                    try {
                        String name = XECDHECryptography.getXECPublicKeyName(publicKey);
                        return SupportedGroup.valueOf(name);
                    }
                    catch (GeneralSecurityException name) {}
                } else {
                    String algorithm = publicKey.getAlgorithm();
                    String oid = Asn1DerDecoder.getEdDsaStandardAlgorithmName(algorithm, null);
                    if ("OID.1.3.101.112".equals(oid) || "EdDSA".equalsIgnoreCase(oid)) {
                        return X25519;
                    }
                    if ("OID.1.3.101.113".equals(oid)) {
                        return X448;
                    }
                    LOGGER.warn("No supported curve {}/{}", (Object)publicKey.getAlgorithm(), (Object)oid);
                }
            }
            return null;
        }

        public static boolean isEcPublicKey(PublicKey publicKey) {
            if (publicKey instanceof ECPublicKey) {
                return true;
            }
            return XECPublicKeyClass != null && XECPublicKeyClass.isInstance(publicKey);
        }

        public static boolean isSupported(List<SupportedGroup> list, List<X509Certificate> certificateChain) {
            for (X509Certificate certificate : certificateChain) {
                SupportedGroup group;
                PublicKey publicKey = certificate.getPublicKey();
                if (!SupportedGroup.isEcPublicKey(publicKey) || (group = SupportedGroup.fromPublicKey(certificate.getPublicKey())) != null && group.isUsable() && list.contains((Object)group)) continue;
                return false;
            }
            return true;
        }

        public int getKeySizeInBytes() {
            return this.keySizeInBytes;
        }

        public boolean isUsable() {
            return this.usable;
        }

        public boolean isRecommended() {
            return this.recommended;
        }

        public static List<SupportedGroup> getUsableGroups() {
            return Initialize.USABLE_GROUPS;
        }

        public static List<SupportedGroup> getPreferredGroups() {
            return Initialize.PREFERRED_GROUPS;
        }
    }
}

