/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network.stack;

import java.util.Iterator;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.californium.core.coap.BlockOption;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.MessageObserver;
import org.eclipse.californium.core.coap.MessageObserverAdapter;
import org.eclipse.californium.core.coap.OptionSet;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.coap.Token;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.stack.AbstractLayer;
import org.eclipse.californium.core.network.stack.Block1BlockwiseStatus;
import org.eclipse.californium.core.network.stack.Block2BlockwiseStatus;
import org.eclipse.californium.core.network.stack.CleanupMessageObserver;
import org.eclipse.californium.core.network.stack.KeyUri;
import org.eclipse.californium.elements.util.LeastRecentlyUsedCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockwiseLayer
extends AbstractLayer {
    private static final int MINIMAL_BLOCK_SIZE = 16;
    private static final Logger LOGGER = LoggerFactory.getLogger(BlockwiseLayer.class);
    private static final Logger HEALTH_LOGGER = LoggerFactory.getLogger(LOGGER.getName() + ".health");
    private final LeastRecentlyUsedCache<KeyUri, Block1BlockwiseStatus> block1Transfers;
    private final LeastRecentlyUsedCache<KeyUri, Block2BlockwiseStatus> block2Transfers;
    private final AtomicInteger ignoredBlock2 = new AtomicInteger();
    private volatile boolean enableStatus;
    private ScheduledFuture<?> statusLogger;
    private int maxMessageSize;
    private int preferredBlockSize;
    private int preferredBlockSzx;
    private int blockTimeout;
    private int maxResourceBodySize;
    private boolean strictBlock2Option;
    private int healthStatusInterval;
    private boolean enableAutoFailoverOn413;

    public BlockwiseLayer(NetworkConfig config) {
        this.maxMessageSize = config.getInt("MAX_MESSAGE_SIZE", 1024);
        this.preferredBlockSize = config.getInt("PREFERRED_BLOCK_SIZE", 512);
        this.preferredBlockSzx = BlockOption.size2Szx(this.preferredBlockSize);
        this.blockTimeout = config.getInt("BLOCKWISE_STATUS_LIFETIME", 300000);
        this.maxResourceBodySize = config.getInt("MAX_RESOURCE_BODY_SIZE", 8192);
        int maxActivePeers = config.getInt("MAX_ACTIVE_PEERS", 150000);
        this.block1Transfers = new LeastRecentlyUsedCache(maxActivePeers, TimeUnit.MILLISECONDS.toSeconds(this.blockTimeout));
        this.block1Transfers.setEvictingOnReadAccess(false);
        this.block2Transfers = new LeastRecentlyUsedCache(maxActivePeers, TimeUnit.MILLISECONDS.toSeconds(this.blockTimeout));
        this.block2Transfers.setEvictingOnReadAccess(false);
        this.strictBlock2Option = config.getBoolean("BLOCKWISE_STRICT_BLOCK2_OPTION", false);
        this.healthStatusInterval = config.getInt("HEALTH_STATUS_INTERVAL", 60);
        this.enableAutoFailoverOn413 = config.getBoolean("BLOCKWISE_ENTITY_TOO_LARGE_AUTO_FAILOVER", true);
        LOGGER.info("BlockwiseLayer uses MAX_MESSAGE_SIZE={}, PREFERRED_BLOCK_SIZE={}, BLOCKWISE_STATUS_LIFETIME={}, MAX_RESOURCE_BODY_SIZE={}, BLOCKWISE_STRICT_BLOCK2_OPTION={}", this.maxMessageSize, this.preferredBlockSize, this.blockTimeout, this.maxResourceBodySize, this.strictBlock2Option);
    }

    @Override
    public void start() {
        if (this.healthStatusInterval > 0 && HEALTH_LOGGER.isDebugEnabled() && this.statusLogger == null) {
            this.statusLogger = this.secondaryExecutor.scheduleAtFixedRate(new Runnable(){

                @Override
                public void run() {
                    if (BlockwiseLayer.this.enableStatus) {
                        HEALTH_LOGGER.debug("{} block1 transfers", (Object)BlockwiseLayer.this.block1Transfers.size());
                        Iterator iterator = BlockwiseLayer.this.block1Transfers.valuesIterator();
                        int max = 5;
                        while (iterator.hasNext()) {
                            HEALTH_LOGGER.debug("   block1 {}", iterator.next());
                            if (--max != 0) continue;
                        }
                        HEALTH_LOGGER.debug("{} block2 transfers", (Object)BlockwiseLayer.this.block2Transfers.size());
                        iterator = BlockwiseLayer.this.block2Transfers.valuesIterator();
                        max = 5;
                        while (iterator.hasNext()) {
                            HEALTH_LOGGER.debug("   block2 {}", iterator.next());
                            if (--max != 0) continue;
                        }
                        HEALTH_LOGGER.debug("{} block2 responses ignored", (Object)BlockwiseLayer.this.ignoredBlock2.get());
                    }
                }
            }, this.healthStatusInterval, this.healthStatusInterval, TimeUnit.SECONDS);
        }
    }

    @Override
    public void destroy() {
        if (this.statusLogger != null) {
            this.statusLogger.cancel(false);
            this.statusLogger = null;
        }
    }

    @Override
    public void sendRequest(Exchange exchange, Request request) {
        Request requestToSend = request;
        if (this.isTransparentBlockwiseHandlingEnabled() && !request.isMulticast()) {
            BlockOption block2 = request.getOptions().getBlock2();
            if (block2 != null && block2.getNum() > 0) {
                LOGGER.debug("outbound request contains block2 option, creating random-access blockwise status");
                this.addRandomAccessBlock2Status(exchange, request);
            } else {
                KeyUri key = BlockwiseLayer.getKey(exchange, request);
                Block2BlockwiseStatus status = this.getBlock2Status(key);
                if (status != null) {
                    this.clearBlock2Status(key, status);
                    status.completeOldTransfer(null);
                }
                if (this.requiresBlockwise(request)) {
                    requestToSend = this.startBlockwiseUpload(exchange, request, this.preferredBlockSize);
                }
            }
        }
        exchange.setCurrentRequest(requestToSend);
        this.lower().sendRequest(exchange, requestToSend);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Request startBlockwiseUpload(Exchange exchange, final Request request, int blocksize) {
        KeyUri key = BlockwiseLayer.getKey(exchange, request);
        LeastRecentlyUsedCache<KeyUri, Block1BlockwiseStatus> leastRecentlyUsedCache = this.block1Transfers;
        synchronized (leastRecentlyUsedCache) {
            Block1BlockwiseStatus status = this.getBlock1Status(key);
            if (status != null) {
                status.cancelRequest();
                this.clearBlock1Status(key, status);
            }
            status = this.getOutboundBlock1Status(key, exchange, request, blocksize);
            final Request block = status.getNextRequestBlock();
            block.setDestinationContext(request.getDestinationContext());
            Token token = request.getToken();
            if (token != null) {
                block.setToken(token);
            }
            block.addMessageObserver(new MessageObserverAdapter(){

                @Override
                public void onReadyToSend() {
                    if (request.getToken() == null) {
                        request.setToken(block.getToken());
                    }
                    if (!request.hasMID()) {
                        request.setMID(block.getMID());
                    }
                }
            });
            this.addBlock1CleanUpObserver(block, key, status);
            this.prepareBlock1Cleanup(status, key);
            return block;
        }
    }

    @Override
    public void receiveRequest(Exchange exchange, Request request) {
        if (this.isTransparentBlockwiseHandlingEnabled()) {
            BlockOption block2 = request.getOptions().getBlock2();
            if (request.getOptions().hasBlock1()) {
                this.handleInboundBlockwiseUpload(exchange, request);
            } else if (block2 != null && block2.getNum() > 0) {
                KeyUri key = BlockwiseLayer.getKey(exchange, request);
                Block2BlockwiseStatus status = this.getBlock2Status(key);
                if (status == null) {
                    LOGGER.debug("peer wants to retrieve individual block2 {} of {}, delivering request to application layer", (Object)block2, (Object)key);
                    exchange.setRequest(request);
                    this.upper().receiveRequest(exchange, request);
                } else {
                    this.handleInboundRequestForNextBlock(exchange, request, key, status);
                }
            } else {
                exchange.setRequest(request);
                this.upper().receiveRequest(exchange, request);
            }
        } else {
            exchange.setRequest(request);
            this.upper().receiveRequest(exchange, request);
        }
    }

    private void handleInboundBlockwiseUpload(Exchange exchange, Request request) {
        if (this.requestExceedsMaxBodySize(request)) {
            int maxResourceBodySize = this.getMaxResourceBodySize(request);
            Response error = Response.createResponse(request, CoAP.ResponseCode.REQUEST_ENTITY_TOO_LARGE);
            error.setPayload(String.format("body too large, can process %d bytes max", maxResourceBodySize));
            error.getOptions().setSize1(maxResourceBodySize);
            exchange.setCurrentResponse(error);
            this.lower().sendResponse(exchange, error);
        } else {
            BlockOption block1 = request.getOptions().getBlock1();
            LOGGER.debug("inbound request contains block1 option {}", (Object)block1);
            KeyUri key = BlockwiseLayer.getKey(exchange, request);
            Block1BlockwiseStatus status = this.getInboundBlock1Status(key, exchange, request);
            if (block1.getNum() == 0 && status.getCurrentNum() > 0) {
                status = this.resetInboundBlock1Status(key, exchange, request);
            }
            if (block1.getNum() != status.getCurrentNum()) {
                LOGGER.warn("peer sent wrong block, expected no. {} but got {}. Responding with 4.08 (Request Entity Incomplete)", (Object)status.getCurrentNum(), (Object)block1.getNum());
                this.sendBlock1ErrorResponse(key, status, exchange, request, CoAP.ResponseCode.REQUEST_ENTITY_INCOMPLETE, "wrong block number");
            } else if (!status.hasContentFormat(request.getOptions().getContentFormat())) {
                this.sendBlock1ErrorResponse(key, status, exchange, request, CoAP.ResponseCode.REQUEST_ENTITY_INCOMPLETE, "unexpected Content-Format");
            } else if (!status.addBlock(request.getPayload())) {
                this.sendBlock1ErrorResponse(key, status, exchange, request, CoAP.ResponseCode.REQUEST_ENTITY_TOO_LARGE, "body exceeded expected size " + status.getBufferSize());
            } else {
                status.setCurrentNum(status.getCurrentNum() + 1);
                if (block1.isM()) {
                    LOGGER.debug("acknowledging incoming block1 [num={}], expecting more blocks to come", (Object)block1.getNum());
                    Response piggybacked = Response.createResponse(request, CoAP.ResponseCode.CONTINUE);
                    piggybacked.getOptions().setBlock1(block1.getSzx(), true, block1.getNum());
                    exchange.setCurrentResponse(piggybacked);
                    this.lower().sendResponse(exchange, piggybacked);
                } else {
                    LOGGER.debug("peer has sent last block1 [num={}], delivering request to application layer", (Object)block1.getNum());
                    exchange.setBlock1ToAck(block1);
                    Request assembled = new Request(request.getCode());
                    status.assembleReceivedMessage(assembled);
                    assembled.setMID(request.getMID());
                    assembled.setToken(request.getToken());
                    assembled.setScheme(request.getScheme());
                    assembled.getOptions().setBlock2(request.getOptions().getBlock2());
                    this.clearBlock1Status(key, status);
                    exchange.setRequest(assembled);
                    this.upper().receiveRequest(exchange, assembled);
                }
            }
        }
    }

    private void sendBlock1ErrorResponse(KeyUri key, Block1BlockwiseStatus status, Exchange exchange, Request request, CoAP.ResponseCode errorCode, String message) {
        BlockOption block1 = request.getOptions().getBlock1();
        Response error = Response.createResponse(request, errorCode);
        error.getOptions().setBlock1(block1.getSzx(), block1.isM(), block1.getNum());
        error.setPayload(message);
        this.clearBlock1Status(key, status);
        exchange.setCurrentResponse(error);
        this.lower().sendResponse(exchange, error);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleInboundRequestForNextBlock(Exchange exchange, Request request, KeyUri key, Block2BlockwiseStatus status) {
        boolean complete;
        Response block;
        Block2BlockwiseStatus block2BlockwiseStatus = status;
        synchronized (block2BlockwiseStatus) {
            BlockOption block2 = request.getOptions().getBlock2();
            block = status.getNextResponseBlock(block2);
            complete = status.isComplete();
            if (!complete) {
                this.prepareBlock2Cleanup(status, key);
                LOGGER.debug("peer has requested intermediary block of blockwise transfer: {}", (Object)status);
            }
        }
        if (complete) {
            LOGGER.debug("peer has requested last block of blockwise transfer: {}", (Object)status);
            this.clearBlock2Status(key, status);
        }
        exchange.setCurrentResponse(block);
        this.lower().sendResponse(exchange, block);
    }

    @Override
    public void sendResponse(Exchange exchange, Response response) {
        Response responseToSend = response;
        if (this.isTransparentBlockwiseHandlingEnabled()) {
            BlockOption block1;
            BlockOption requestBlock2 = exchange.getRequest().getOptions().getBlock2();
            BlockOption responseBlock2 = response.getOptions().getBlock2();
            if (requestBlock2 != null && requestBlock2.getNum() > 0) {
                if (responseBlock2 != null) {
                    if (requestBlock2.getNum() != responseBlock2.getNum()) {
                        LOGGER.warn("resource [{}] implementation error, peer requested block {} but resource returned block {}", exchange.getRequest().getURI(), requestBlock2.getNum(), responseBlock2.getNum());
                        responseToSend = Response.createResponse(exchange.getRequest(), CoAP.ResponseCode.INTERNAL_SERVER_ERROR);
                        responseToSend.setType(response.getType());
                        responseToSend.setMID(response.getMID());
                        responseToSend.addMessageObservers(response.getMessageObservers());
                    }
                } else if (response.hasBlock(requestBlock2)) {
                    Block2BlockwiseStatus.crop(responseToSend, requestBlock2);
                } else {
                    responseToSend = Response.createResponse(exchange.getRequest(), CoAP.ResponseCode.BAD_OPTION);
                    responseToSend.setType(response.getType());
                    responseToSend.setMID(response.getMID());
                    responseToSend.getOptions().setBlock2(requestBlock2);
                    responseToSend.addMessageObservers(response.getMessageObservers());
                }
            } else if (this.requiresBlockwise(exchange, response, requestBlock2)) {
                KeyUri key = BlockwiseLayer.getKey(exchange, response);
                Block2BlockwiseStatus status = this.resetOutboundBlock2Status(key, exchange, response);
                BlockOption block2 = requestBlock2 != null ? requestBlock2 : new BlockOption(this.preferredBlockSzx, false, 0);
                responseToSend = status.getNextResponseBlock(block2);
            }
            if ((block1 = exchange.getBlock1ToAck()) != null) {
                exchange.setBlock1ToAck(null);
                responseToSend.getOptions().setBlock1(block1);
            }
        }
        exchange.setCurrentResponse(responseToSend);
        this.lower().sendResponse(exchange, responseToSend);
    }

    @Override
    public void receiveResponse(Exchange exchange, Response response) {
        if (this.isTransparentBlockwiseHandlingEnabled() && !exchange.getRequest().isMulticast()) {
            Block2BlockwiseStatus status;
            KeyUri key;
            if (response.isError()) {
                switch (response.getCode()) {
                    case REQUEST_ENTITY_INCOMPLETE: 
                    case REQUEST_ENTITY_TOO_LARGE: {
                        if (this.handleEntityTooLarge(exchange, response)) {
                            return;
                        }
                        KeyUri key2 = BlockwiseLayer.getKey(exchange, exchange.getCurrentRequest());
                        Block1BlockwiseStatus status2 = this.getBlock1Status(key2);
                        if (status2 == null) break;
                        this.clearBlock1Status(key2, status2);
                    }
                }
                if (exchange.getRequest() != exchange.getCurrentRequest()) {
                    Response resp = new Response(response.getCode());
                    resp.setToken(exchange.getRequest().getToken());
                    if (exchange.getRequest().getType() == CoAP.Type.CON) {
                        resp.setType(CoAP.Type.ACK);
                        resp.setMID(exchange.getRequest().getMID());
                    } else {
                        resp.setType(CoAP.Type.NON);
                    }
                    resp.setSourceContext(response.getSourceContext());
                    resp.setPayload(response.getPayload());
                    resp.setOptions(response.getOptions());
                    resp.setRTT(exchange.calculateRTT());
                    exchange.setResponse(resp);
                    this.upper().receiveResponse(exchange, resp);
                } else {
                    this.upper().receiveResponse(exchange, response);
                }
                return;
            }
            if (response.getMaxResourceBodySize() == 0) {
                response.setMaxResourceBodySize(exchange.getRequest().getMaxResourceBodySize());
            }
            if (this.discardBlock2(key = BlockwiseLayer.getKey(exchange, response), status = this.getBlock2Status(key), exchange, response)) {
                return;
            }
            if (!response.hasBlockOption()) {
                exchange.setResponse(response);
                this.upper().receiveResponse(exchange, response);
            } else {
                if (response.getOptions().hasBlock1()) {
                    this.handleBlock1Response(exchange, response);
                }
                if (response.getOptions().hasBlock2()) {
                    this.handleBlock2Response(exchange, response);
                }
            }
        } else {
            exchange.setResponse(response);
            this.upper().receiveResponse(exchange, response);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean handleEntityTooLarge(Exchange exchange, Response response) {
        if (this.enableAutoFailoverOn413) {
            if (response.getOptions().hasBlock1()) {
                Block1BlockwiseStatus status;
                BlockOption block1 = response.getOptions().getBlock1();
                KeyUri key = BlockwiseLayer.getKey(exchange, exchange.getRequest());
                Request blockRequest = null;
                LeastRecentlyUsedCache<KeyUri, Block1BlockwiseStatus> leastRecentlyUsedCache = this.block1Transfers;
                synchronized (leastRecentlyUsedCache) {
                    status = this.getBlock1Status(key);
                    if (status == null) {
                        Request request = exchange.getRequest();
                        if (!exchange.getRequest().isCanceled() && block1.getNum() == 0 && block1.getSize() < request.getPayloadSize()) {
                            blockRequest = this.startBlockwiseUpload(exchange, request, Math.min(block1.getSize(), this.preferredBlockSize));
                        }
                    }
                }
                if (status == null) {
                    if (blockRequest != null) {
                        exchange.setCurrentRequest(blockRequest);
                        this.lower().sendRequest(exchange, blockRequest);
                        return true;
                    }
                } else {
                    if (!status.hasMatchingToken(response)) {
                        LOGGER.debug("discarding obsolete block1 response: {}", (Object)response);
                        return true;
                    }
                    if (exchange.getRequest().isCanceled()) {
                        this.clearBlock1Status(key, status);
                        return true;
                    }
                    if (status.getCurrentNum() == 0 && block1.getSize() < status.getCurrentSize()) {
                        this.sendBlock(exchange, response, key, status, 0, block1.getSzx());
                        return true;
                    }
                }
            } else if (!exchange.getRequest().isCanceled()) {
                Request requestToSend = null;
                LeastRecentlyUsedCache<KeyUri, Block1BlockwiseStatus> leastRecentlyUsedCache = this.block1Transfers;
                synchronized (leastRecentlyUsedCache) {
                    if (this.getBlock1Status(BlockwiseLayer.getKey(exchange, exchange.getRequest())) == null) {
                        Request request = exchange.getRequest();
                        Integer maxSize = null;
                        if (response.getOptions().hasSize1() && response.getOptions().getSize1() >= 16 && response.getOptions().getSize1() < request.getPayloadSize()) {
                            maxSize = response.getOptions().getSize1();
                        } else if (request.getPayloadSize() > 16) {
                            maxSize = request.getPayloadSize() - 1;
                        }
                        if (maxSize != null) {
                            int blocksize = Integer.highestOneBit(maxSize);
                            requestToSend = this.startBlockwiseUpload(exchange, request, Math.min(blocksize, this.preferredBlockSize));
                        }
                    }
                }
                if (requestToSend != null) {
                    exchange.setCurrentRequest(requestToSend);
                    this.lower().sendRequest(exchange, requestToSend);
                    return true;
                }
            }
        }
        return false;
    }

    private void handleBlock1Response(Exchange exchange, Response response) {
        BlockOption block1 = response.getOptions().getBlock1();
        LOGGER.debug("received response acknowledging block1 {}", (Object)block1);
        KeyUri key = BlockwiseLayer.getKey(exchange, exchange.getRequest());
        Block1BlockwiseStatus status = this.getBlock1Status(key);
        if (status == null) {
            LOGGER.debug("discarding unexpected block1 response: {}", (Object)response);
        } else if (!status.hasMatchingToken(response)) {
            LOGGER.debug("discarding obsolete block1 response: {}", (Object)response);
        } else if (exchange.getRequest().isCanceled()) {
            this.clearBlock1Status(key, status);
        } else if (!status.isComplete()) {
            if (block1.isM()) {
                if (response.getCode() == CoAP.ResponseCode.CONTINUE) {
                    this.sendNextBlock(exchange, response, key, status);
                } else {
                    this.clearBlock1Status(key, status);
                    exchange.getRequest().setRejected(true);
                }
            } else {
                this.sendNextBlock(exchange, response, key, status);
            }
        } else {
            this.clearBlock1Status(key, status);
            if (response.getOptions().hasBlock2()) {
                LOGGER.debug("Block1 followed by Block2 transfer");
            } else {
                exchange.setResponse(response);
                this.upper().receiveResponse(exchange, response);
            }
        }
    }

    private void sendNextBlock(Exchange exchange, Response response, KeyUri key, Block1BlockwiseStatus status) {
        int newSzx;
        int newSize;
        BlockOption block1 = response.getOptions().getBlock1();
        int currentSize = status.getCurrentSize();
        if (block1.getSize() < currentSize) {
            newSize = block1.getSize();
            newSzx = block1.getSzx();
        } else {
            newSize = currentSize;
            newSzx = status.getCurrentSzx();
        }
        int nextNum = (status.getCurrentNum() + 1) * status.getCurrentSize() / newSize;
        this.sendBlock(exchange, response, key, status, nextNum, newSzx);
    }

    private void sendBlock(Exchange exchange, Response response, KeyUri key, Block1BlockwiseStatus status, int num, int szx) {
        Request nextBlock = null;
        LOGGER.trace("sending Block1 num={}", (Object)num);
        try {
            if (status.isComplete()) {
                LOGGER.debug("stopped block1 transfer, droping request.");
            } else {
                nextBlock = status.getNextRequestBlock(num, szx);
                nextBlock.setToken(response.getToken());
                nextBlock.setDestinationContext(status.getFollowUpEndpointContext(response.getSourceContext()));
                this.addBlock1CleanUpObserver(nextBlock, key, status);
                LOGGER.debug("sending (next) Block1 [num={}]: {}", (Object)num, (Object)nextBlock);
                exchange.setCurrentRequest(nextBlock);
                this.prepareBlock1Cleanup(status, key);
                this.lower().sendRequest(exchange, nextBlock);
            }
        }
        catch (RuntimeException ex) {
            LOGGER.warn("cannot process next block request, aborting request!", ex);
            if (nextBlock != null) {
                nextBlock.setSendError(ex);
            }
            exchange.getRequest().setSendError(ex);
        }
    }

    /*
     * Enabled aggressive block sorting
     */
    private boolean discardBlock2(KeyUri key, Block2BlockwiseStatus status, Exchange exchange, Response response) {
        boolean starting;
        BlockOption block = response.getOptions().getBlock2();
        if (status == null) {
            if (block == null) return false;
            if (block.getNum() == 0) return false;
            LOGGER.debug("discarding stale block2 response [{}, {}] received without ongoing block2 transfer for {}", exchange.getNotificationNumber(), response, key);
            exchange.setComplete();
            return true;
        }
        boolean bl = starting = block == null || block.getNum() == 0;
        if (!starting) {
            if (status.matchTransfer(exchange)) return false;
            LOGGER.debug("discarding outdate block2 response [{}, {}] received during ongoing block2 transfer {}", exchange.getNotificationNumber(), response, status.getObserve());
            status.completeNewTranfer(exchange);
            return true;
        }
        if (status.isNew(response)) {
            LOGGER.debug("discarding outdated block2 transfer {}, current is [{}]", (Object)status.getObserve(), (Object)response);
            this.clearBlock2Status(key, status);
            status.completeOldTransfer(exchange);
            return false;
        }
        LOGGER.debug("discarding old block2 transfer [{}], received during ongoing block2 transfer {}", (Object)response, (Object)status.getObserve());
        status.completeNewTranfer(exchange);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleBlock2Response(Exchange exchange, Response response) {
        BlockOption block2 = response.getOptions().getBlock2();
        KeyUri key = BlockwiseLayer.getKey(exchange, response);
        if (exchange.getRequest().isCanceled()) {
            Block2BlockwiseStatus status = this.getBlock2Status(key);
            if (status != null) {
                this.clearBlock2Status(key, status);
            }
            if (response.isNotification()) {
                this.upper().receiveResponse(exchange, response);
            }
        } else if (this.responseExceedsMaxBodySize(response)) {
            String msg = String.format("requested resource body [%d bytes] exceeds max buffer size [%d bytes], aborting request", response.getOptions().getSize2(), this.getMaxResourceBodySize(response));
            LOGGER.debug(msg);
            exchange.getRequest().setOnResponseError(new IllegalStateException(msg));
            exchange.getRequest().cancel();
        } else {
            Block2BlockwiseStatus status;
            LeastRecentlyUsedCache<KeyUri, Block2BlockwiseStatus> leastRecentlyUsedCache = this.block2Transfers;
            synchronized (leastRecentlyUsedCache) {
                status = this.getBlock2Status(key);
                if (this.discardBlock2(key, status, exchange, response)) {
                    return;
                }
                status = this.getInboundBlock2Status(key, exchange, response);
            }
            if (block2.getNum() == status.getCurrentNum()) {
                LOGGER.debug("processing incoming block2 response [num={}]: {}", (Object)block2.getNum(), (Object)response);
                if (status.isRandomAccess()) {
                    exchange.setResponse(response);
                    this.clearBlock2Status(key, status);
                    this.upper().receiveResponse(exchange, response);
                } else {
                    if (!status.addBlock(response)) {
                        String msg = "cannot process payload of block2 response, aborting request";
                        LOGGER.debug(msg);
                        exchange.getRequest().setOnResponseError(new IllegalStateException(msg));
                        exchange.getRequest().cancel();
                        return;
                    }
                    if (block2.isM()) {
                        this.requestNextBlock(exchange, response, key, status);
                    } else {
                        LOGGER.debug("all {} blocks have been retrieved, assembling response and delivering to application layer", (Object)status.getBlockCount());
                        Response assembled = new Response(response.getCode());
                        status.assembleReceivedMessage(assembled);
                        assembled.setRTT(exchange.calculateRTT());
                        this.clearBlock2Status(key, status);
                        LOGGER.debug("assembled response: {}", (Object)assembled);
                        exchange.setCurrentRequest(exchange.getRequest());
                        exchange.setResponse(assembled);
                        this.upper().receiveResponse(exchange, assembled);
                    }
                }
            } else {
                this.ignoredBlock2.incrementAndGet();
                LOGGER.warn("ignoring block2 response with wrong block number {} (expected {}) - {}: {}", block2.getNum(), status.getCurrentNum(), exchange.getCurrentRequest().getToken(), response);
            }
        }
    }

    private void requestNextBlock(Exchange exchange, Response response, KeyUri key, Block2BlockwiseStatus status) {
        int newSzx;
        int newSize;
        int currentSize = status.getCurrentSize();
        BlockOption block2 = response.getOptions().getBlock2();
        if (block2.getSzx() > this.preferredBlockSzx) {
            newSize = this.preferredBlockSize;
            newSzx = this.preferredBlockSzx;
        } else {
            newSize = currentSize;
            newSzx = status.getCurrentSzx();
        }
        int nextNum = status.getCurrentNum() + currentSize / newSize;
        Request request = exchange.getRequest();
        Request block = new Request(request.getCode());
        try {
            block.setType(request.getType());
            block.setDestinationContext(status.getFollowUpEndpointContext(response.getSourceContext()));
            if (!response.isNotification()) {
                block.setToken(response.getToken());
            } else if (exchange.isNotification()) {
                request.addMessageObserver(new CleanupMessageObserver(exchange));
            }
            block.setOptions(new OptionSet(request.getOptions()));
            block.getOptions().setBlock2(newSzx, false, nextNum);
            block.getOptions().removeObserve();
            block.addMessageObservers(request.getMessageObservers());
            this.addBlock2CleanUpObserver(block, key, status);
            status.setCurrentNum(nextNum);
            if (status.isComplete()) {
                LOGGER.debug("stopped block2 transfer, droping response.");
            } else {
                LOGGER.debug("requesting next Block2 [num={}]: {}", (Object)nextNum, (Object)block);
                exchange.setCurrentRequest(block);
                this.prepareBlock2Cleanup(status, key);
                this.lower().sendRequest(exchange, block);
            }
        }
        catch (RuntimeException ex) {
            LOGGER.warn("cannot process next block request, aborting request!", ex);
            block.setSendError(ex);
        }
    }

    private static KeyUri getKey(Exchange exchange, Request request) {
        if (exchange.isOfLocalOrigin()) {
            return KeyUri.fromOutboundRequest(request);
        }
        return KeyUri.fromInboundRequest(request);
    }

    private static KeyUri getKey(Exchange exchange, Response response) {
        if (exchange.isOfLocalOrigin()) {
            return KeyUri.fromInboundResponse(exchange.getRequest(), response);
        }
        return KeyUri.fromOutboundResponse(exchange.getRequest(), response);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Block1BlockwiseStatus getOutboundBlock1Status(KeyUri key, Exchange exchange, Request request, int blocksize) {
        LeastRecentlyUsedCache<KeyUri, Block1BlockwiseStatus> leastRecentlyUsedCache = this.block1Transfers;
        synchronized (leastRecentlyUsedCache) {
            Block1BlockwiseStatus status = this.block1Transfers.get(key);
            if (status == null) {
                status = Block1BlockwiseStatus.forOutboundRequest(exchange, request, blocksize);
                this.block1Transfers.put(key, status);
                this.enableStatus = true;
                LOGGER.debug("created tracker for outbound block1 transfer {}, transfers in progress: {}", (Object)status, (Object)this.block1Transfers.size());
            }
            return status;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Block1BlockwiseStatus getInboundBlock1Status(KeyUri key, Exchange exchange, Request request) {
        Block1BlockwiseStatus status;
        int maxPayloadSize = this.getMaxResourceBodySize(request);
        LeastRecentlyUsedCache<KeyUri, Block1BlockwiseStatus> leastRecentlyUsedCache = this.block1Transfers;
        synchronized (leastRecentlyUsedCache) {
            status = this.block1Transfers.get(key);
            if (status == null) {
                status = Block1BlockwiseStatus.forInboundRequest(exchange, request, maxPayloadSize);
                this.block1Transfers.put(key, status);
                this.enableStatus = true;
                LOGGER.debug("created tracker for inbound block1 transfer {}, transfers in progress: {}", (Object)status, (Object)this.block1Transfers.size());
            }
        }
        this.prepareBlock1Cleanup(status, key);
        return status;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Block1BlockwiseStatus resetInboundBlock1Status(KeyUri key, Exchange exchange, Request request) {
        Block1BlockwiseStatus newStatus;
        Block1BlockwiseStatus removedStatus;
        LeastRecentlyUsedCache<KeyUri, Block1BlockwiseStatus> leastRecentlyUsedCache = this.block1Transfers;
        synchronized (leastRecentlyUsedCache) {
            removedStatus = this.block1Transfers.remove(key);
            LOGGER.debug("inbound block1 transfer reset at {} by peer: {}", (Object)removedStatus, (Object)request);
            newStatus = this.getInboundBlock1Status(key, exchange, request);
        }
        if (removedStatus != null) {
            removedStatus.setComplete(true);
        }
        return newStatus;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Block2BlockwiseStatus getOutboundBlock2Status(KeyUri key, Exchange exchange, Response response) {
        Block2BlockwiseStatus status;
        LeastRecentlyUsedCache<KeyUri, Block2BlockwiseStatus> leastRecentlyUsedCache = this.block2Transfers;
        synchronized (leastRecentlyUsedCache) {
            status = this.block2Transfers.get(key);
            if (status == null) {
                status = Block2BlockwiseStatus.forOutboundResponse(exchange, response, this.preferredBlockSize);
                this.block2Transfers.put(key, status);
                this.enableStatus = true;
                LOGGER.debug("created tracker for outbound block2 transfer {}, transfers in progress: {}", (Object)status, (Object)this.block2Transfers.size());
            }
        }
        this.prepareBlock2Cleanup(status, key);
        return status;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Block2BlockwiseStatus getInboundBlock2Status(KeyUri key, Exchange exchange, Response response) {
        int maxPayloadSize = this.getMaxResourceBodySize(response);
        LeastRecentlyUsedCache<KeyUri, Block2BlockwiseStatus> leastRecentlyUsedCache = this.block2Transfers;
        synchronized (leastRecentlyUsedCache) {
            Block2BlockwiseStatus status = this.block2Transfers.get(key);
            if (status == null) {
                status = Block2BlockwiseStatus.forInboundResponse(exchange, response, maxPayloadSize);
                this.block2Transfers.put(key, status);
                this.enableStatus = true;
                LOGGER.debug("created tracker for {} inbound block2 transfer {}, transfers in progress: {}, {}", key, status, this.block2Transfers.size(), response);
            }
            return status;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private KeyUri addRandomAccessBlock2Status(Exchange exchange, Request request) {
        int size;
        KeyUri key = BlockwiseLayer.getKey(exchange, request);
        Block2BlockwiseStatus status = Block2BlockwiseStatus.forRandomAccessRequest(exchange, request);
        LeastRecentlyUsedCache<KeyUri, Block2BlockwiseStatus> leastRecentlyUsedCache = this.block2Transfers;
        synchronized (leastRecentlyUsedCache) {
            this.block2Transfers.put(key, status);
            size = this.block1Transfers.size();
        }
        this.enableStatus = true;
        this.addBlock2CleanUpObserver(request, key, status);
        LOGGER.debug("created tracker for random access block2 retrieval {}, transfers in progress: {}", (Object)status, (Object)size);
        return key;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Block2BlockwiseStatus resetOutboundBlock2Status(KeyUri key, Exchange exchange, Response response) {
        Block2BlockwiseStatus newStatus;
        Block2BlockwiseStatus previousStatus;
        LeastRecentlyUsedCache<KeyUri, Block2BlockwiseStatus> leastRecentlyUsedCache = this.block2Transfers;
        synchronized (leastRecentlyUsedCache) {
            previousStatus = this.block2Transfers.remove(key);
            newStatus = this.getOutboundBlock2Status(key, exchange, response);
        }
        if (previousStatus != null && !previousStatus.isComplete()) {
            LOGGER.debug("stop previous block transfer {} {} for new {}", key, previousStatus, response);
            previousStatus.completeResponse();
        } else {
            LOGGER.debug("block transfer {} for {}", (Object)key, (Object)response);
        }
        return newStatus;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Block1BlockwiseStatus getBlock1Status(KeyUri key) {
        LeastRecentlyUsedCache<KeyUri, Block1BlockwiseStatus> leastRecentlyUsedCache = this.block1Transfers;
        synchronized (leastRecentlyUsedCache) {
            return this.block1Transfers.get(key);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Block2BlockwiseStatus getBlock2Status(KeyUri key) {
        LeastRecentlyUsedCache<KeyUri, Block2BlockwiseStatus> leastRecentlyUsedCache = this.block2Transfers;
        synchronized (leastRecentlyUsedCache) {
            return this.block2Transfers.get(key);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Block1BlockwiseStatus clearBlock1Status(KeyUri key, Block1BlockwiseStatus status) {
        int size;
        Block1BlockwiseStatus removedTracker;
        LeastRecentlyUsedCache<KeyUri, Block1BlockwiseStatus> leastRecentlyUsedCache = this.block1Transfers;
        synchronized (leastRecentlyUsedCache) {
            removedTracker = this.block1Transfers.remove(key, status);
            size = this.block1Transfers.size();
        }
        if (removedTracker != null) {
            LOGGER.debug("removing block1 tracker [{}], block1 transfers still in progress: {}", (Object)key, (Object)size);
            removedTracker.setComplete(true);
        }
        return removedTracker;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Block2BlockwiseStatus clearBlock2Status(KeyUri key, Block2BlockwiseStatus status) {
        int size;
        Block2BlockwiseStatus removedTracker;
        LeastRecentlyUsedCache<KeyUri, Block2BlockwiseStatus> leastRecentlyUsedCache = this.block2Transfers;
        synchronized (leastRecentlyUsedCache) {
            removedTracker = this.block2Transfers.remove(key, status);
            size = this.block2Transfers.size();
        }
        if (removedTracker != null) {
            LOGGER.debug("removing block2 tracker [{}], block2 transfers still in progress: {}", (Object)key, (Object)size);
            removedTracker.setComplete(true);
        }
        return removedTracker;
    }

    private boolean requiresBlockwise(Request request) {
        boolean blockwiseRequired;
        boolean bl = blockwiseRequired = request.getPayloadSize() > this.maxMessageSize;
        if (blockwiseRequired) {
            LOGGER.debug("request body [{}/{}] requires blockwise transfer", (Object)request.getPayloadSize(), (Object)this.maxMessageSize);
        }
        return blockwiseRequired;
    }

    private boolean requiresBlockwise(Exchange exchange, Response response, BlockOption requestBlock2) {
        boolean blockwiseRequired;
        boolean bl = blockwiseRequired = response.getPayloadSize() > this.maxMessageSize;
        if (requestBlock2 != null) {
            boolean bl2 = blockwiseRequired = blockwiseRequired || this.strictBlock2Option || response.getPayloadSize() > requestBlock2.getSize();
        }
        if (blockwiseRequired) {
            LOGGER.debug("response body [{}/{}] requires blockwise transfer", (Object)response.getPayloadSize(), (Object)this.maxMessageSize);
        }
        return blockwiseRequired;
    }

    private boolean isTransparentBlockwiseHandlingEnabled() {
        return this.maxResourceBodySize > 0;
    }

    private boolean responseExceedsMaxBodySize(Response response) {
        return response.getOptions().hasSize2() && response.getOptions().getSize2() > this.getMaxResourceBodySize(response);
    }

    private boolean requestExceedsMaxBodySize(Request request) {
        return request.getOptions().hasSize1() && request.getOptions().getSize1() > this.getMaxResourceBodySize(request);
    }

    private int getMaxResourceBodySize(Message message) {
        int maxPayloadSize = message.getMaxResourceBodySize();
        if (maxPayloadSize == 0) {
            maxPayloadSize = this.maxResourceBodySize;
        }
        return maxPayloadSize;
    }

    protected void prepareBlock1Cleanup(final Block1BlockwiseStatus status, final KeyUri key) {
        LOGGER.debug("scheduling clean up task for block1 transfer {}", (Object)key);
        ScheduledFuture<?> taskHandle = this.scheduleBlockCleanupTask(new Runnable(){

            @Override
            public void run() {
                try {
                    if (!status.isComplete()) {
                        LOGGER.debug("block1 transfer timed out: {}", (Object)key);
                        status.timeoutCurrentTranfer();
                    }
                    BlockwiseLayer.this.clearBlock1Status(key, status);
                }
                catch (Exception e) {
                    LOGGER.debug("Unexcepted error while block1 cleaning", e);
                }
            }
        });
        status.setBlockCleanupHandle(taskHandle);
    }

    private MessageObserver addBlock1CleanUpObserver(Request message, final KeyUri key, final Block1BlockwiseStatus status) {
        MessageObserverAdapter observer = new MessageObserverAdapter(){

            @Override
            public void onCancel() {
                BlockwiseLayer.this.clearBlock1Status(key, status);
            }

            @Override
            protected void failed() {
                BlockwiseLayer.this.clearBlock1Status(key, status);
            }
        };
        message.addMessageObserver(observer);
        return observer;
    }

    private MessageObserver addBlock2CleanUpObserver(Request message, final KeyUri key, final Block2BlockwiseStatus status) {
        MessageObserverAdapter observer = new MessageObserverAdapter(){

            @Override
            public void onCancel() {
                BlockwiseLayer.this.clearBlock2Status(key, status);
            }

            @Override
            protected void failed() {
                BlockwiseLayer.this.clearBlock2Status(key, status);
            }
        };
        message.addMessageObserver(observer);
        return observer;
    }

    protected void prepareBlock2Cleanup(final Block2BlockwiseStatus status, final KeyUri key) {
        LOGGER.debug("scheduling clean up task for block2 transfer {}", (Object)key);
        ScheduledFuture<?> taskHandle = this.scheduleBlockCleanupTask(new Runnable(){

            @Override
            public void run() {
                try {
                    if (!status.isComplete()) {
                        LOGGER.debug("block2 transfer timed out: {}", (Object)key);
                        status.timeoutCurrentTranfer();
                    }
                    BlockwiseLayer.this.clearBlock2Status(key, status);
                }
                catch (Exception e) {
                    LOGGER.debug("Unexcepted error while block2 cleaning", e);
                }
            }
        });
        status.setBlockCleanupHandle(taskHandle);
    }

    private ScheduledFuture<?> scheduleBlockCleanupTask(Runnable task) {
        if (this.executor.isShutdown()) {
            LOGGER.info("Endpoint is being destroyed: skipping block clean-up");
            return null;
        }
        return this.executor.schedule(task, (long)this.blockTimeout, TimeUnit.MILLISECONDS);
    }

    public boolean isEmpty() {
        return this.block1Transfers.size() == 0 && this.block2Transfers.size() == 0;
    }
}

